/**
$Id: findconflicts.cc,v 2.4 1999/03/13 19:12:47 diego Exp $
Free functions to find all the memory conflicts in the program.
*/
#include <suif_copyright.h>

#define _MODULE_ "CSSAME"

#include <suif.h>
#include <useful.h>

#include "ccfg.h"

/* Local helper function used by ccfg::findConflicts() */
static void _find_conflicts(p_var_sym var, list_conflict *conf);

/**
Find all the conflicting references in the procedure. For every two
statements that may execute concurrently, determine whether they access the
same variable in a conflicting manner. The possible conflicts are: DD, DU
and UD.
*/
list_conflict *
find_memory_conflicts(tree_proc *tp)
{
    D_STACK(find_memory_conflicts);
    D_SELFTEST_HEADER(110, "find_memory_conflicts");
    D_SELFTEST(110) { cout << "\nLooking for reference conflicts\n"; }

    list_conflict *conf = new list_conflict;	/* [MEMORY LEAK] */
    ::set_conflicts(tp, conf);

    /* Traverse every reference for every variable in the procedure. Compare
     * every concurrent reference with every other concurrent reference for
     * the variable to see if they conflict. A concurrent reference is one
     * made by a concurrent statement.
     */
    map<p_var_sym, bool> analyzed;
    set_varref *refs = ::get_varrefs(tp);
    for (set_varref::iterator i = refs->begin(); i != refs->end(); i++) {
	p_varref ref = *i;
	p_var_sym var = ref->var();
	if (analyzed.find(var) != analyzed.end()) {
	    continue;	/* Variable has been analyzed already. */
	}
	analyzed[var] = true;

	D_SELFTEST(111) { cout << "Analyzing variable '" << var << "'\n"; } 
	::_find_conflicts(var, conf);
    }

    D_SELFTEST(110) {
	cout << "\nConflicts found in procedure '" << tp->proc()->name()
	    << "':\n";
	copy(conf->begin(), conf->end(), ostream_iterator<conflict>(cout,"\n"));
	cout << endl;
    }

    D_SELFTEST_FOOTER(110);

    return conf;
}


/*---------------------------------------------------------------------------
	Free functions to find memory memory conflicts in the procedure
---------------------------------------------------------------------------*/
static void _find_conflicts(p_varref ref1, set_varref *refs,
	list_conflict *conf);

/**
Traverse all the references to the given variable looking for conflicting
accesses in concurrent threads.
*/
void
_find_conflicts(p_var_sym var, list_conflict *conf)
{
    D_STACK(_find_conflicts);
    D_SELFTEST_HEADER(120, "_find_conflicts(var_sym)");

    set_varref *var_refs = ::get_varrefs(var.ptr());

    D_SELFTEST(120) { 
	cout << "Analyzing reference conflicts for variable " << var->name()
	    << endl;
    }

    if (var_refs == NULL) {
	D_SELFTEST(120) { cout << "There are no references to the variable.\n";}
    }

    for (set_varref::iterator i = var_refs->begin(); i != var_refs->end(); i++){
	p_varref ref = *i;

	D_SELFTEST(120) {
	    cout << "\nAnalyzing reference "; ref->print(); cout << endl;
	}

	if (ref->are_conflicts_computed()) {
	    D_SELFTEST(120) { cout << "Conflicts gathered already.\n"; }
	    continue;
	}

	::_find_conflicts(ref, var_refs, conf);
	ref->set_conflicts_computed();
    }

    D_SELFTEST(120) {
	cout << "\n\nTotal conflicts found for variable " << var->name()
	    << endl;
	copy(conf->begin(), conf->end(),
	     ostream_iterator<conflict>(cout, "\n"));
    }

    D_SELFTEST_FOOTER(120);
}



/**
Return the list of conflicts between the given reference and the set of
references.
*/
void
_find_conflicts(p_varref ref1, set_varref *refs, list_conflict *conf)
{
    D_SELFTEST_HEADER(122, "_find_conflicts(varref, set_varref)");

    D_SELFTEST(122) {
	cout << "Analyzing conflicts for reference " << ref1 << endl;
    }

    if (ref1->is_clone()) {
	D_SELFTEST(120) {
	    cout << "Reference is a clone. Skipping.\n";
	}
	return;
    }

    vector_thread_body *threads1, *threads2;
    threads1 = ::get_thread_bodies(ref1->instr()->parent());
    if (!threads1) {
	D_SELFTEST(120) {
	    cout << "Reference is in a sequential section. Skipping.\n";
	}
	return;
    }

    D_SELFTEST(121) {
	cout << "Checking reference " << ref1 << " against:\n";
	copy(refs->begin(), refs->end(),
	     ostream_iterator<p_varref>(cout, "\n"));
    }

    for (set_varref::iterator i = refs->begin(); i != refs->end(); i++) {
	p_varref ref2 = *i;

	D_SELFTEST(122) {
	    cout << endl;
	    cout << "First reference:  "; ref1->print(); cout << endl;
	    cout << "Second reference: "; ref2->print(); cout << endl;
	}

	if (ref2->is_clone()) {
	    D_SELFTEST(122) {
	       cout << "The second reference is a clone.\n";
	    }
	    continue;
	}

	threads2 = ::get_thread_bodies(ref2->instr()->parent());
	if (!threads2) {
	    D_SELFTEST(122) {
		cout << "Second reference is in a sequential section.\n";
	    }
	    continue;
	}

	if (!::have_concurrent_threads(threads1, threads2)) {
	    D_SELFTEST(122) {
		cout << "The threads cannot execute concurrently.\n";
	    }
	    continue;
	}
	

	if ((ref1->isD() && ref2->isD())   ||
	    (ref1->isD() && ref2->isU())   ||
	    (ref1->isU() && ref2->isD())   ||
	    (ref1->isHSync() && ref2->isTSync()) ||
	    (ref1->isTSync() && ref2->isHSync()) ||
	    (ref1->isUnlock() && ref2->isLock())) 
	{
	    /* Add conflict to the list of conflicts. Make sure that the
	     * conflict is registered to the original reference in case
	     * that ref2 is a clone.
	     */
	    D_SELFTEST(122) {
		cout << "They conflict. Adding the conflict to the list.\n";
	    }
	    conf->push_back(conflict(ref1, ref2));
	} else {
	    D_SELFTEST(122) { cout << "They do not conflict.\n"; }
	}
    }

    D_SELFTEST_FOOTER(122);
}
