/**
$Id: graph.cc,v 2.28 1999/03/15 21:28:08 diego Exp $
*/
/*  Concurrent Control Flow Graph Implementation */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    Copyright (c) 1995,1996 The President and Fellows of Harvard University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file.
*/
#include <suif_copyright.h>

#include<vector>

#include <iostream.h>
#include <string.h>

#define _MODULE_ "CSSAME"

#include <suif.h>
#include <par.h>
#include <odc-util.h>
#include <d_lib.h>

#include "ccfg.h"

using namespace std;

/**
Build a new flow graph.  Flow graphs represent entire procedures.  
*/
ccfg::ccfg(p_tree_proc proc)
{
    D_SELFTEST_HEADER(1, "ccfg::ccfg");
    D_SELFTEST_HEADER(130, "ccfg::ccfg");

    D_SELFTEST(1) {
	cout << "Building CCFG for '" << proc->proc()->name() << "'\n\n";
    }

    /* initialization of private variables */
    tp = proc; 
    en = NULL;
    ex = NULL;
    doms = NULL;
    pdoms = NULL;
    idom = NULL;
    ipdom = NULL;
    df = NULL;
    rdf = NULL;
    lp_depth = NULL;
    _graph_built = false;
    _refs_collected = false;

    /* Create the initial graph with all the fall-through edges and
     * control structures. Some edges will have to be added in a second
     * pass (ie, branches for unstructured control-flow).
     */
    D_SELFTEST(130) { cout << "\n\nAdding nodes to the graph\n"; }
    GraphBody b = this->addTree(proc, false, p_ccfg_node(), p_thread_body());
    this->set_entry_node(b._begin);
    this->set_exit_node(b._end);


    /* Add a slicing edge from the entry to the exit. */
    b._begin->add_succ(b._end);

    D_SELFTEST(130) { cout << "\n\nCCFG after first pass:\n" << *this; }

    /* Second pass, add the remaining edges */
    this->addEdges();

    D_SELFTEST(1) {
	cout << "\nConcurrent Control Flow Graph for function "
	    << tp->proc()->name() << endl << *this;
    }

    D_SELFTEST(130) { cout << "CCFG after second pass:\n" << *this; }

    /* Annotate the procedure's symbol with its CCFG for future reference */
    ::set_ccfg(proc, this);

    _graph_built = true;

    D_SELFTEST_FOOTER(130);
    D_SELFTEST_FOOTER(1);
}


/*---------------------------------------------------------------------------
	    Methods to add subgraphs for each of the SUIF tree types
---------------------------------------------------------------------------*/
GraphBody
ccfg::addTree(p_tree_block block, bool ispar, p_ccfg_node pparent, 
	p_thread_body tb)
{
    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_block)");

    GraphBody b;

    if (!block) {	/* If block is empty, return nil subgraph */
	D_SELFTEST(190) { cout << "Returning nil subgraph " << b << endl; }
	D_SELFTEST_FOOTER(190);

	return b;
    }

    /* Add the begin node */
    p_ccfg_begin begin(new ccfg_begin(block, ispar, pparent, tb));
    this->addNode(begin);
    D_SELFTEST(191) { cout << "Added begin node: " << begin << endl; }


    /* Add the body */
    D_SELFTEST(191) {
	cout << "About to add sub-graph for tree_block:\n"; block->print();
    }
    b = this->addTree(block->body(), false, ispar, pparent, tb);

    /* Add the end node */
    p_ccfg_end end(new ccfg_end(block, ispar, pparent, tb, begin));
    this->addNode(end);
    D_SELFTEST(192) { cout << "Added end node: " << end << endl; }

    /* Build a sub-graph with the body and entry nodes */
    begin->add_succ(b._begin);
    b._end->add_succ(end);

    GraphBody subgraph(begin, end, b._last_tn);

    D_SELFTEST(190) { cout << "Returning sub-graph " << subgraph << endl; }
    D_SELFTEST_FOOTER(190);

    return subgraph;
}



/**
Adds a tree_node_list to the graph. It recursively traverses the given block
adding the sub-trees it finds to the graph. If the optional argument
'bracket' is true, the subgraph will be bracketed by two 'begin' and 'end'
nodes.
*/
GraphBody
ccfg::addTree(tree_node_list *body, bool bracket, bool ispar, 
	p_ccfg_node pparent, p_thread_body tb)
{
    p_ccfg_begin begin;
    p_ccfg_end end;
    GraphBody subgraph;

    D_SELFTEST_HEADER(191, "ccfg::addTree(tree_node_list)");

    if (!body || body->is_empty()) { /* Return nil subgraph if body is empty */
	D_SELFTEST(191) { cout << "Returning nil subgraph " << subgraph <<endl;}
	D_SELFTEST_FOOTER(70);

	return subgraph;
    }

    if (bracket) {
	/* Add the begin node */
	begin = new ccfg_begin(body->head()->contents, ispar, pparent, tb);
	this->addNode(begin);

	D_SELFTEST(192) { cout << "Added begin node: " << begin << endl; }
    }

    /* Iterate over all the instructions in the list adding sub-graphs and
     * attaching them to the subgraph being built.
     */
    D_SELFTEST(193) { 
	cout << "Adding sub-graph for tree_node_list:\n";
	body->print();
	cout << endl;
    }

    tree_node_list_iter iter(body);
    while (!iter.is_empty()) {
	p_tree_node tn = iter.step();

	/* Skip over null instructions. */
	if (is_nop(tn)) { continue; }

	D_SELFTEST(193) { cout << "Adding sub-graph for tree:\n"; tn->print(); }

	GraphBody b = this->addTree(tn, ispar, pparent, tb);
	subgraph.attach(b);

	D_SELFTEST(192) { cout << "Added sub-graph: " << b << endl; }

	/* Reset the iterator to the statement following the last statement
	 * of the subgraph. This is only needed for tree_instrs because
	 * high-level SUIF structures are skipped by the tree_node_list
	 * iterator.
	 */
	if (tn->is_instr()) {
	    iter.set(subgraph._last_tn->list_e());
	    tn = iter.step();
	}
    }

    if (bracket) {
	/* Add the end node */
	end = new ccfg_end(body->head()->contents, ispar, pparent, tb, begin);
	this->addNode(end);

	D_SELFTEST(192) { cout << "Added end node: " << end << endl; }

	/* Bracket the subgraph with the begin and end nodes. Note that the
	 * sub-graph just built may actually be empty. If this is the case,
	 * we simply link the begin and end nodes.
	 */
	if (subgraph.isEmpty()) {
	    begin->add_succ(end);
	} else {
	    begin->add_succ(subgraph._begin);
	    subgraph._end->add_succ(end);
	}

	subgraph._begin = begin;
	subgraph._end   = end;
    }

    D_SELFTEST(191) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(191);

    return subgraph;
}



/**
Create a subgraph for any SUIF tree_node. It determines which SUIF
construct it's dealing with and forwards the request to the appropriate
method.
*/
GraphBody
ccfg::addTree(p_tree_node tn, bool ispar, p_ccfg_node pparent, p_thread_body tb)
{
    GraphBody b;

    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_node)");

    D_SELFTEST(192) { cout << "Adding sub-graph for tree: "; tn->print(); }

    /* Note that cobegin and parloops *must* be tested first because they
     * also match other SUIF trees.
     */
    if (is_cobegin(tn)) {
	p_tree_cobegin cobegin;
	vector_tree_par *parents = ::get_tree_par_parents(tn.ptr()); 
	assert(parents);
	for (unsigned i = 0; i < parents->size(); i++) {
	    if ((*parents)[i]->entry() == tn) {
		cobegin = (*parents)[i];
		break;
	    }
	}
	assert(cobegin);
	b = this->addTree(cobegin, pparent, tb);
    } else if (is_parloop(tn)) {
	p_tree_parloop parloop;
	vector_tree_par *parents = ::get_tree_par_parents(tn.ptr()); 
	assert(parents);
	for (unsigned i = 0; i < parents->size(); i++) {
	    if ((*parents)[i]->entry() == tn) {
		parloop = (*parents)[i];
		break;
	    }
	}
	assert(parloop);
	b = this->addTree(parloop, pparent, tb);
    } else if (tn->is_instr()) {
	b = this->addTree(p_tree_instr(tn), ispar, pparent, tb);
    } else if (tn->is_loop()) {
	b = this->addTree(p_tree_loop(tn), ispar, pparent, tb);
    } else if (tn->is_for()) {
	b = this->addTree(p_tree_for(tn), ispar, pparent, tb);
    } else if (tn->is_if()) {
	b = this->addTree(p_tree_if(tn), ispar, pparent, tb);
    } else if (tn->is_block()) {
	b = this->addTree(p_tree_block(tn), ispar, pparent, tb);
    } else {
	error_line(-1, tn.ptr(), "Unknown tree_node kind %d. Definitely not "
		    "good.\n", tn->kind());
    }

    D_SELFTEST(190) { cout << "Returning sub-graph " << b << endl; }
    D_SELFTEST_FOOTER(190);

    return b;
}


/**
Adds a tree_instr. This method creates a single maximal basic block.
Several things have to be considered here:

1- Label instructions go in their own ccfg_label nodes. If the first
   instruction is a label, a label node is built and immediately returned.

2- Control transfer instructions should be the last instruction of the
   block.

3- Conflicting use references should be the first instruction of the block.

4- Conflicting definition references should be the last instruction of the
   block.
*/
GraphBody
ccfg::addTree(p_tree_instr ti, bool ispar, p_ccfg_node pparent, p_thread_body tb)
{
    D_SELFTEST_HEADER(192, "ccfg::addTree(tree_instr)");

    /* Label instruction go in their own nodes. This will avoid standard
     * basic blocks from having multiple incoming control edges. This is a
     * nice property for some dataflow analyses like the building of the
     * CSSAME form.
     */
    if (is_label(ti)) {
	p_ccfg_label labnode = new ccfg_label(ti, NULL, ispar, pparent, tb);
	this->addNode(labnode);
	GraphBody b(labnode, labnode, ti);

	D_SELFTEST(192) { cout << "Returning sub-graph " << b << endl; }
	D_SELFTEST_FOOTER(192);

	return b;
    }

    /* Null instructions are not even considered. */
    if (is_nop(ti)) {
	return GraphBody();
    }

    /* Build a new basic block with the given instruction tree. Keep
     * extending the basic block until we find one of the instructions that
     * mark the beginning of a new block.
     */
    p_ccfg_block block;
    block = new ccfg_block(ti->parent(), ti, ti, ispar, pparent, tb);
    this->addNode(block);
    GraphBody b(block, block, ti);
    
    D_SELFTEST(192) { cout << "Adding tree_instrs to basic block " << b <<endl;}

    /* Special cases. Check for control transfer instructions, conflicting
     * definitions, tail synchronization statements and mutex
     * synchronization statements. If any of these are the first
     * instruction, we are done.
     */
    if (is_endOfBlock(ti)) {
	D_SELFTEST(192) { 
	    cout << "First instruction is a node finalizer. Returning "
		<< "sub-graph: " << b << endl;
	}
	D_SELFTEST_FOOTER(192);

	return b;
    }

    /* Extend the newly built basic block. Start at the instruction
     * following the one we just inserted into the block.
     */
    tree_node_list_iter iter(ti->parent());
    iter.set(ti->list_e());
    iter.step();

    while (!iter.is_empty()) {
	p_tree_node tn = iter.step();

	/* Skip over null instructions. */
	if (is_nop(tn)) { continue; }

	/* If the tree node is not allowed inside a ccfg_block or it has to
	 * be the first instruction of ccfg_block, then we return
	 * what we have so far.
	 */
	if (!is_allowedInBlock(tn) || is_startOfBlock(tn)) {
	    break;
	}

	/* Add the instruction to the basic block */
	p_tree_instr ti = tn;
	block->extend(ti);
	b._last_tn = ti;

	/* If the current instruction must be the last instruction of the
	 * basic block, we return after adding the instruction.
	 */
	if (is_endOfBlock(ti)) {
	    break;
	}
    }

    D_SELFTEST(192) { cout << "Returning sub-graph: " << b << endl; }
    D_SELFTEST_FOOTER(192);

    return b;
}


/**
Create a subgraph for a tree_loop structure.

Associate the implicit loop labels 'continue', 'break' and 'top-of-loop' to
their corresponding nodes. The 'continue' label goes between the body and the
test, the 'break' label goes at the end of the loop and the 'top-of-loop' label
goes at the beginning of the body. 
*/
GraphBody ccfg::addTree(p_tree_loop tl, bool ispar, p_ccfg_node pparent,
    p_thread_body tb) 
{ 
    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_loop)");

    /* Add the begin node */
    p_ccfg_begin begin = new ccfg_begin(tl, ispar, pparent, tb);
    this->addNode(begin);
    D_SELFTEST(191) { cout << "Added begin node: " << begin << endl; }

    /* Add the 'top-of-loop' label */
    p_ccfg_label toplab = new ccfg_label(tl, tl->toplab(), ispar, pparent, tb);
    this->addNode(toplab);

    /* Create the subgraphs for the loop body */
    GraphBody body = this->addTree(tl->body(), true, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for loop body " << body << endl;}

    /* Add the 'continue' label */
    p_ccfg_label contlab= new ccfg_label(tl, tl->contlab(), ispar, pparent, tb);
    this->addNode(contlab);

    /* Add the test subgraph */
    GraphBody test = this->addTree(tl->test(), false, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for test " << test << endl; }

    /* Add the 'break' label */
    p_ccfg_label brklab = new ccfg_label(tl, tl->brklab(), ispar, pparent, tb);
    this->addNode(brklab);

    /* Add the end node */
    p_ccfg_end end = new ccfg_end(tl, ispar, pparent, tb, begin);
    this->addNode(end);
    D_SELFTEST(191) { cout << "Added end node: " << end << endl; }

    /* Join the nodes to the body
     * 
     * begin -> toplab -> body -> contlab -> test -> brklab -> end
     *             ^                           +
     *             |                           |
     *             +---------------------------+
     */
    begin->add_succ(toplab);

    toplab->add_succ(body._begin);
    body._end->add_succ(contlab);

    contlab->add_succ(test._begin);
    test._end->add_succ(toplab);
    test._end->add_succ(brklab);

    brklab->add_succ(end);

    GraphBody subgraph = GraphBody(begin, end, test._last_tn, true);

    D_SELFTEST(190) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(190);

    return subgraph;
}



/**
Create a subgraph for a tree_for structure.
*/
GraphBody
ccfg::addTree(p_tree_for tf, bool ispar, p_ccfg_node pparent, p_thread_body tb)
{
    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_for)");

    /* Add the begin node */
    p_ccfg_begin begin = new ccfg_begin(tf, ispar, pparent, tb);
    this->addNode(begin);
    D_SELFTEST(191) { cout << "Added begin node: " << begin << endl; }

    /* Create the sub-graphs for the tree_for */
    GraphBody lb = this->addTree(tf->lb_list(), false, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for lb: " << lb << endl; }

    GraphBody ub = this->addTree(tf->ub_list(), false, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for ub: " << ub << endl; }

    GraphBody step = this->addTree(tf->step_list(), false, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for step: " << step << endl; }

    GraphBody lpad = this->addTree(tf->landing_pad(), false, ispar, pparent, 
	    tb);
    D_SELFTEST(191) { cout << "Added sub-graph for lpad: " << lpad << endl; }

    /* Add the continue label node. This node sits between the body of the
     * for loop and the test node.
     */
    p_ccfg_label contlab=new ccfg_label(tf, tf->contlab(), ispar, pparent, tb);
    this->addNode(contlab);
    D_SELFTEST(191) { cout << "Added contlab node " << contlab << endl; }

    /* The test node checks for the upper bound and increments the index var */
    p_ccfg_test test_node = new ccfg_test(tf, ispar, pparent, tb);
    this->addNode(test_node);
    D_SELFTEST(191) { cout << "Added test node " << test_node << endl; }


    /* Add the body of the tree_for */
    GraphBody body = this->addTree(tf->body(), true, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for body: " << body << endl; }

    /* Add the break label node */
    p_ccfg_label brklab = new ccfg_label(tf, tf->brklab(), ispar, pparent, tb);
    this->addNode(brklab);

    /* Add the end node */
    p_ccfg_end end = new ccfg_end(tf, ispar, pparent, tb, begin);
    this->addNode(end);
    D_SELFTEST(191) { cout << "Added end node " << end << endl; }

    /* Join the subgraphs and return a subgraph for the structure. The
     * subgraph is (the subgraph in [] is only created if the landing pad is
     * not empty):
     *                       +--------------------------------+
     *                       |                                |
     *                       |                                v
     * begin -> lb -> ub -> step [ -> t1 -> lp -> ] body -> label -> test --+
     *                                               ^                |     |
     *                                               |                |     v
     *                                               +----------------+   brklab
     *                                                                      |
     *                                                                      v
     *                                                                     end
     */
    begin->add_succ(lb._begin);
    lb._end->add_succ(ub._begin);
    ub._end->add_succ(step._begin);
    if (lpad.isEmpty()) {
	step._end->add_succ(contlab);
    } else {
	p_ccfg_test test1 = new ccfg_test(tf, ispar, pparent, tb);
	this->addNode(test1);
	step._end->add_succ(test1);
	test1->add_succ(lpad._begin);
	lpad._end->add_succ(body._begin);
    }
    body._end->add_succ(contlab);
    contlab->add_succ(test_node);
    test_node->add_succ(body._begin);
    test_node->add_succ(brklab);
    brklab->add_succ(end);

    GraphBody subgraph = GraphBody(begin, end, body._last_tn, true);

    D_SELFTEST(190) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(190);

    return subgraph;
}



/**
Create a subgraph for a tree_if structure
*/
GraphBody
ccfg::addTree(p_tree_if ti, bool ispar, p_ccfg_node pparent, p_thread_body tb)
{
    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_if)");

    /* Add the begin node */
    p_ccfg_begin begin = new ccfg_begin(ti, ispar, pparent, tb);
    this->addNode(begin);
    D_SELFTEST(191) { cout << "Added begin node: " << begin << endl; }

    /* Create the sub-graphs for the tree_if */
    GraphBody header = this->addTree(ti->header(), false, ispar, pparent, tb);
    D_SELFTEST(191) { cout << "Added sub-graph for header: " << header << endl; }

    GraphBody then_part = this->addTree(ti->then_part(), true, ispar, pparent, 
	    tb);
    D_SELFTEST(191) { cout << "Added sub-graph for then: " << then_part << endl;}

    /* Add the jumpto label node at the beginning of the else part. */
    p_ccfg_label jumpto = new ccfg_label(ti, ti->jumpto(), ispar, pparent, tb);
    this->addNode(jumpto);

    GraphBody else_part = this->addTree(ti->else_part(), true, ispar, pparent, 
	    tb);
    D_SELFTEST(191) { cout << "Added sub-graph for else: " << else_part << endl;}

    /* Add the end node */
    p_ccfg_end end = new ccfg_end(ti, ispar, pparent, tb, begin);
    this->addNode(end);
    D_SELFTEST(191) { cout << "Added end node: " << end << endl; }

    /* Join the subgraphs and return a subgraph for the structure.
     *
     *                 [ +-> then ---------------+ ]
     *                  /                         \
     * begin -> header +                           +-> end
     *                  \                         /
     *                   +-> jumpto -> [ else ] -+
     *
     */
    p_tree_node last_tn = header._last_tn;
    begin->add_succ(header._begin);
    header._end->add_succ(jumpto);

    if (!then_part.isEmpty()) {
	header._end->add_succ(then_part._begin);
	then_part._end->add_succ(end);
	last_tn = then_part._last_tn;
    }

    if (!else_part.isEmpty()) {
	jumpto->add_succ(else_part._begin);
	else_part._end->add_succ(end);
	last_tn = else_part._last_tn;
    } else {
	jumpto->add_succ(end);
    }

    GraphBody subgraph = GraphBody(begin, end, last_tn, true);

    D_SELFTEST(190) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(190);

    return subgraph;
}



/**
Create a subgraph for tree_cobegin structure
*/
GraphBody
ccfg::addTree(p_tree_cobegin cobegin, p_ccfg_node pparent, p_thread_body tb)
{
    vector<GraphBody> tg;

    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_cobegin)");

    D_SELFTEST(193) {
	cout << "Creating graph for cobegin structure:\n";
	cobegin->print();
    }

    /* Add the cobegin node */
    p_ccfg_cobegin begin = new ccfg_cobegin(cobegin->entry(), pparent, tb);
    this->addNode(begin);
    D_SELFTEST(191) { cout << "Added begin node: " << begin << endl; }

    /* Add the body of each thread */
    for (unsigned i = 0; i < cobegin->num_threads(); i++) {
	GraphBody b = this->addTree(cobegin->thread(i), begin);
	D_SELFTEST(191) {
	    cout << "Added sub-graph for thread #" << i << ": " << b << endl;
	}
	tg.push_back(b);
    }

    /* Add the default label for the cobegin (for the mbr instruction) */
    p_ccfg_label deflab = new ccfg_label(cobegin->exit(), cobegin->deflab(), 
	    true, pparent, tb);
    this->addNode(deflab);
    D_SELFTEST(191) { cout << "Added default label node: " << deflab << endl; }

    /* Add the coend node */
    p_ccfg_coend end = new ccfg_coend(cobegin->exit(), pparent, tb, begin);
    this->addNode(end);
    D_SELFTEST(191) { cout << "Added end node: " << end << endl; }

    /* Attach the thread sub-graphs to the cobegin and coend nodes */
    for (unsigned i = 0; i < cobegin->num_threads(); i++) {
	begin->add_succ(tg[i]._begin);
	tg[i]._end->add_succ(end);
    }
    begin->add_succ(deflab);	/* Slicing edge so that we can get
				   phi-terms at the coend node*/
    deflab->add_succ(end);

    GraphBody subgraph = GraphBody(begin, end, cobegin->exit(), true);

    D_SELFTEST(190) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(190);

    return subgraph;
}


/**
Add the body of each thread. This should really be done by calling the
addTree for tree_node_lists, but because of limitations in SUIF, thread
bodies are not real lists (a statement cannot belong to more than one
tree_node_list).
*/
GraphBody
ccfg::addTree(p_thread_body tb, p_ccfg_node pparent)
{
    GraphBody body;

    D_SELFTEST_HEADER(191, "ccfg::addTree(thread_body)");

    /* Add the begin node */
    p_ccfg_begin begin = new ccfg_begin(tb->first(), true, pparent, tb);
    this->addNode(begin);
    D_SELFTEST(192) { cout << "Added begin node: " << begin << endl; }

    thread_node_iter iter(tb);
    while (!iter.is_empty()) {
	p_tree_node tn = iter.step();

	/* Skip over null instructions. */
	if (is_nop(tn)) { continue; }

	D_SELFTEST(193) { cout << "Adding sub-graph for tree:\n"; tn->print(); }

	GraphBody b = this->addTree(tn, true, pparent, tb);
	body.attach(b);

	D_SELFTEST(193) { cout << "Added sub-graph: " << b << endl; }

	/* Reset the iterator to the statement following the last statement
	 * of the subgraph. This is only needed for tree_instrs because
	 * high-level SUIF structures are skipped by the tree_node_list
	 * iterator.
	 */
	if (tn->is_instr()) {
	    iter.set(body._last_tn->list_e());
	    iter.step();
	}
    }

    /* Add the end node */
    p_ccfg_end end = new ccfg_end(tb->last(), true, pparent, tb, begin);
    this->addNode(end);
    D_SELFTEST(192) { cout << "Added end node: " << end << endl; }

    /* Return the subgraph for the thread */
    begin->add_succ(body._begin);
    body._end->add_succ(end);

    GraphBody subgraph = GraphBody(begin, end, body._last_tn);

    D_SELFTEST(191) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(191);

    return subgraph;
}


/**
Create a subgraph for a tree_parloop structure
*/
GraphBody
ccfg::addTree(p_tree_parloop parloop, p_ccfg_node pparent, p_thread_body tb)
{
    D_SELFTEST_HEADER(190, "ccfg::addTree(tree_parloop)");

    /* Add the cobegin node */
    p_ccfg_cobegin begin = new ccfg_cobegin(parloop->loop(), pparent, tb);
    this->addNode(begin);
    D_SELFTEST(191) { cout << "Added cobegin node: " << begin << endl; }


    D_SELFTEST(193) {
	cout << "Creating graph for parloop structure:\n";
	parloop->print();
    }

    /* Add the body of the parloop */
    GraphBody b;
    b = this->addTree(parloop->loop(), true, begin, parloop->thread(0));
    D_SELFTEST(191) { cout << "Added sub-graph: " << b << endl; }

    /* Add the coend node */
    p_ccfg_coend end = new ccfg_coend(parloop->loop(), pparent, tb, begin);
    this->addNode(end);
    D_SELFTEST(191) { cout << "Added coend node: " << end << endl; }

    /* Return the subgraph for the parloop */
    begin->add_succ(b._begin);
    b._end->add_succ(end);

    GraphBody subgraph = GraphBody(begin, end, b._last_tn, true);

    D_SELFTEST(190) { cout << "Returning sub-graph: " << subgraph << endl; }
    D_SELFTEST_FOOTER(190);

    return subgraph;
}


/*----------------------------------------------------------------------------
	    Methods to add edges for unstructured control flow
---------------------------------------------------------------------------*/
/**
Add edges between nodes that end in jumps and label nodes. Since
unstructured flow is only found in nodes that end in a jump and label
nodes. We check all the nodes that end in a jump, if their target is one of
the label nodes, we add an edge from the jump node to the label node.
*/
void
ccfg::addEdges()
{
    D_SELFTEST_HEADER(200, "ccfg::addEdges");

    for (unsigned i = 0; i < this->num_nodes(); i++) {
	p_ccfg_node n = this->node(i);
	
	D_SELFTEST(201) { cout << "Checking node #" << i << endl; }

	/* Only block nodes can generate unstructured control flow. */
	if (!n->is_block()) {
	    D_SELFTEST(201) { cout << "Not a block node. Skipping.\n\n"; }
	    continue;
	}

	p_ccfg_block block = n;
	p_tree_instr ti = block->in_tail();

	D_SELFTEST(200) { cout << "Checking block node #" << i << endl; }

	/* Conditional and unconditional branches */
	if (is_cbr(ti) || is_ubr(ti)) {
	    p_label_sym target = ((in_bj *)ti->instr())->target();
	    this->addEdge(block, target);
	}

	/* Multiway branches are similar, but we have to add multiple
	 * edges. We also make sure that the mbr is not actually a cobegin
	 * which is treated separately.
	 */
	if (is_mbr(ti) && !is_cobegin(ti)) {
	    in_mbr *mbr = (in_mbr *)ti->instr();
	    for (unsigned j = 0; j < mbr->num_labs(); j++) {
		this->addEdge(block, mbr->label(j));
	    }
	    this->addEdge(block, mbr->default_lab());
	}
    }

    D_SELFTEST_FOOTER(200);
}


/**
Adds an edge between the block node and the label node holding the target
label. 

Note that if we do not find the target node we can safely return,
because labels that do not have nodes associated with them are hidden
labels for high-level SUIF structures (tree_if, tree_for, etc).
*/
void
ccfg::addEdge(p_ccfg_block block, p_label_sym target)
{
    p_ccfg_label target_node = label_ccfg_node(target);
    if (!target_node) {
	cout << "Fatal. Target node for label " << target << " not found."
	    << endl << endl;
	assert(false);
    }

    D_SELFTEST(200) {
	cout << "Adding edge from block node "; block->print_brief();
	cout << " to node "; target_node->print_brief();
	cout << endl;
    }

    block->add_succ(target_node);
}



/*----------------------------------------------------------------------------
		   Destructor and helpers for class ccfg
---------------------------------------------------------------------------*/
/** Private helper functions for the CCFG destructor */
static void removeAnnotations(instruction *instr, void *x);
static void removeAnnotations(tree_node *tn, void *x);
static void removeAnnotationsFrom(suif_object *obj);

/**
Destroy the ccfg graph and remove the ccfg annotations on all of the
tree_instrs and label_syms.  Notice that this is not entirely necessary
since the ccfg annotations will not be written out with the procedure.
*/
ccfg::~ccfg ()
{
    D_SELFTEST_HEADER(140, "ccfg::~ccfg");

    /* Remove nodes. */
    D_SELFTEST(140) { cout << "Removing nodes from the graph\n"; }

    _nds.erase(_nds.begin(), _nds.end());

    /* Remove various bitsets. */
    D_SELFTEST(140) { cout << "Removing various bitsets\n"; }

    if (doms) delete[] doms;
    if (pdoms) delete[] pdoms;
    if (idom) delete[] idom;
    if (ipdom) delete[] ipdom;
    if (df) delete[] df;
    if (rdf) delete[] rdf;
    if (lp_depth) delete[] lp_depth;

    /* Remove annotations from the procedure's body */
    D_SELFTEST(140) { cout << "Removing SUIF annotations\n"; }

    /* Remove all the CCFG/CSSAME annotations from the symbols. */
    base_symtab *current = tp->proc_syms();
    for (; current != NULL; current = current->parent()) {
	sym_node_list_iter iter(current->symbols());
	while (!iter.is_empty()) { removeAnnotationsFrom(iter.step()); }
    }

    current = tp->proc_syms();
    base_symtab_list_iter child_iter(current->children());
    while (!child_iter.is_empty()) {
	sym_node_list_iter iter(child_iter.step()->symbols());
	while (!iter.is_empty()) { removeAnnotationsFrom(iter.step()); }
    }

    /* Remove all the annotations from every statement in the procedure. */
    tp->map(removeAnnotations, NULL);

    /* Finally, remove any annotation that we might have placed on the
     * procedure itself.
     */
    removeAnnotationsFrom(tp.ptr());

    D_SELFTEST_FOOTER(140);
}


/**
Helper function mapped over all the tree nodes in the program to remove the
CCFG annotations used.
*/
void
removeAnnotations(tree_node *tn, void *)
{
    D_STACK(removeAnnotations_CCFG);
    D_SELFTEST_HEADER(240, "removeAnnotations_CCFG");

    if (tn->is_instr()) {
	/* Remove annotations from the instructions contained in tn. */
	tree_instr *ti = (tree_instr *)tn;
	ti->instr_map(removeAnnotations, NULL);
    } else {
	/* Recursively map this function over the tree nodes in tn. */
	tn->map(removeAnnotations, NULL);
    }

    /* Remove CSSAME annotations from tn itself. */
    removeAnnotationsFrom(tn);

    D_SELFTEST(240) {
	cout << "CCFG annotations for tree_node AFTER removal:\n";
	tn->print();
	cout << endl << endl;
	if (tn->are_annotations()) {
	    annote_list_iter iter(tn->annotes());
	    int cnt = 1;
	    while (!iter.is_empty()) { 
		cout << cnt++ << "- "; iter.step()->print(); cout << endl;
	    }
	    cout << endl << endl;
	}
    }

    D_SELFTEST_FOOTER(240);
}


/**
Helper function mapped over all the instructions in a tree_instr node to
remove the data flow annotations used.
*/
void
removeAnnotations(instruction *instr, void *)
{
    D_STACK(removeAnnotationsInstr_CCFG);
    D_SELFTEST_HEADER(250, "removeAnnotationsInstr_CCFG");

    removeAnnotationsFrom(instr);

    D_SELFTEST(250) {
	cout << "CCFG annotations for instruction AFTER removal: ";
	instr->print();
	cout << endl << endl;
	if (instr->are_annotations()) {
	    annote_list_iter iter(instr->annotes());
	    int cnt = 1;
	    while (!iter.is_empty()) { 
		cout << cnt++ << "- "; iter.step()->print(); cout << endl;
	    }
	    cout << endl << endl;
	}
    }

    D_SELFTEST_FOOTER(250);
}


/**
Removes all the CCFG annotations from the given SUIF object.
NOTE: If you add a new k_ annotation for CCFG and/or CSSAME, make sure 
      to remove it here.
*/
void
removeAnnotationsFrom(suif_object *obj)
{
    D_STACK(removeAnnotationsFrom);
    D_SELFTEST_HEADER(260, "removeAnnotationsFrom");

    obj->get_annote(k_ccfg);
    obj->get_annote(k_ccfg_begin);
    obj->get_annote(k_ccfg_cobegin);
    obj->get_annote(k_ccfg_coend);
    obj->get_annote(k_ccfg_end);
    obj->get_annote(k_ccfg_node);
    obj->get_annote(k_ccfg_test);
    obj->get_annote(k_ccfg_toplab);
    obj->get_annote(k_conflicts);
    obj->get_annote(k_currdef);
    obj->get_annote(k_defs);
    obj->get_annote(k_entry_set);
    obj->get_annote(k_exit_set);
    obj->get_annote(k_lock_refs);
    obj->get_annote(k_mutex_struct);
    obj->get_annote(k_mutex_sync);
    obj->get_annote(k_phi_terms);
    obj->get_annote(k_pi_terms);
    obj->get_annote(k_proc_entry);
    obj->get_annote(k_reached_uses);
    obj->get_annote(k_reaching_defs);
    obj->get_annote(k_refs);
    obj->get_annote(k_sync_var);
    obj->get_annote(k_unlock_refs);
    obj->get_annote(k_uses);

    /* [HACK] 
     * These annotations should be removed by the tree_par destructor.
     */
    obj->get_annote(k_par_parents);
    obj->get_annote(k_thread_bodies);

    D_SELFTEST(260) {
	cout << "\nAnnotations left for SUIF object ";
	if (obj->is_tree_obj()) {
	    cout << "tree_node:\n";
	    ((tree_node *)obj)->print();
	} else if (obj->is_instr_obj()) {
	    cout << "instruction:\n";
	    ((instruction *)obj)->print();
	} else if (obj->is_sym_obj()) {
	    cout << "symbol:\n";
	    ((sym_node *)obj)->print();
	} else {
	    cout << " ???:\n";
	}
	cout << endl;
	obj->print_annotes();
	cout << endl;
    }

    D_SELFTEST_FOOTER(260);
}


/* 
 * Set the entry or exit node of the graph.  These should probably be
 * protected, because users don't get to add entry or end
 * nodes to a CCFG.  
 */
void
ccfg::set_entry_node (p_ccfg_node n)
{
    assert(n->is_begin());
    en = n;
}

void
ccfg::set_exit_node (p_ccfg_node n)
{
    assert(n->is_end());
    ex = n;
}


/**
Remove unreachable nodes from a flow graph.  This simplifies finding
dominators and may also be used after manually removing nodes (e.g.
unwanted BEGIN and END markers, nop instructions, etc.) from the graph.
Note: There should never be unreachable nodes in the reverse graph.
*/
static void unreachable_visit(p_ccfg_node n, bit_set *mark);

void
ccfg::remove_unreachable_blocks()
{
    unsigned num = this->num_nodes();
    unsigned i;

    /* find the reachable blocks */
    bit_set mark(0, num);
    unreachable_visit(entry_node(), &mark);

    /* copy the reachable nodes to a new array */
    vector_ccfg_node new_nds;
    for (i = 0; i < num; i++) {
	p_ccfg_node nd = node(i);
	if (mark.contains(i)) {
	    nd->set_number(new_nds.size());
	    new_nds.push_back(nd);
	} else {
	    /* detach the unreachable node from its successors */
	    ccfg_node_list_iter succ_iter(nd->succs());
	    while (!succ_iter.is_empty()) {
		ccfg_node *succ = succ_iter.step();
		nd->remove_succ(succ);
	    }
	    /* detach the unreachable node from its predecessors -- this is
               needed when there's a back-edge from a higher numbered node
	       to a lower numbered node */
	    ccfg_node_list_iter pred_iter(nd->preds());
	    while (!pred_iter.is_empty()) {
		ccfg_node *pred = pred_iter.step();
		nd->remove_pred(pred);
	    }
	    assert(nd->succs()->is_empty() && nd->preds()->is_empty());
	}
    }

    _nds = new_nds;

    delete[] lp_depth; 
    lp_depth = NULL; 
}

static void
unreachable_visit(p_ccfg_node n, bit_set *mark)
{
    /* mark the node as reachable */
    mark->add(n->number());

    /* visit its successors */
    ccfg_node_list_iter succ_iter(n->succs());
    while (!succ_iter.is_empty()) {
	p_ccfg_node succ = succ_iter.step();
	if (!mark->contains(succ->number())) {
	    unreachable_visit(succ, mark);
	}
    }
}




/**
Build a list of the flow graph nodes in reverse postorder. When forward ==
false, build it in reverse, giving plain postorder.
*/
static void rpostord_visit(ccfg_node_list *result, p_ccfg_node n, bit_set *mark,
			   bool forward);

ccfg_node_list *
ccfg::reverse_postorder_list (bool forward)
{
    ccfg_node_list *result = new ccfg_node_list;
    bit_set mark(0, num_nodes());
    rpostord_visit(result, entry_node(), &mark, forward);
    return result;
}


void
rpostord_visit (ccfg_node_list *result, p_ccfg_node n, bit_set *mark,
		bool forward)
{
    ccfg_node_list_iter iter(n->succs());
    while (!iter.is_empty()) {
	p_ccfg_node s = iter.step();

	/* check if the node has already been visited */
	if (mark->contains(s->number())) continue;

	mark->add(s->number());
	rpostord_visit(result, s, mark, forward);
    }

    /* add this block to the front (for rpo) or back (plain po)
     * of the result list */
    if (forward)
	result->push(n.ptr());
    else
        result->append(n.ptr());
}


/**
Find the dominators of each node.  The "remove_unreachable_blocks" method must
be called prior to this or the results may be incorrect.
*/
void
ccfg::find_dominators ()
{
    /* delete any old information */
    if (doms) delete[] doms;
    if (idom) delete[] idom;

    /* allocate and initialize the new bit vectors */
    doms = _dominators(true);
    idom = immed_dominators(true);
}


/**
Find the postdominators of each node.  A legal graph should never have
unreachable nodes in the reverse graph.  If that is not true, this will not
work correctly.
*/
void
ccfg::find_postdominators ()
{
    /* delete any old information */
    if (pdoms) delete[] pdoms;
    if (ipdom) delete[] ipdom;

    /* compute the dominators */
    pdoms = _dominators(false);
    ipdom = immed_dominators(false);
}


/**
Compute (post)dominators.  This is a protected method that is used internally
to do the actual work of finding dominators.
*/
bit_set *
ccfg::_dominators(bool forward)
{
    ccfg_node_list *node_list = reverse_postorder_list(forward);
    bool changed = false;
    unsigned num = num_nodes();

    /* allocate and initialize the new bit vectors */
    bit_set *d = new bit_set[num];
    for (unsigned i = 0; i < num; i++) {
	d[i].expand(0, num);
	d[i].universal();
    }

    /* set up the first node */
    p_ccfg_node start = (forward ? entry_node() : exit_node());
    d[start->number()].clear();
    d[start->number()].add(start->number());

    bit_set t(0, num_nodes());

    /* iterate until no changes */
    do {
	changed = false;
	ccfg_node_list_iter node_iter(node_list);
	while (!node_iter.is_empty()) {
	    p_ccfg_node n = node_iter.step();
	    if (n == start) continue;

	    /* get intersection of predecessors of n */
	    t.universal();
	    ccfg_node_list_iter pred_iter(forward ? n->preds() : n->succs());
	    while (!pred_iter.is_empty()) {
		ccfg_node *p = pred_iter.step();
		t *= d[p->number()];
	    }

	    /* include itself in dominator set */
	    t.add(n->number());

	    /* check if there were any changes */
	    if (d[n->number()] != t) {
		d[n->number()] = t;
		changed = true;
	    }
	}
    } while (changed);

    delete node_list;
    return d;
}


/**
Find the immediate (post)dominators.  This is a protected method that is only
used internally to do the actual work of finding immediate dominators.
*/
ccfg_node **
ccfg::immed_dominators (bool forward)
{
    unsigned num = num_nodes();
    unsigned root = (forward ? entry_node()->number() : exit_node()->number());
    bit_set *d = (forward ? doms : pdoms);

    ccfg_node **im = new ccfg_node*[num];
    im[root] = NULL;

    for (unsigned n = 0; n < num; n++) {

	/* the root node has no dominators */
	if (n == root) continue;

	/* remove self for comparison */
	d[n].remove(n);

	/* check nodes in dominator set */
	bool found = false;
	bit_set_iter diter(&d[n]);
	while (!diter.is_empty()) {
	    unsigned i = diter.step();

	    if (i < num && d[n] == d[i]) {
		found = true;
		im[n] = node(i).ptr();
	    }
	}

	if (!found) {
	    warning_line(NULL, "can't find idom of node %u", n);
	    im[n] = NULL;
	}

	/* re-add self when done */
	d[n].add(n);
    }

    return im;
}


/**
Find the dominance frontier of each node.  The dominators must have
already been computed.
*/
void
ccfg::find_df ()
{
    assert_msg(doms, ("ccfg::find_df - need to compute dominators first"));

    /* delete any old information */
    if (df) delete[] df;

    df = new bit_set[num_nodes()];
    dom_frontiers(entry_node(), true);
}


/**
Find the reverse dominance frontier of each node.  The postdominators
must have already been computed.
*/
void
ccfg::find_reverse_df ()
{
    assert_msg(pdoms, ("ccfg::find_reverse_df - need postdominators first"));

    /* delete any old information */
    if (rdf) delete[] rdf;

    rdf = new bit_set[num_nodes()];
    dom_frontiers(exit_node(), false);
}


/**
Compute dominance frontiers using a postorder traversal of the dominator
tree.  This is a protected method that is only used internally.
*/
void
ccfg::dom_frontiers (p_ccfg_node x, bool forward)
{
    unsigned n, num = num_nodes();
    ccfg_node **xidom;
    bit_set *dom_front;
    ccfg_node_list *nodes;

    if (forward) {
	xidom = idom;
	dom_front = df;
	nodes = x->succs();
    } else {
	xidom = ipdom;
	dom_front = rdf;
	nodes = x->preds();
    }

    /* visit all children (i.e. immediate dominatees) first */
    for (n = 0; n < num; n++) {
	if (xidom[n] == x.ptr()) {
	    dom_frontiers(node(n), forward);
	}
    }

    /* calculate dominance frontier, from paper RCytron_89 */
    dom_front[x->number()].expand(0, num);

    /* local loop, uses CCFG */
    ccfg_node_list_iter succ_iter(nodes);
    while (!succ_iter.is_empty()) {
	ccfg_node *s = succ_iter.step();

	if (x.ptr() != xidom[s->number()]) {
	    dom_front[x->number()].add(s->number());
	}
    }

    /* up loop, uses dominator tree */
    for (n = 0; n < num; n++) {
	if ((xidom[n] == x.ptr()) && !dom_front[n].is_empty()) {
	    for (unsigned y = 0; y < num; y++) {
		if (dom_front[n].contains(y) && (x.ptr() != xidom[y])) {
		    dom_front[x->number()].add(y);
		}
	    }
	}
    }
}


/* Access routines for dominators and dominance frontiers. */

bool
ccfg::dominates (int n_dominator, int n_dominatee)
{
    assert_msg(doms, ("ccfg::dominates() -- run find_dominators() first"));
    return doms[n_dominatee].contains(n_dominator);
}

bool
ccfg::dominates (p_ccfg_node dominator, p_ccfg_node dominatee)
{
    return dominates(dominator->number(), dominatee->number());
}

bool
ccfg::postdominates (int n_dominator, int n_dominatee)
{
    assert_msg(pdoms, ("ccfg::postdominates() -- "
		       "run find_postdominators() first"));
    return pdoms[n_dominatee].contains(n_dominator);
}

bool
ccfg::postdominates (p_ccfg_node dominator, p_ccfg_node dominatee)
{
    return postdominates(dominator->number(), dominatee->number());
}

/* Return the set of dominators of node n */
bit_set *
ccfg::dominators(int n)
{
    assert_msg(doms, ("ccfg::dominators() -- run find_dominators() first"));
    return &(doms[n]);
}

bit_set *
ccfg::dominators(p_ccfg_node n)
{ 
    return dominators(n->number());
}

/* Return the set of post-dominators of node n */
bit_set *
ccfg::postdominators(int n)
{
    assert_msg(pdoms, ("ccfg::postdominators() -- run find_postdominators() first"));
    return &(pdoms[n]);
}

bit_set *
ccfg::postdominators(p_ccfg_node n)
{ 
    return postdominators(n->number());
}

p_ccfg_node
ccfg::immed_dom (int n)
{
    assert_msg(idom, ("ccfg::immed_dom() -- run find_dominators() first"));
    return idom[n];
}

p_ccfg_node
ccfg::immed_dom (p_ccfg_node n)
{
    return immed_dom(n->number());
}

p_ccfg_node
ccfg::immed_postdom (int n)
{
    assert_msg(ipdom, ("ccfg::immed_postdom() -- "
		       "run find_postdominators() first"));
    return ipdom[n];
}

p_ccfg_node
ccfg::immed_postdom (p_ccfg_node n)
{
    return immed_postdom(n->number());
}

bit_set *
ccfg::dom_frontier (int n)
{
    assert_msg(df, ("ccfg::dom_frontier() -- run find_df first"));
    return &df[n];
}

bit_set *
ccfg::dom_frontier(p_ccfg_node n)
{
    return dom_frontier(n->number());
}

bit_set *
ccfg::reverse_dom_frontier (int n)
{
    assert_msg(rdf, ("ccfg::reverse_dom_frontier() -- "
		     "run find_reverse_df() first"));
    return &rdf[n];
}

bit_set *
ccfg::reverse_dom_frontier(p_ccfg_node n)
{
    return reverse_dom_frontier(n->number());
}


void 
ccfg::find_natural_loops()
{
    D_SELFTEST_HEADER(170, "ccfg::find_natural_loops");

    int i;
    int num = num_nodes();
    int *node_stack = new int[num]; 

    if (lp_depth) delete[] lp_depth; 
    lp_depth = new int[num]; 

    assert((doms != NULL) && (pdoms != NULL));

    /* initialize */
    for (i = 0; i < num; i++) 
	lp_depth[i] = 0; 

    /* find all the loop entries and loop-back blocks */
    for (i = 0; i < num; i++) {
	p_ccfg_node cn = node(i);
	if (cn->is_block() || cn->is_test() || cn->is_label()) {
	    int top_stack = 0; 
	    bit_set this_loop(0, num); 

	    /* if some predecessor induces a back edge, 
	     * then use it to seed the loop
	     */
	    D_SELFTEST(170) {
		cout << "Checking if predecessors of " << cn->number()
		    << " induce a back edge" << endl;
	    }

	    ccfg_node_list_iter pred_iter(cn->preds());
	    while(!pred_iter.is_empty()) {
		ccfg_node *pred_cn = pred_iter.step();
		D_SELFTEST(170) {
		    cout << "\tchecking predecessor " << pred_cn->number()
			<< endl;
		}
		
		if (dominates(cn, pred_cn)) {
		    D_SELFTEST(170) {
			cout << "\tYes. " << pred_cn->number()
			    << " is dominated by " << cn->number()
			    << ". Add it to the stack.\n";
		    }
		    node_stack[top_stack++] = pred_cn->number(); 
		}
	    }

	    /* there was some back edge, so this is a header */
	    if (top_stack) {
		++lp_depth[i]; 
		this_loop.add(i); 
	    }

	    /* walk backwards around loop, adding nodes */
	    while (top_stack) {
		int top = node_stack[--top_stack]; 
		if (!this_loop.contains(top)) {
		    D_SELFTEST(170) {
			cout << "Incrementing lp_depth for node " << top
			    << endl;
		    }
		    
		    this_loop.add(top); 
		    ++lp_depth[top]; 
		    ccfg_node_list_iter top_iter(node(top)->preds());
		    while (!top_iter.is_empty()) {
			int m = top_iter.step()->number(); 
			node_stack[top_stack++] = m; 
			assert(top_stack < num); 
		    }
		}
	    }
	}
    }

    delete[] node_stack; 

    D_SELFTEST(170) {
	cout << "Loop depths for nodes in the graph\n";
	for (unsigned i = 0; i < this->num_nodes(); i++) {
	    cout << this->node(i) << " -> Loop depth: " << lp_depth[i] << endl;
	}

	cout << endl << endl << "Loop headers for each node in the graph\n";
	for (unsigned i = 0; i < this->num_nodes(); i++) {
	    cout << this->node(i) << "\n\t-> Loop Header: " <<
		this->outermostLoopHeader(this->node(i)) << endl;
	}
	cout << endl;
    }


    D_SELFTEST_FOOTER(170);
}

int
ccfg::loop_depth (int n)
{
    assert_msg(lp_depth, ("ccfg::loop_depth() -- "
	"run find_natural_loops() first"));
    return lp_depth[n];
}

int
ccfg::loop_depth(p_ccfg_node n)
{
    return loop_depth(n->number());
}

void 
ccfg::set_loop_depth(int n, int d)
{
    assert_msg(lp_depth, ("ccfg::loop_depth() -- "
	"run find_natural_loops() first"));
    lp_depth[n] = d;
}

void 
ccfg::set_loop_depth(p_ccfg_node n, int d)
{
    set_loop_depth(n->number(), d);
}


bool
ccfg::is_loop_begin (int n)
{
    return is_loop_begin(node(n));
}

bool
ccfg::is_loop_begin(p_ccfg_node cn)
{
    ccfg_node_list_iter pred_iter(cn->preds());
    while (!pred_iter.is_empty()) {
	ccfg_node *pred_cn = pred_iter.step();
	if (dominates(cn, pred_cn)) return true;
    }
    return false;
}

bool
ccfg::is_loop_end (int n)
{
    return is_loop_end(node(n));
}

bool
ccfg::is_loop_end(p_ccfg_node cn)
{
    ccfg_node_list_iter succ_iter(cn->succs());
    while (!succ_iter.is_empty()) {
	ccfg_node *succ_cn = succ_iter.step();
	if (dominates(succ_cn, cn)) return true;
    }
    return false;
}

bool
ccfg::is_loop_exit (int n)
{
    return is_loop_exit(node(n));
}

bool
ccfg::is_loop_exit(p_ccfg_node cn)
{
    ccfg_node_list_iter succ_iter(cn->succs());
    while (!succ_iter.is_empty()) {
	ccfg_node *succ_cn = succ_iter.step();
	if (loop_depth(succ_cn) != loop_depth(cn)) return true;
    }
    return false;
}


/**
Returns the header node for the outermost loop enclosing the given node. It
works by going up in the dominator tree looking for loop header nodes. We
stop when we find the first node header with depth 1.
*/
p_ccfg_node
ccfg::outermostLoopHeader(p_ccfg_node node)
{
    D_SELFTEST_HEADER(180, "ccfg::outermostLoopHeader");

    assert_msg(lp_depth, ("Run ccfg::find_natural_loops() first\n"));

    D_SELFTEST(180) {
	cout << "Looking for loop header for node #" << node->number() << endl;
    }

    /* Check trivial cases first. If the node is not inside a loop, return
     * NULL. If the node is already an outermost loop header, return it.
     */
    if (this->loop_depth(node) < 1) {
	D_SELFTEST(180) { cout << "\nNode's loop depth is less than 1.\n"; }
	D_SELFTEST_FOOTER(180);

	return NULL;
    }

    if (this->is_loop_begin(node) && this->loop_depth(node) == 1) {
	D_SELFTEST(180) { cout << "Node is its own loop header\n"; }
	D_SELFTEST_FOOTER(180);

	return node;
    }

    /* Walk up the dominator tree looking for a loop header with loop depth
     * of 1.
     */
    p_ccfg_node header;
    p_ccfg_node candidate = this->immed_dom(node);

    while (candidate) {

	D_SELFTEST(180) {
	    cout << "Checking dominator(" << node->number() << ") = " 
		<< candidate->number() << endl;
	}


	if (this->is_loop_begin(candidate)) {
	    D_SELFTEST(180) {
		cout << "\tNode " << candidate->number()
		    << " is a loop header and dominates node " << node->number()
		    << endl;
	    }

	    header = candidate;
	    if (this->loop_depth(header) == 0) {
		break;			/* We found the outermost header */
	    }
	}
	candidate = this->immed_dom(candidate);
    }

    D_SELFTEST(180) { cout << "Loop header for node #" << node->number()
	<< " is node #" << ((header) ? header->number() : -1) << endl;
    }
    D_SELFTEST_FOOTER(180);

    return header;
}



/*---------------------------------------------------------------------------
			      Protected functions
---------------------------------------------------------------------------*/
/**
Add a node to a ccfg, without any error checking.  
*/
void
ccfg::addNode(p_ccfg_node n)
{
    unsigned num = _nds.size();
    _nds.push_back(n);
    n->set_number(num);
    n->set_parent(this);
    delete[] lp_depth;
    lp_depth = new int[num + 1];
}


void
ccfg::print(ostream& f) const
{
    f << num_nodes() << " blocks" << endl;

    for (unsigned i = 0; i < num_nodes(); ++i) {
	node(i)->print(f);
	f << endl;
    }
}

ostream& 
operator<<(ostream &f, const ccfg& graph)
{
    graph.print(f);

    return f;
}


/**
Returns true if blocks a and b can execute concurrently. That is, both a and b
are in separate threads and share a common cobegin/parloop ancestor.
*/
bool
ccfg::conc(p_ccfg_node a, p_ccfg_node b)
{
    /* Check if they're both the same node */
    if (a->number() == b->number()) return false;

    /* Check if one of them is not in a thread */
    if (!a->par_parent() || !b->par_parent()) return false;

    /* Check if they are on the same thread */
    if (a->thread() == b->thread()) return false;

    /* Find out the concurrent ancestors for each node, if they share at
     * least one ancestor then they may execute concurrently.
     */
    bit_set *a_panc = a->par_ancestors();
    bit_set *b_panc = b->par_ancestors();

    return (*a_panc ^ *b_panc);
}


/**
Returns the set of all the parallel ancestors for the given node.
*/
bit_set *
ccfg::par_ancestors(p_ccfg_node n)
{
    bit_set *panc = new bit_set(0, this->num_nodes());
    p_ccfg_node par_parent = n->par_parent();

    panc->clear();
    
    while (par_parent) {
	panc->add(par_parent->number());
	par_parent = par_parent->par_parent();
    }

    return panc;
}


static bool path_visit(p_ccfg_node from, p_ccfg_node to, bit_set& visited);

/**
Returns true if there is a control path from the first node to the second.
*/
bool
ccfg::isPath(p_ccfg_node from, p_ccfg_node to)
{
    D_SELFTEST_HEADER(230, "ccfg::isPath");

    D_SELFTEST(230) { 
	cout << "Checking if there's a path from node:\n\t" << from << endl;
	cout << "to node:\n\t" << to << endl;
    }

    /* Trivial test, if both nodes are in different graphs just return false */
    if (from->parent() != to->parent()) {
	return false;
    }

    /* The visited bit vector is used to know whether we visited a given
     * node. All the bits are initially cleared.
     *
     * [HACK]  We use the entry's node graph and not this graph because we
     * might be dealing with a mutex body from another graph. We really
     * should not need to do this. Each CCFG should mind their own
     * business.
     */
    bit_set visited(0, from->parent()->num_nodes());
    D_SELFTEST(231) {
	cout << "Created visited bitset with " << visited.ub() - visited.lb()
	    << " bits.\n";
    }
    bool is_path = path_visit(from, to, visited);

    D_SELFTEST(230) {
	cout << ((is_path) ? "Yes, there is a path." : "No, there is no path.")
	    << endl;
    }
    D_SELFTEST_FOOTER(230);

    return is_path;
}


static bool
path_visit(p_ccfg_node from, p_ccfg_node to, bit_set& visited)
{
    D_STACK(path_visit);
    D_SELFTEST(231) {
	cout << "\nUsing bitset with " << visited.ub() - visited.lb()
	    << " bits\n";
	cout << "Visiting path from node\n\t" << from << endl;
	cout << "to node\n\t" << to << endl;
    }

    /* Simplest case, both nodes are the same */
    if (from == to) { 
	D_SELFTEST(231) { cout << "Found a path.\n"; }
	return true;
    }

    /* Nodes in different graphs, there can't be a path */
    if (from->parent() != to->parent()) { 
	D_SELFTEST(231) { cout << "Nodes belong to different graphs.\n"; }
	return false;
    }

    /* Recursive plunge. Depth first search on our successors. Check if
     * there is a path from each of our successors to the node. Eureka jump
     * on the first one that we find.
     */
    bool foundPath = false;
    ccfg_node_list_iter succ_iter(from->succs());
    while (!succ_iter.is_empty()) {
	p_ccfg_node succ = succ_iter.step();

	D_SELFTEST(231) {cout << "Visiting from's successor " << succ << endl;}

	if (visited.contains(succ->number())) {
	    /* If we've already visited this node, give up. */
	    D_SELFTEST(231) { cout << "Node already visited. Giving up.\n"; }
	    foundPath = false;
	    break;
	}

	D_SELFTEST(231) { cout << "Marking node as visited and recursing.\n"; }
	visited.add(succ->number());
	foundPath = path_visit(succ, to, visited);
	if (foundPath == true) {
	    break;
	}
    }

    return foundPath;
}



/*---------------------------------------------------------------------------
			Methods for class GraphBody
---------------------------------------------------------------------------*/
/**
Merge this sub-graph with the given graph. The end result of this operation
is the following:

this graph:  begin1 -> body1 -> end1
new graph:   begin2 -> body2 -> end2

this->attach(new):  begin1 -> body1 -> end1 [ -> ] begin2 -> body2 -> end2

The edge between the old end1 and begin2 depends on whether this graph's
last instruction falls through or not.

After merging, the end node for this graph is reset to end2.
*/
void
GraphBody::attach(const GraphBody& b)
{
    D_SELFTEST_HEADER(270, "GraphBody::attach");

    D_SELFTEST(270) {
	cout << "Attaching subgraph: " << b << endl;
	cout << "to subgraph:        " << *this << endl;
    }

    /* If the subgraph is empty, do nothing. */
    if (b.isEmpty()) {
	return;
    }

    if (this->isEmpty()) {
	/* First time. Initialize this graph body with the given graph body */
	*this = b;
    } else {
	/* Attach the new graph body to this one. Note that we only attach
	 * the new subgraph if the current subgraph has a fall-through
	 * edge.
	 *
	 * If the last instruction of this graph body does not fall through
	 * and the graph body is not one of the high-level control
	 * structures (ie, it's an unconditional branch or a multi-way
	 * branch or a return statement) then we don't attach the new
	 * subgraph. It will be attached later when we add the edges for
	 * all the jump instructions after all the subgraphs have been
	 * built.
	 */
	if (is_fall_through(_last_tn) || _is_ctrl_struct) {
	    _end->add_succ(b._begin);
	}
	_end     = b._end;
	_last_tn = b._last_tn;
	_is_ctrl_struct = b._is_ctrl_struct;
    }

    D_SELFTEST(270) { cout << "Resulting subgraph: " << *this << endl; }

    D_SELFTEST_FOOTER(270);
}


ostream& operator<<(ostream& s, const GraphBody& b)
{
    if (b.isEmpty()) { s << "[]"; return s; }

    s << "[";
    b._begin->print_brief(s);
    cout << ", ";
    b._end->print_brief(s);
    cout << ", " << "last_tn: ";
    if (b._last_tn->is_instr()) {
	s << p_tree_instr(b._last_tn)->instr()->number();
    } else {
	s << b._last_tn->number();
    }
    s << "]";

    s << ((b._is_ctrl_struct) ? " (CONTROL)" : " (block)");

    return s;
}
