/**
$Id: graph.h,v 2.15 1999/03/05 14:50:38 diego Exp $
*/
/*  Control Flow Graph */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    Copyright (c) 1995, 1996, 1997 The President and Fellows of Harvard 
    University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file.
*/
#ifndef CCFG_GRAPH_H
#define CCFG_GRAPH_H

#include <set>
#include <list>
#include <iostream.h>

#include <suif_copyright.h>

#include <d_lib.h>

using namespace std;

class ccfg_node;
class ccfg_block;
class conflict;
class varref;

DECLARE_LIST_CLASS(ccfg_node_list, ccfg_node*);


/* Graph body represents the entry and exit nodes to a sub-graph. */
class GraphBody { D_CLASS(GraphBody)
public:
    p_ccfg_node _begin;
    p_ccfg_node _end;
    p_tree_node _last_tn;
    bool _is_ctrl_struct;

    GraphBody(p_ccfg_node b = 0, p_ccfg_node e = 0, p_tree_node last_tn = 0, 
	    bool ctrl = false) : 
	_begin(b), _end(e), _last_tn(last_tn), _is_ctrl_struct(ctrl) {}

    GraphBody(const GraphBody& b) : 
	_begin(b._begin), _end(b._end), _last_tn(b._last_tn),
	_is_ctrl_struct(b._is_ctrl_struct) {}

    GraphBody& operator=(const GraphBody& b) {
	_begin = b._begin; _end = b._end; _last_tn = b._last_tn;
	_is_ctrl_struct = b._is_ctrl_struct;
	return *this;
    }

    bool isEmpty() const { return (!_begin && !_end && !_last_tn); }
    void attach(const GraphBody& b);
};

ostream& operator<<(ostream& s, const GraphBody& b);


/**
The control flow graph defined here is intended to be general enough for
almost everyone to use (at least as a base -- other information can be
added on top of it).  

It supports nodes containing either individual tree_instrs or basic blocks
of contiguous tree_instrs. High-level tree nodes are represented by
multiple flow graph nodes corresponding to the code that will be created
when they are expanded. Moreover, each high-level tree node is delimited by
a pair of BEGIN and END nodes to make it easier to match up the flow graph
with the SUIF code.
*/
class ccfg { D_CLASS(ccfg)
public:
    ccfg(p_tree_proc proc);
    ~ccfg();

    /* Access methods */
    p_tree_proc tproc()					{ return tp; }
    unsigned num_nodes() const				{ return _nds.size(); }
    const p_ccfg_node &node(unsigned i)	const 		{ return _nds[i]; }
    const p_ccfg_node &operator[](unsigned i) const	{ return _nds[i]; }
    p_ccfg_node& entry_node()				{ return en; }
    p_ccfg_node& exit_node()				{ return ex; }
    void set_entry_node(p_ccfg_node n);
    void set_exit_node(p_ccfg_node n);

    /* built-in analysis routines */
    void find_dominators();
    void find_postdominators();
    void find_df();                     /* dominance frontier */
    void find_reverse_df();

    /* access dfa information */
    bool dominates(int n_dominator, int n_dominatee);
    bool dominates(p_ccfg_node dominator, p_ccfg_node dominatee);
    bool postdominates(int n_dominator, int n_dominatee);
    bool postdominates(p_ccfg_node dominator, p_ccfg_node dominatee);
    p_ccfg_node immed_dom(int n);
    p_ccfg_node immed_dom(p_ccfg_node n);
    p_ccfg_node immed_postdom(int n);
    p_ccfg_node immed_postdom(p_ccfg_node n);
    bit_set *dom_frontier(int n);
    bit_set *dom_frontier(p_ccfg_node n);
    bit_set *reverse_dom_frontier(int n);
    bit_set *reverse_dom_frontier(p_ccfg_node n);
    bit_set *dominators(int n);
    bit_set *dominators(p_ccfg_node n);
    bit_set *postdominators(int n);
    bit_set *postdominators(p_ccfg_node n);

    void find_natural_loops(); 
    int loop_depth(int n);
    int loop_depth(p_ccfg_node n);
    void set_loop_depth(p_ccfg_node n, int d); 
    void set_loop_depth(int n, int d); 
    
    /* Accessibility methods */
    bool isPath(p_ccfg_node from, p_ccfg_node to);

    /* Returns the header node of the outermost loop enclosing 'node'. */
    p_ccfg_node outermostLoopHeader(p_ccfg_node node);

    bool is_loop_begin(int n);       /* true if block is loop entry */
    bool is_loop_begin(p_ccfg_node cn);
    bool is_loop_end(int n);         /* true if block jumps to loop entry */
    bool is_loop_end(p_ccfg_node cn);
    bool is_loop_exit(int n);        /* true if block is a loop exit */ 
    bool is_loop_exit(p_ccfg_node cn);

    /* list generator routines */
    ccfg_node_list *reverse_postorder_list(bool forward=true);

    /* Parallel information routines */
    bool conc(p_ccfg_node a, p_ccfg_node b);
    bool conc(unsigned a, unsigned b) { return conc(node(a), node(b)); }

    bit_set *par_ancestors(p_ccfg_node n);
    bit_set *par_ancestors(unsigned n) { return par_ancestors(node(n)); }

    /* Data references. Collected on demand. Used mainly by the CSSAME class. */
    void findRefs();
    set_varref& refs() { return _refs; }
    set_varuse& uses() { return _uses; }
    set_vardef& defs() { return _defs; }
    set_phiterm& phiterms() { return _phiterms; }
    set_phiterm& piterms() { return _piterms; }

    /* clean-up routines */
    void remove_unreachable_blocks();

    void print(ostream& f=cout) const;

protected:
    p_tree_proc tp;		/// proc of this ccfg

    vector_ccfg_node _nds;	/// vector of nodes
    p_ccfg_node en;		/// entry node
    p_ccfg_node ex;		/// exit node

    bit_set *doms, *pdoms;	/// dominators and postdominators
    ccfg_node **idom, **ipdom;	/// immediate dominators and postdoms
    bit_set *df, *rdf;		/// dominance frontier and reverse df
    int *lp_depth;		/// loop depth

    set_varref _refs;		/// Set of variable references in the procedure
    set_varuse _uses;		/// Set of variable uses in the procedure
    set_vardef _defs;		/// Set of variable definitions in the procedure
    set_phiterm _phiterms;	/// Set of PHI terms in the procedure
    set_phiterm _piterms;	/// Set of PI terms in the procedure

    bool _graph_built;		/// Set by constructor to flag a complete graph.
    bool _refs_collected;	/// Set by ccfg::findRefs().

    /* internal methods for building the graph */
    void addNode(p_ccfg_node n);
    void addEdges();
    void addEdge(p_ccfg_block block, p_label_sym target);

    GraphBody addTree(p_tree_block block, bool ispar, p_ccfg_node pparent, 
	    p_thread_body tb);

    GraphBody addTree(tree_node_list *body, bool bracket, bool ispar, 
	    p_ccfg_node pparent, p_thread_body tb);

    GraphBody addTree(p_tree_node tn, bool ispar, p_ccfg_node pparent, 
	    p_thread_body tb);

    GraphBody addTree(p_tree_instr ti, bool ispar, p_ccfg_node pparent, 
	    p_thread_body tb);

    GraphBody addTree(p_tree_loop tl, bool ispar, p_ccfg_node pparent, 
	    p_thread_body tb);

    GraphBody addTree(p_tree_for tf, bool ispar, p_ccfg_node pparent, 
	    p_thread_body tb);

    GraphBody addTree(p_tree_if ti, bool ispar, p_ccfg_node pparent, 
	    p_thread_body tb);

    GraphBody addTree(p_tree_cobegin cobegin, p_ccfg_node pparent,
	    p_thread_body tb);

    GraphBody addTree(p_thread_body tb, p_ccfg_node pparent);

    GraphBody addTree(p_tree_parloop parloop, p_ccfg_node pparent, 
	    p_thread_body tb);


    /* helper routine to clone nodes or create new empty nodes */
    void attach(p_ccfg_node );

    /* dfa functions used internally */
    bit_set *_dominators(bool forward);
    ccfg_node **immed_dominators(bool forward);
    void dom_frontiers(p_ccfg_node x, bool forward);

    /* helper routines for clean-up */
    void ccfg_cleanup(p_ccfg_node nd);
};

ostream& operator<<(ostream &f, const ccfg& graph);

#endif /* CCFG_GRAPH_H */
