/**
$Id: mutex-analysis.cc,v 2.26 1999/03/15 21:28:10 diego Exp $
*/
#define _MODULE_ "CSSAME"

#include <list>

#include <suif.h>
#include <odc-util.h>
#include <d_lib.h>

#include "ccfg.h"
#include "cssame.h"

using namespace std;


/**
Traverse the CCFG looking for mutex bodies. This method matches up all the
lock and unlock operations in the program and creates mutex structures for
all the different lock variables in the program. Unmatched lock and unlock
operations are reported as warnings.
*/
void
cssame::findMutexBodies()
{
    D_SELFTEST_HEADER(400, "cssame::findMutexBodies");

    /* For each synchronization variable, create a bitset representing all
     * the nodes that lock the variable and another bitset representing all
     * the nodes that unlock the variable.
     */
    map<p_var_sym, bool> analyzed;
    set_varref::iterator i;
    for (i = _graph->refs().begin(); i != _graph->refs().end(); i++) {
	p_varref ref = *i;
	if (ref->is_clone() || !ref->isMutexSync()) {
	    continue;	/* Only original lock and unlock operations. */
	}

	/* [BUG] Since this procedure finds mutex bodies in different
	 * functions, it should only need to run once for each lock
	 * variable. If a variable already has mutex structures gathered by
	 * a previous procedure, then don't bother again and use what we
	 * already have.
	 */
	p_var_sym var = ref->var();
	if (analyzed.find(var) != analyzed.end()) {
	    continue;	/* Variable has been analyzed already. */
	}
	analyzed[var] = true;

	D_SELFTEST(400) { cout << "Finding mutex bodies for " << var << "\n"; }

	/* 'pn' is an entry partition for the variable (ie, the set of
	 * nodes that lock the variable)
	 */
	set_ccfg_node *pn = this->find_entry_set(var);
	::set_entry_set(var.ptr(), pn);

	/* 'px' is an exit partition for the variable (ie, the set of nodes
	 * that unlock the variable)
	 */
	set_ccfg_node *px = this->find_exit_set(var);
	::set_exit_set(var.ptr(), px);

	/* Add the synchronization variable to the set of synch variables. */
	_mutexVars.insert(var);

	/* Find the mutex structure for the variable. A mutex structure for
	 * a variable is the set of mutex bodies for that variable.
	 */
	mutex_struct *s = this->find_mutex_struct(var, pn, px);
	set_mutex_struct(var.ptr(), s);

	/* Clean up the mutex structure by removing illegal mutex bodies */
	this->cleanup_mutex_struct(var, s);

	D_SELFTEST(400) {
	    cout << "Strict interval for variable " << var << endl;
	    copy(s->begin(),s->end(),ostream_iterator<p_mutex_body>(cout,"\n"));
	}
    }

    D_SELFTEST_FOOTER(400);
}


/**
Build a bitset with the set of nodes that lock the given variable
*/
set_ccfg_node *
cssame::find_entry_set(const p_var_sym& var) const
{
    set_ccfg_node *pn = new set_ccfg_node;

    /* Mark all the nodes that lock the variable */
    set_lockref *locks = ::get_lockrefs(var); assert(locks);
    for (set_lockref::iterator i = locks->begin(); i != locks->end(); i++){
	p_lockref lock = *i;

	if (!lock->is_clone()) {
	    pn->insert(lock->node());
	    D_SELFTEST(401) {
		cout << "Added node " << lock->node() << " to the entry set.\n";
	    }
	}
    }

    return pn;
}



/**
Build a bitset with the set of nodes that lock the given variable
*/
set_ccfg_node *
cssame::find_exit_set(const p_var_sym& var) const
{
    set_ccfg_node *px = new set_ccfg_node;

    /* Mark all the nodes that unlock the variable */
    set_unlockref *unlocks = ::get_unlockrefs(var); assert(unlocks);
    set_unlockref::iterator i;
    for (i = unlocks->begin(); i != unlocks->end(); i++) {
	p_unlockref unlock = *i;

	if (!unlock->is_clone()) {
	    px->insert(unlock->node());
	    D_SELFTEST(401) {
		cout << "Added node " << unlock->node() <<" to the exit set.\n";
	    }
	}
    }

    return px;
}



/**
This method returns a mutex structure for the given lock variable. The
mutex structure is the list of all the mutex bodies for the given variable.
A mutex body is a pair (n, x) which represents the set of nodes strictly
dominated by n and postdominated by x.
*/
mutex_struct *
cssame::find_mutex_struct(p_var_sym var, set_ccfg_node *pn, set_ccfg_node *px)
{
    D_SELFTEST_HEADER(390, "cssame::find_mutex_struct");

    D_SELFTEST(390) {
	cout << "Looking for mutex structures for variable " << var << endl;
    }

    mutex_struct *s = new mutex_struct;

    /* First step. Find all the candidate mutex bodies. 
     * Intersect the dominators
     * and post-dominators for every pair (n, x) with n in pn and x in
     * px. Add to the initial set of candidates every pair (n, x) with a
     * non-empty intersection.
     *
     * foreach n in Pn
     *     foreach x in Px
     *         if n in DOM(x) ^ x in PDOM(n) then
     *             interval->entry = n
     *             interval->exit = x
     *             interval->body = DOM(x) ^ PDOM(n)
     *             S = S U {interval}
     *         endif
     *     endfor
     * endfor
     *
     * Traverse all the nodes in pn and px looking for pairs of nodes
     * (n, x) such that n dominates x and x postdominates n.
     */
    set_ccfg_node::iterator pn_iter;
    for (pn_iter = pn->begin(); pn_iter != pn->end(); pn_iter++) {
	p_ccfg_node e = *pn_iter;
	assert(e->is_block());
	p_ccfg_block entry = e;

	set_ccfg_node::iterator px_iter;
	for (px_iter = px->begin(); px_iter != px->end(); px_iter++) {
	    p_ccfg_node x = *px_iter;
	    assert(x->is_block());
	    p_ccfg_block exit = x;

	    if (e->parent() != x->parent()) {
		continue;	/* Don't build mutex bodies across graphs. */
	    }

	    D_SELFTEST(391) {
		cout << "Checking entry node " << entry << endl
		    << "and exit node " << exit << endl;
	    }

	    /* If the entry node is in the set of dominators for the
	     * exit node and the exit node is in the set of
	     * postdominators for the entry node, we add the pair to
	     * the set of possible intervals.
	     */
	    if (entry->dominates(exit) && exit->postdominates(entry)) {
		p_mutex_body b(new mutex_body(var, entry, exit, s));
		s->insert(b);
		D_SELFTEST(391) {
		    cout << "Added mutex body " << b << endl
			<< "to the mutex structure for " << var << endl << endl;
		}
	    } else {
		D_SELFTEST(391) {
		    cout << "They do not form an interval. Skipping.\n\n";
		}
	    }
	}
    }

    D_SELFTEST(390) {
	cout << "Initial set of strict intervals\n";
	copy(s->begin(), s->end(), ostream_iterator<p_mutex_body>(cout, "\n"));
	cout << endl;
    }

    /* Second step. Filter out unwanted intervals by making sure that each
     * interval (n, x) doesn't include other nodes in pn U px. This avoids
     * the following problem:
     *
     *	(1)	lock x;
     *		...
     *	(2)	unlock x;
     *		...
     *	(3)	lock x;
     *		...
     *	(4)	unlock x;
     *
     * The previous pass would have created the intervals (1, 2), (1, 4)
     * and (3, 4). Obviously the interval (1, 4) is bogus, so we get rid of
     * it.
     *
     * foreach n in pn U px
     *     if n <> entry ^ n <> exit ^ n is in this CCFG then
     *          if entry dom n ^ exit pdom n then
     *              remove interval from S
     *          endif
     *     endif
     * endfor
     *
     * [BUG] This definition is too restrictive. We should be able to deal
     * with irregular mutex bodies or mutex bodies containing parallel
     * structures, etc.
     */
    mutex_struct to_rm;
    for (mutex_struct::iterator i = s->begin(); i != s->end(); i++) {
	p_mutex_body interval = *i;
	p_ccfg_node entry = interval->entry();
	p_ccfg_node exit = interval->exit();

	set_ccfg_node nodes;
	set_union(pn->begin(), pn->end(), px->begin(), px->end(),
		inserter(nodes, nodes.begin()));

	set_ccfg_node::iterator nodes_iter = nodes.begin();
	for (; nodes_iter != nodes.end(); nodes_iter++) {
	    p_ccfg_node n = *nodes_iter;
	    if (n->parent() != entry->parent()) {
		continue;	/* Only care for nodes in this graph */
	    }

	    if (n != entry && n != exit) {
		if (entry->dominates(n) && exit->postdominates(n)) {
		    to_rm.insert(interval);
		}
	    }
	}
    }

    for (mutex_struct::iterator i = to_rm.begin(); i != to_rm.end(); i++) {
	p_mutex_body interval = *i;
	D_SELFTEST(391) { cout << "Removing interval " << interval << endl; }
	s->erase(interval);
    }

    D_SELFTEST_FOOTER(390);
    return s;
}


/**
Checks all the mutex bodies for the given mutex structure. Once the initial
mutex structure has been created, we keep the bodies that are valid
mutex bodies and report errors on the unmatched nodes.
*/
void
cssame::cleanup_mutex_struct(p_var_sym var, mutex_struct *s)
{
    /* Make deep copies of the entry and exit partitions. We will clear
     * all the bits for nodes that participate in valid mutex bodies. The
     * remaining bits will represent nodes that are not in the mutex
     * structure for some reason.
     */
    set_ccfg_node pn(*(::get_entry_set(var)));
    set_ccfg_node px(*(::get_exit_set(var)));

    for (mutex_struct::iterator i = s->begin(); i != s->end(); i++) {
	p_mutex_body b = *i;
	pn.erase(b->entry());
	px.erase(b->exit());
    }


    /* The remaining bits in pn and px represent nodes that do not form a
     * valid mutex body. Traverse all the nodes in pn and px looking for
     * pairs of nodes (n, x) such that there is a path from n to x but n
     * does not dominate x and/or x does not postdominate n.
     */
    set_ccfg_node::iterator pn_iter;
    for (pn_iter = pn.begin(); pn_iter != pn.end(); pn_iter++) {
	p_ccfg_node e = *pn_iter;
	assert(e->is_block());
	p_ccfg_block entry = e;

	set_ccfg_node::iterator px_iter;
	for (px_iter = px.begin(); px_iter != px.end(); px_iter++) {
	    p_ccfg_node x = *px_iter;
	    assert(x->is_block());
	    p_ccfg_block exit = x;

	    if (_graph->isPath(entry, exit)) {
		/* There is a path from entry to exit. If entry dominates the
		 * exit but the exit does not postdominate the entry, the
		 * problem is that the lock may not be released.
		 */
		if (entry->dominates(exit) && !exit->postdominates(entry)) {
		    warning_line(entry->last_instr().ptr(), 
			    ("Lock may not be released"));
		    pn.erase(entry);
		    px.erase(exit);
		}

		/* If entry does not dominate the exit but the exit
		 * postdominates the entry, the problem is that the user may be
		 * releasing an unheld lock.
		 */
		if (!entry->dominates(exit) && exit->postdominates(entry)) {
		    warning_line(exit->last_instr().ptr(), 
			    ("May be releasing an unheld lock"));
		    pn.erase(entry);
		    px.erase(exit);
		}
	    }
	}
    }

    /* Any remaining nodes in pn and px are unmatched synchronization
     * operations
     */
    for (pn_iter = pn.begin(); pn_iter != pn.end(); pn_iter++) {
	p_ccfg_node n = *pn_iter;
	warning_line(n->last_instr().ptr(), ("Unmatched lock operation\n"));
    }

    for (pn_iter = px.begin(); pn_iter != px.end(); pn_iter++) {
	p_ccfg_node x = *pn_iter;
	warning_line(x->last_instr().ptr(), ("Unmatched unlock operation\n"));
    }
}


//***************************************************************************
//		 Member functions for class mutex_body
//***************************************************************************
/**
Constructor for mutex bodies. The mutex body is defined by an entry node
'n' and an exit node 'x'. All the nodes inside that interval are added to
the body.
*/
mutex_body::mutex_body(p_var_sym var, p_ccfg_block n, p_ccfg_block x, 
	mutex_struct *s) : _var(var), _entry(n), _exit(x), _mxstruct(s)
{
    ccfg *graph = n->parent();

    D_SELFTEST_HEADER(440, "mutex_body::mutex_body");

    for (unsigned i = 0; i < graph->num_nodes(); i++) {
	if (n->dominates(i) && x->postdominates(i)) {
	    _body.insert(graph->node(i));
	}
    }
    _body.erase(n);		/* Entry is not included in body */

    /* Set the pointers to the entry and exit instructions */
    _entry_ti = _entry->first_exec();
    _exit_ti  = _exit->last_exec();

    D_SELFTEST(440) {
	cout << "New mutex body built: " << *this << endl;
	cout << "\tat address " << (void *)this << endl << endl;
    }

    D_SELFTEST_FOOTER(440);
}

/**
Copy constructor.
*/
mutex_body::mutex_body(const mutex_body& o) 
    : _var(o._var), _entry(o._entry), _exit(o._exit), _entry_ti(o._entry_ti), 
    _exit_ti(o._exit_ti), _exitRDefs(o._exitRDefs), 
    _upwardExposedUses(o._upwardExposedUses), _mxstruct(o._mxstruct)
{
    D_SELFTEST_HEADER(440, "mutex_body::mutex_body - Copy constructor");

    _body = o._body;

    D_SELFTEST(440) {
	cout << "Original mutex body:  " << o << endl;
	cout << "\tat address " << (void *)&o << endl << endl;
	cout << "New mutex body built: " << *this << endl;
	cout << "\tat address " << (void *)this << endl << endl;
    }

    D_SELFTEST_FOOTER(440);
}


/**
Remove the mutex body from memory.
*/
mutex_body::~mutex_body()
{
    D_SELFTEST_HEADER(450, "mutex_body::~mutex_body");
    D_SELFTEST(450) {
	cout << "Removing mutex_body: " << *this << endl;
	cout << "\tat address " << (void *)this << endl;
    }

    D_SELFTEST_FOOTER(450);
}


/**
Assignment operator
*/
mutex_body &
mutex_body::operator=(const mutex_body& o)
{
    _var = o._var;
    _entry = o._entry;
    _exit = o._exit;
    _body = o._body;
    _entry_ti = o._entry_ti;
    _exit_ti = o._exit_ti;
    _exitRDefs = o._exitRDefs;
    _upwardExposedUses = o._upwardExposedUses;
    _mxstruct = o._mxstruct;

    return *this;
}

/**
Displays the strict interval on the given stream.
*/
void
mutex_body::print_brief(ostream& f) const
{
    f << "(" << source_line_num(_entry->last_instr().ptr()) << "-"
	<< source_line_num(_exit->last_instr().ptr()) << "]";
}

void
mutex_body::print(ostream& f) const
{
    this->print_brief(f);

    f << " = { ";
    set_ccfg_node::iterator iter;
    for (iter = _body.begin(); iter != _body.end(); iter++) { 
	f << (*iter)->number() << " ";
// 	f << (*iter) << "\n";
    }
    f << "}";
}

ostream& operator<<(ostream& f, const mutex_body& interval)
{
    interval.print(f);
    return f;
}

/**
Display the statements in the mutex body on the given stream.
*/
void
mutex_body::printBody(ostream& f) const
{
    tree_node_list_e *elem, *first, *last;
    
    first = _entry_ti->list_e()->next();
    last = _exit_ti->list_e()->next();
    for (elem = first; elem != last; elem = elem->next()) {
	elem->contents->print();
    }
    f << endl;
}


/**
Computes the set of definitions that reach the exit node of the interval. We
compute this set by traversing all the definitions and phi-terms in the body of
the interval.

A defintion will reach the exit node if it postdominates every definition or
phi-term for the same variable inside the body of the interval. We keep all
such definitions in the set _exitRDefs.
*/
void
mutex_body::computeExitRDefs()
{
    D_SELFTEST_HEADER(480, "mutex_body::computeExitRDefs");

    _exitRDefs.erase(_exitRDefs.begin(), _exitRDefs.end());

    /* Gather all definitions in the interval. */
    vector_vardef defs_list;

    set_ccfg_node::iterator body_iter;
    for (body_iter = _body.begin(); body_iter != _body.end(); body_iter++) {
	p_ccfg_node node = *body_iter;

	set_vardef::iterator defs_iter = node->defs().begin();
	for (; defs_iter != node->defs().end(); defs_iter++) {
	    p_vardef def = *defs_iter;
	    if (def->isPHI() || def->isPI()) {
		continue;	/* We only want real definitions */
	    }
	    defs_list.push_back(*defs_iter);
	}
    }

    D_SELFTEST(480) {
	cout << "Definitions to check:\n";
	copy(defs_list.begin(), defs_list.end(), 
		ostream_iterator<p_vardef>(cout, "\n"));
	cout << endl;
    }

    /* Determine which defintions for a given variable can reach the end of
     * the interval. A definition can reach the exit node if and only if it
     * is not postdominated by any other definition in the body.
     */
    vector_vardef::iterator defs_iter1 = defs_list.begin();
    for (; defs_iter1 != defs_list.end(); defs_iter1++) {
	p_vardef def1 = *defs_iter1;
	bool reachesExit = true;

	/* Does any other definition in the body postdominate def1? */
	vector_vardef::iterator defs_iter2 = defs_list.begin();
	for (; defs_iter2 != defs_list.end(); defs_iter2++) {
	    p_vardef def2 = *defs_iter2;
	    if (def2->var() != def1->var() || def1 == def2) {
		continue;
	    }

	    /* If def2 postdominates def1 then def1 cannot reach the end of
	     * the interval.
	     */
	    if (def2->node()->postdominates(def1->node())) {
		reachesExit = false;
		break;
	    }
	}

	/* If the definition reaches the exit node, add it to the list. */
	if (reachesExit) {
	    _exitRDefs.insert(def1);
	}
    }

    D_SELFTEST(480) {
	cout << "List of exit reaching defintions for mutex body\n"
	    << *this << "\n\t";
	copy(_exitRDefs.begin(), _exitRDefs.end(), 
	     ostream_iterator<p_vardef>(cout, "\n\t"));
	cout << endl;
    }

    D_SELFTEST_FOOTER(480);
}


/**
Computes the set of upward exposed uses for the mutex body. This is
the list of variable uses whose control reaching definition comes from a
node outside the mutex body.
*/
void
mutex_body::computeUpwardExposedUses()
{
    D_SELFTEST_HEADER(490, "mutex_body::computeUpwardExposedUses");

    _upwardExposedUses.erase(_upwardExposedUses.begin(), 
	    _upwardExposedUses.end());

    /* Traverse all the uses in the mutex body looking for uses that have
     * at least one control reaching definition outside the mutex body.
     * Notice that we only care for the control reaching definitions when
     * computing upward exposure. We are not interested in concurrent
     * reaching definitions. These are processed when rewriting the PI
     * terms within the mutex body.
     */
    set_ccfg_node::iterator body_iter;
    for (body_iter = _body.begin(); body_iter != _body.end(); body_iter++) {
	p_ccfg_node node = *body_iter;

	set_varuse::iterator use_iter = node->uses().begin();
	for (; use_iter != node->uses().end(); use_iter++) {
	    p_varuse use = *use_iter;

	    D_SELFTEST(491) { 
		cout << "\nAnalyzing use: " << use << endl;
		cout << "\tReaching definitions:" << endl;
		copy(use->reachingDefs().begin(), use->reachingDefs().end(),
		     ostream_iterator<p_vardef>(cout, "\n\t")
		    );
	    }

	    vector_vardef::iterator rdef_iter = use->reachingDefs().begin();
	    for (; rdef_iter != use->reachingDefs().end(); rdef_iter++) {
		p_varref rdef = *rdef_iter;
		
		/* If the reaching definition dominates the mutex body's
		 * entry node, then this use is upward exposed. Add it to
		 * the list of upward exposed uses for the mutex body and
		 * continue with the next reference.
		 */
		if (rdef->node()->dominates(this->entry())) {
		    _upwardExposedUses.insert(use);
		    D_SELFTEST(491) {
			cout << "Use " << use << " is upward exposed\n";
		    }
		    break;
		}
	    }
	}
    }

    D_SELFTEST(490) {
	cout << "List of upward exposed uses for mutex body\n"
	    << *this << "\n\t";
	copy(_upwardExposedUses.begin(), _upwardExposedUses.end(), 
	     ostream_iterator<p_varuse>(cout, "\n\t"));
	cout << endl;
    }

    D_SELFTEST_FOOTER(490);
}


static bool can_remove(p_vardef arg, p_phiterm pi, p_mutex_body mb);

/**
Analyze every PI function to remove superfluous arguments.

Two conditions are checked (See tech report
ftp://ftp.cs.ualberta.ca/pub/TechReports/1998/TR98-11/TR98-11.ps for
details):

\begin{enumerate}
\item	Consecutive kills. For every PI function in the interval, check if
	any of the arguments comes from another mutex body in the same
	structure. If the argument comes from another mutex body and it's
	not in the list of definitions reaching the exit of that mutex body
	then we can safely remove it.

\item	Protected uses. If the control reaching definition of the variable
	referenced by the PI term is inside the mutex body, then all the
	arguments coming from mutex bodies in the same mutex structure can
	be removed.
\end{enumerate}
*/
void
mutex_body::rewritePiFunctions()
{
    D_SELFTEST_HEADER(460, "mutex_body::rewritePiFunctions");

    ccfg *graph = _entry->parent();

    /* Build a list of all the PI terms inside this mutex body. */
    vector_phiterm pi_list;
    set_ccfg_node::iterator body_iter;
    for (body_iter = _body.begin(); body_iter != _body.end(); body_iter++) {
	p_ccfg_node node = *body_iter;

	set_phiterm::iterator pi_iter = node->piterms().begin();
	for (; pi_iter != node->piterms().end(); pi_iter++) {
	    p_phiterm pi = *pi_iter;
	    if (!pi->removed()) {
		pi_list.push_back(pi);
	    }
	}
    }


    D_SELFTEST(460) {
	cout << "Checking PI terms in interval " << *this << endl;
    }

    /* Traverse all the PI terms in the mutex body looking for arguments to
     * remove.
     */
    vector_phiterm::iterator pi_iter = pi_list.begin();
    for (; pi_iter != pi_list.end(); pi_iter++) {
	p_phiterm pi = *pi_iter;

	D_SELFTEST(460) { 
	    cout << "\tChecking PI term "; pi->print(); cout << endl;
	}

	vector_vardef::iterator args_iter = pi->phi_chain().begin();
	for (; args_iter != pi->phi_chain().end(); args_iter++) {
	    /* Skip the first argument of the $\pi$ term because it is the
	     * control reaching definition which we do not care about.
	     */
	    if (args_iter == pi->phi_chain().begin()) {
		continue;
	    }

	    p_vardef arg = *args_iter;
	    if (::can_remove(arg, pi, this)) {
		args_iter = pi->phi_chain().erase(args_iter);
	    }
	}
    } 

    D_SELFTEST_FOOTER(460);
}



/**
Returns true if the given pi-function argument can be removed from the list
of arguments to the pi-function.
*/
bool
can_remove(p_vardef arg, p_phiterm pi, p_mutex_body mb)
{
    D_STACK(can_remove);
    D_SELFTEST_HEADER(470, "can_remove");

    D_SELFTEST(470) {
	cout << "Checking if argument " << arg << " can be removed from\n"
	    << "mutex body " << mb << endl;
    }

    /* Check if the argument comes from a node in another mutex
     * body in this mutex structure. If the argument does not come
     * from a mutex body in the same mutex structure, then we have
     * to ignore it because that reference is not protected by the
     * same lock.
     */
    bool same_mxstruct = false;
    p_mutex_body sibling;
    mutex_struct::iterator i = mb->mxstruct()->begin();
    for (; i != mb->mxstruct()->end(); i++) {
	sibling = *i;
	if (sibling == mb) {
	    continue;
	}

	D_SELFTEST(471) {
	    cout << "\tChecking if argument: " << arg << endl
		<< "\tcomes from mutex body: " << sibling << endl;
	}

	/* If the argument 'arg' is inside a sibling mutex body,
	 * check if it complies with the removal conditions
	 * (successive kills and protected uses)
	 */
	if (sibling->body().find(arg->node()) != sibling->body().end()) {
	    D_SELFTEST(471) { cout << "It does.\n"; }
	    same_mxstruct = true;
	    break;
	} 
    }
    
    if (!same_mxstruct) {
	D_SELFTEST(471) { cout << "It doesn't. Can't remove it.\n"; }
	D_SELFTEST_FOOTER(470);
	return false;
    }


    D_SELFTEST(470) {
	cout << "The argument comes from another mutex body in the same "
	    << "mutex structure\n";
	cout << "Checking condition 1: Protected uses.\n";
    }

    /* If the argument comes from another mutex body in the same mutex
     * structure, we check whether the reference associated with the pi
     * term is protected inside this mutex body.
     *
     * Condition 1: protected uses. If the use affected by this $\pi$ term
     * is not upward exposed (ie, its control reaching definitions can only
     * come from inside this mutex body, then we can discard the argument.
     */
    p_varuse use = pi->immediateUses()[0];
    if (mb->upwardExposedUses().find(use) == mb->upwardExposedUses().end()) {
	D_SELFTEST(470) {
	    cout << "The reference " << use << " is protected by another\n";
	    cout << "definition in the same mutex body. Remove the argument\n";
	    D_SELFTEST_FOOTER(470);
	}

	return true;
    } else {
	D_SELFTEST(470) {
	    cout << "The reference " << use << " is not protected\n";
	}
    }

    D_SELFTEST(470) { cout << "Checking condition 2: successive kills.\n"; }


    /* Condition 2: successive kills. If the argument comes from another
     * mutex body, we check if the argument reaches the exit of that mutex
     * body. If not, we can discard it.
     */
    if (sibling->exitRDefs().find(arg) == sibling->exitRDefs().end()) {
	D_SELFTEST(470) {
	    cout << "The argument does not reach the end of its mutex body. "
		<< "Remove the argument.\n";
	    D_SELFTEST_FOOTER(470);
	}

	return true;
    } else {
	D_SELFTEST(470) {
	    cout << "The argument reaches the end of its mutex body. Can't "
		<< "remove it\n";
	}
    }

    D_SELFTEST_FOOTER(470);
    return false;
}
