/**
$Id: node.cc,v 2.21 1999/03/13 19:12:50 diego Exp $
*/
/*  Control Flow Graph Node Implementation */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    Copyright (c) 1995,1996 The President and Fellows of Harvard University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "CSSAME"

#include <iostream.h>
#include <iomanip.h>

#include <stdlib.h>

#include <suif.h>
#include <par.h>
#include <str.h>
#include <d_lib.h>

#include "ccfg.h"


/* local function prototypes */
static void remove_node(ccfg_node_list *l, p_ccfg_node n);
static void print_node_list(ostream& fp, ccfg_node_list *l);

/* Column widths for printing nodes */
static const unsigned SUCCSW = 12;
static const unsigned PREDSW = 12;
static const unsigned DESCW = 16;
static const unsigned NUMW = 4;


/*---------------------------------------------------------------------------
			  Methods for class ccfg_node
---------------------------------------------------------------------------*/
ccfg_node::ccfg_node(bool parallel, p_ccfg_node par_parent, p_thread_body t)
{
    _parallel = parallel;
    _par_parent = par_parent;
    _thread = t;
    set_number((unsigned)-1);
    set_parent(NULL);
}


ccfg_node::~ccfg_node()
{
    D_SELFTEST_HEADER(150, "ccfg_node::~ccfg_node");
    D_SELFTEST_FOOTER(150);
}


static String
node_list(ccfg_node_list *l)
{
    String list;
    ccfg_node_list_iter cnli(l);

    while (!cnli.is_empty()) {
        list += String(cnli.step()->number());
        if (!cnli.is_empty()) {
            list += " ";
	}
    }

    return list;
}


void
ccfg_node::print_base(ostream& fp) const
{
    fp << "[" << this->parent()->tproc()->proc()->name() << "] ";
    fp << ((this->is_parallel()) ? "par " : "seq ");

    fp << setw(PREDSW) << setiosflags(ios::left) 
	<< String("p ") + node_list(this->preds())
	<< resetiosflags(ios::left);

    fp << setw(SUCCSW) << setiosflags(ios::left) 
	<< String("s ") + node_list(this->succs()) 
	<< resetiosflags(ios::left);
}


void
ccfg_node::print_brief(ostream& fp) const
{
    fp << this->number() << ": " << this->name();
}


void
ccfg_node::add_succ(p_ccfg_node n)
{
    /* avoid duplicate edges */
    ccfg_node_list_iter succ_iter(succs());
    while (!succ_iter.is_empty()) {
	p_ccfg_node succ = succ_iter.step();
	if (succ == n) return;
    }

    /* add the forward and backward edges */
    succs()->append(n.ptr());
    n->preds()->append(this);
}


void
ccfg_node::remove_pred(p_ccfg_node n)
{
    remove_node(preds(), n.ptr());
    remove_node(n->succs(), this);
}


void
ccfg_node::remove_succ (p_ccfg_node n)
{
    remove_node(succs(), n.ptr());
    remove_node(n->preds(), this);
}



/*---------------------------------------------------------------------------
			 Methods for ccfg_marker class
---------------------------------------------------------------------------*/
ccfg_marker::ccfg_marker(p_tree_node t, bool parallel, p_ccfg_node par_parent, 
	p_thread_body tb, p_ccfg_marker companion) : 
	ccfg_node(parallel, par_parent, tb)
{
    this->set_companion(companion);
    tn = t;
}


ccfg_marker::~ccfg_marker()
{
    D_SELFTEST_HEADER(150, "ccfg_marker::~ccfg_marker");
    D_SELFTEST_FOOTER(150);
}



/**
Return the last instruction for this marker node. Note that we are actually
interested in the last instruction in the header of the associated control
structure, not the body. For instance, the last_instr of a while loop is
the expression at the head of the loop.
*/
p_tree_instr
ccfg_marker::last_instr() const
{
    switch (tn->kind()) {
	case TREE_LOOP: return last_ti(p_tree_loop(tn)->test());
	case TREE_FOR:  return last_ti(p_tree_for(tn)->ub_list());
	case TREE_IF:   return last_ti(p_tree_if(tn)->header());
	case TREE_BLOCK: return last_ti(p_tree_block(tn)->body());
	case TREE_INSTR: return last_ti((tn->parent()));
	default: {
	    if (this->is_cobegin() || this->is_coend()) {
		return tn;
	    }

	    assert_msg(false, ("Invalid tree_node kind %d\n", tn->kind()));
	}
    }

    return NULL;
}


/**
Tell this node which marker is its companion. Useful for linking begin
nodes with their corresponding end nodes when bracketing high-level
structures.
*/
void
ccfg_marker::set_companion(p_ccfg_marker other)
{
    _companion = other;
    if (other) { other->_companion = this; }
}


/*----------------------------------------------------------------------------
			  Methods for class ccfg_begin
---------------------------------------------------------------------------*/
ccfg_begin::ccfg_begin(p_tree_node t, bool parallel, p_ccfg_node par_parent,
	p_thread_body tb, p_ccfg_marker companion)
    : ccfg_marker(t, parallel, par_parent, tb, companion)
{
    set_begin_node(t.ptr(), this);
}


ccfg_begin::~ccfg_begin()
{
    D_SELFTEST_HEADER(150, "ccfg_begin::~ccfg_begin");
    D_SELFTEST_FOOTER(150);
}


void
ccfg_begin::print(ostream& fp) const
{
    char *name;

    if (tn->is_loop()) {
	name = "begin loop {";
    } else if (tn->is_for()) {
	name = "begin for {";
    } else if (tn->is_if()) {
	name = "begin if {";
    } else if (tn->is_block()) {
	name = "begin block {";
    } else if (tn->is_proc()) {
	name = "begin proc {";
    } else {
	name = "begin body {";
    }

    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left) 
	<< String(": ") + name << resetiosflags(ios::left);

    this->print_base(fp);
    fp << " (tn: " << tn->number() << ")";
}


/*----------------------------------------------------------------------------
			 Methods for class ccfg_cobegin
---------------------------------------------------------------------------*/
ccfg_cobegin::ccfg_cobegin(p_tree_node t, p_ccfg_node par_parent, 
	p_thread_body tb, p_ccfg_marker companion)
    : ccfg_begin(t, true, par_parent, tb, companion)
{
    set_cobegin_node(t.ptr(), this);
}


ccfg_cobegin::~ccfg_cobegin()
{
    D_SELFTEST_HEADER(150, "ccfg_cobegin::~ccfg_cobegin");
    D_SELFTEST_FOOTER(150);
}

void
ccfg_cobegin::print(ostream& fp) const
{
    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left) 
	<< ": cobegin {" << resetiosflags(ios::left);

    this->print_base(fp);

    if (tn->is_instr()) {
	fp << " (tn: " << p_tree_instr(tn)->instr()->number() << ")";
    } else {
	fp << " (tn: " << tn->number() << ")";
    }
}



/*----------------------------------------------------------------------------
			Methods for class ccfg_end
---------------------------------------------------------------------------*/
ccfg_end::ccfg_end(p_tree_node t, bool parallel, p_ccfg_node par_parent,
	p_thread_body tb, p_ccfg_marker companion)
    : ccfg_marker(t, parallel, par_parent, tb, companion)
{
    set_end_node(t.ptr(), this);
}


ccfg_end::~ccfg_end()
{
    D_SELFTEST_HEADER(150, "ccfg_end::~ccfg_end");
    D_SELFTEST_FOOTER(150);
}


void
ccfg_end::print(ostream& fp) const
{
    char *name;

    if (tn->is_loop()) {
	name = "end loop }";
    } else if (tn->is_for()) {
	name = "end for }";
    } else if (tn->is_if()) {
	name = "end if }";
    } else if (tn->is_block()) {
	name = "end block }";
    } else if (tn->is_proc()) {
	name = "end proc }";
    } else {
	name = "end body }";
    }


    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left) 
	<< String(": ") + name << resetiosflags(ios::left);

    this->print_base(fp);
    fp << " (tn: " << tn->number() << ")";
}


/*----------------------------------------------------------------------------
			  Methods for class ccfg_coend
---------------------------------------------------------------------------*/
ccfg_coend::ccfg_coend(p_tree_node t, p_ccfg_node par_parent, p_thread_body tb, 
    p_ccfg_marker companion) : ccfg_end(t, true, par_parent, tb, companion)
{
    set_coend_node(t.ptr(), this);
}


ccfg_coend::~ccfg_coend()
{
    D_SELFTEST_HEADER(150, "ccfg_coend::~ccfg_coend");
    D_SELFTEST_FOOTER(150);
}


void
ccfg_coend::print(ostream& fp) const
{
    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left) 
	<< ": coend }" << resetiosflags(ios::left);

    this->print_base(fp);

    if (tn->is_instr()) {
	fp << " (tn: " << p_tree_instr(tn)->instr()->number() << ")";
    } else {
	fp << " (tn: " << tn->number() << ")";
    }
}





/*----------------------------------------------------------------------------
		       Methods for class ccfg_label
---------------------------------------------------------------------------*/
ccfg_label::ccfg_label(p_tree_node t, p_label_sym label, bool parallel, 
	p_ccfg_node par_parent, p_thread_body tb)
    : ccfg_marker(t, parallel, par_parent, tb, 0)
{
    set_node(t.ptr(), this);

    if (t->is_instr() && !::is_cobegin(t)) {
	/* Annotate the label symbol with the flow graph node so that
	 * blocks that jump to this label can find the node (see
	 * ccfg::addEdges).
	 */
	p_tree_instr ti = t;
	if_ops op = ti->instr()->opcode();
	assert(op == io_lab);

	_label = ((in_lab *)ti->instr())->label();
	set_node(_label.ptr(), this);
    } else {
	/* Label nodes for control structures. The caller must indicate
	 * which label symbol should be annotated with this label node.
	 */
	assert(label);
	_label = label;
	set_node(_label.ptr(), this);
    }
}



ccfg_label::~ccfg_label()
{
    D_SELFTEST_HEADER(150, "ccfg_label::~ccfg_label");
    D_SELFTEST_FOOTER(150);
}


void
ccfg_label::print(ostream& fp) const
{
    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left)
	<< String(": label ") + _label->name() << resetiosflags(ios::left);

    this->print_base(fp);

    if (tn->is_instr()) {
	fp << " (tn: " << p_tree_instr(tn)->instr()->number() << ")";
    } else {
	fp << " (tn: " << tn->number() << ")";
    }
}


/*---------------------------------------------------------------------------
			     Methods for ccfg_test
---------------------------------------------------------------------------*/
ccfg_test::ccfg_test(p_tree_for t, bool parallel, p_ccfg_node par_parent,
    p_thread_body tb) : ccfg_node(parallel, par_parent, tb)
{
    tf = t;
    set_test_node(tf.ptr(), this);
}

ccfg_test::~ccfg_test()
{
    D_SELFTEST_HEADER(150, "ccfg_test::~ccfg_test");
    D_SELFTEST_FOOTER(150);
}


/**
Return the last instruction of the test node
*/
p_tree_instr
ccfg_test::last_instr() const
{ 
	return last_ti(tf->ub_list());
}


void
ccfg_test::print(ostream& fp) const
{
    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left)
	<< ": test" << resetiosflags(ios::left);

    this->print_base(fp);
    fp << " (tf: " << tf->number() << ")";
}



/*---------------------------------------------------------------------------
		       Methods for class ccfg_instr
---------------------------------------------------------------------------*/
ccfg_instr::~ccfg_instr()
{
    D_SELFTEST_HEADER(150, "ccfg_instr::~ccfg_instr");
    D_SELFTEST_FOOTER(150);
}

ccfg_instr::ccfg_instr(p_tree_instr t, bool parallel, p_ccfg_node par_parent, 
	p_thread_body tb) : ccfg_node(parallel, par_parent, tb)
{
    ti = t;
    set_node(ti.ptr(), this);
}

void
ccfg_instr::print(ostream& fp) const
{
    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left)
	<< ": instr" << resetiosflags(ios::left);

    this->print_base(fp);

    fp << " (ti: " << ti->number() << ")";
}


/*---------------------------------------------------------------------------
		       Methods for class ccfg_block
---------------------------------------------------------------------------*/
/**
Build a new block node with the given list of instructions. The list is
delimited by t1 and t2. It assumes that all the tree nodes between t1 and
t2 are tree_instrs (otherwise this wouldn't be a valid ccfg_block).
*/
ccfg_block::ccfg_block(tree_node_list *tn, p_tree_instr t1, p_tree_instr t2, 
	bool parallel, p_ccfg_node par_parent, p_thread_body tb)
    : ccfg_node(parallel, par_parent, tb)
{
    tnl = tn; 
    ti1 = t1;
    ti2 = t2;

    /* Inform all the instructions between t1 and t2 that this is the node
     * they're attached to.
     */
    ccfg_node_instr_iter i(this);
    while (!i.is_empty()) { set_node(i.step(), this); }
}


/**
Remove the basic block from memory.
*/
ccfg_block::~ccfg_block()
{
    D_SELFTEST_HEADER(150, "ccfg_block::~ccfg_block");
    D_SELFTEST_FOOTER(150);
}



/**
Add the given instruction to this basic block.
*/
void ccfg_block::extend(p_tree_instr t)
{
    ti2 = t;
    set_node(ti2.ptr(), this);
}


/** 
Returns a pointer to the first tnl element that is not a label. If the
block contains just label instructions, it returns NULL.
*/
tnle *
ccfg_block::first_non_label (void) const
{
    tree_node_list_e *tnle = in_head()->list_e(); 
    
    while (tnle && ::is_label(tnle->contents) && tnle != in_tail()->list_e()) {
	tnle = tnle->next(); 
    }

    return tnle;
}


/**
Returns pointer to the last tnl element that is not a control instruction.
Routine returns NULL if the block is just a single control instruction. 
*/
tnle *
ccfg_block::last_non_cti (void) const
{
    if (is_cti(in_tail())) {
	if (in_tail() == in_head())
	    return NULL; // block is one cti instr
	else 
	    return in_tail()->list_e()->prev(); 
    } else {
	return in_tail()->list_e(); 
    }
}

/**
Returns the last statement of the block that is not a label, mrk or nop
instruction.
*/
p_tree_instr 
ccfg_block::last_exec() const
{
    p_tree_instr ti2_exec = in_tail();
    if_ops op = ti2_exec->instr()->opcode();
    while ( (ti2_exec != in_head()) && 
	    (op == io_nop || op == io_mrk || op == io_lab) ) {
	ti2_exec = (tree_instr *)ti2_exec->list_e()->prev()->contents;
	op = ti2_exec->instr()->opcode();
    }

    /* If we reached the head of the block and we didn't find an executable
     * statement, return the last instruction.
     */
    if (ti2_exec == in_head() &&
	(op == io_nop || op == io_mrk || op == io_lab) ) {
	return in_tail();
    } else {
	return ti2_exec;
    }
}


/**
Returns the first statement of the block that is not a label, mrk or nop
instruction.
*/
p_tree_instr 
ccfg_block::first_exec() const
{
    p_tree_instr ti1_exec = in_head();
    if_ops op = ti1_exec->instr()->opcode();
    while ( (ti1_exec != in_tail()) && 
	    (op == io_nop || op == io_mrk || op == io_lab) ) {
	ti1_exec = (tree_instr *)ti1_exec->list_e()->next()->contents;
	op = ti1_exec->instr()->opcode();
    }

    /* If we reached the end of the block and we didn't find an executable
     * statement, return the first instruction.
     */
    if (ti1_exec == in_tail() &&
	(op == io_nop || op == io_mrk || op == io_lab) ) {
	return in_head();
    } else {
	return ti1_exec;
    }
}


void
ccfg_block::print(ostream& fp) const
{
    String terms;
    String label;

    if (is_ubr(ti2) || is_cbr(ti2)) {
	label = ((in_lab *)ti2->instr())->label()->name();
    }

    if (is_ubr(ti2))
	terms = "ubr ";
    else if (is_cbr(ti2))
	terms = "cbr ";
    else if (is_mbr(ti2))
	terms = "mbr ";
    else if (is_call(ti2))
	terms = "call";
    else if (is_ret(ti2))
	terms = "ret";
    else if (is_cti(ti2))
	terms = "cti";

    fp << setw(NUMW) << number() << setw(DESCW) << setiosflags(ios::left)
	<< String(": block ") + terms + label
        << resetiosflags(ios::left);

    this->print_base(fp);

    fp << " (tn: { ";
    p_tree_node tn = ti1;
    while (tn) {
	if (!is_nop(tn)) {
	    if (tn->is_instr()) {
		fp << p_tree_instr(tn)->instr()->number() << " ";
	    } else {
		fp << tn->number() << " ";
	    }
	}

	if (tn->list_e()->next() == NULL || tn == ti2) {
	    tn = NULL;
	} else {
	    tn = tn->list_e()->next()->contents;
	}
    }
    fp << "})";
}


/*----------------------------------------------------------------------------
			 Private helper functions
---------------------------------------------------------------------------*/
/**
Remove ccfg_node n from the ccfg_node_list l
*/
static void
remove_node (ccfg_node_list *l, p_ccfg_node n)
{
    ccfg_node_list_iter node_iter(l);
    while (!node_iter.is_empty()) {
	p_ccfg_node node = node_iter.step();
	if (node == n) {
	    delete l->remove(node_iter.cur_elem());
	    return;
	}
    }
    assert_msg(false, ("remove_node - node '%u' not found", n->number()));
}
