/**
$Id: node.h,v 2.16 1999/02/21 20:39:15 diego Exp $
*/
/*  Control Flow Graph Nodes */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    Copyright (c) 1995, 1996, 1997 The President and Fellows of Harvard 
    University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file.
*/
#ifndef CCFG_NODE_H
#define CCFG_NODE_H

#include <suif_copyright.h>
#include <suif.h>

#include <set>
#include <list>
#include <iostream.h>

#include <d_lib.h>

using namespace std;

/*
 *  Control flow graph nodes are divided into several subclasses
 *  depending on the sort of SUIF object(s) to which they correspond.
 */
enum ccfg_node_kind {
    CCFG_MARKER,	/* generic marker (base class) */
    CCFG_COBEGIN,	/* beginning marker for cobegin/coend and parloops */
    CCFG_COEND,		/* ending marker for cobegin/coend and parloops */
    CCFG_BEGIN,		/* beginning marker for AST node */
    CCFG_END,		/* ending marker for AST node */
    CCFG_INSTR,		/* instruction or expression tree */
    CCFG_BLOCK,		/* basic block of instructions */
    CCFG_TEST,		/* "test" node of a FOR loop */
    CCFG_LABEL		/* implicit AST node label */
};

class ccfg_node;
struct GraphBody;
class ccfg_begin;
class ccfg_end;
class phiterm;


/**
Each flow graph node contains only an ID number and information related to
the flow graph.  Other information can be associated with a node by storing
it in a separate array in the ccfg structure where the node ID number is
used to index into the array.

This is a virtual class that cannot be instantiated.
*/
class ccfg_node { D_CLASS(ccfg_node)
    friend class ccfg;
    friend class cssame;
    friend class GraphBody;

public:
    ccfg_node(bool parallel, p_ccfg_node par_parent, p_thread_body t);
    virtual ~ccfg_node();

    virtual ccfg_node_kind kind() const = 0;
    virtual const char *name() const = 0;

    bool is_begin() const		{ return (kind() == CCFG_BEGIN); }
    bool is_end() const			{ return (kind() == CCFG_END); }
    bool is_instr() const		{ return (kind() == CCFG_INSTR); }
    bool is_block() const		{ return (kind() == CCFG_BLOCK); }
    bool is_test() const		{ return (kind() == CCFG_TEST); }
    bool is_label() const		{ return (kind() == CCFG_LABEL); }
    bool is_cobegin() const		{ return (kind() == CCFG_COBEGIN); }
    bool is_coend() const		{ return (kind() == CCFG_COEND); }
    bool is_marker() const		{ return (kind() == CCFG_MARKER); }
    bool is_parallel() const		{ return _parallel; }

    unsigned number() const             { return nnum; }
    ccfg *parent() const                { return par; }

    ccfg_node_list *preds() const	{ return (ccfg_node_list *)&prs; }
    ccfg_node_list *succs() const	{ return (ccfg_node_list *)&scs; }

    p_ccfg_node par_parent() const	{ return _par_parent; }
    bit_set *par_ancestors() const	{ return par->par_ancestors(nnum); }
    p_thread_body thread() const	{ return _thread; }

    /* Data flow information */
    virtual void findRefs()		{ }
    set_varref& refs()			{ return _refs; }
    set_vardef& defs()			{ return _defs; }
    set_varuse& uses()			{ return _uses; }
    set_phiterm& phiterms()		{ return _phiterms; }
    set_phiterm& piterms()		{ return _piterms; }

    /* Conflict list. This list includes all the conflicts that have this
     * node as the conflict head. This is done to simplify the process of
     * adding PI functions to the graph.
     */
    list_conflict& conflicts()	{ return _conflicts; }


    /* Access dominator information. These are just convenient wrappers
     * around ccfg methods.
     */
    bool dominates(int n_dominatee) { 
	return par->dominates(this->number(), n_dominatee);
    }

    bool dominates(p_ccfg_node dominatee) { 
	return par->dominates(this, dominatee);
    }

    bool postdominates(int n_dominatee) {
	return par->postdominates(this->number(), n_dominatee);
    }

    bool postdominates(p_ccfg_node dominatee) {
	return par->postdominates(this, dominatee);
    }

    p_ccfg_node immed_dom() {
	return par->immed_dom(this);
    }

    p_ccfg_node immed_postdom() {
	return par->immed_postdom(this);
    }

    bit_set *dom_frontier() {
	return par->dom_frontier(this);
    }

    bit_set *reverse_dom_frontier() {
	return par->reverse_dom_frontier(this);
    }

    bit_set *dominators() {
	return par->dominators(this);
    }

    bit_set *postdominators() {
	return par->postdominators(this);
    }

    /* Methods to access instructions contained in the node */
    virtual p_tree_instr last_instr() const = 0;

    /* SSA helper methods */
    p_phiterm hasPi(p_var_sym var);

    ccfg_node_list *cpreds()		{ return &_cpreds; }
    ccfg_node_list *csuccs()		{ return &_csuccs; }

    ccfg_node_list *spreds()		{ return &_spreds; }
    ccfg_node_list *ssuccs()		{ return &_ssuccs; }

    bit_set& prec()		 	{ return _prec; }
    bit_set& prec_ct()		 	{ return _prec_ct; }
    bit_set& prec_sy()		 	{ return _prec_sy; }

    virtual void print(ostream& fp=cout) const = 0;
    virtual void print_brief(ostream& fp=cout) const;


private:
    unsigned nnum;                      /* node number */
    ccfg *par;                          /* parent flow graph */
    ccfg_node_list prs;                 /* ccfg predecessors */
    ccfg_node_list scs;                 /* ccfg successors */
    ccfg_node_list _cpreds;		/* Conflict predecessors */
    ccfg_node_list _csuccs;		/* Conflict successors */
    ccfg_node_list _spreds;		/* Synchronization predecessors */
    ccfg_node_list _ssuccs;		/* Synchronization successors */
    bit_set _prec;			/* Nodes guaranteed to precede me */
    bit_set _prec_ct;			/* Nodes guaranteed to precede me
					   through control flow */
    bit_set _prec_sy;			/* Nodes guaranteed to precede me
					   through synchronization flow */

protected:
    bool _parallel;			/* Is this a parallel node? */
    p_ccfg_node _par_parent;		/* Parallel parent node (ie,
					   enclosing cobegin node) */
    p_thread_body _thread;		/* Thread body for this node. NULL
					   if the node is not inside a
					   thread. */
    set_varref _refs;			/* Set of variable references in
					   this node. */
    set_vardef _defs;			/* Set of variable definitions in
					   this node. */
    set_varuse _uses;			/* Set of variable uses in this
					   node. */
    set_phiterm _phiterms;		/* Set of PHI terms in this node. */
    set_phiterm _piterms;		/* Set of PI terms in this node. */
    list_conflict _conflicts;		/* List of conflicts that have this
					   node as its head. */

    void set_number(unsigned n)         { nnum = n; }
    void set_parent(ccfg *p)             { par = p; }

    void add_succ(p_ccfg_node n);
    void remove_pred(p_ccfg_node n);
    void remove_succ(p_ccfg_node n);
    void insert_node_after(p_ccfg_node n);

    void print_base(ostream& fp) const;
};


/**
The BEGIN and END flow graph nodes are used to indicate the beginning and
ending of tree_nodes (except for tree_instrs). Concurrent section markers
(cobegins and parloops) are represented by classes ccfg_cobegin and
ccfg_coend respectively.
*/
class ccfg_marker : public ccfg_node { D_CLASS(ccfg_marker)
    friend class ccfg;
    friend class cssame;

public:
    ccfg_marker(p_tree_node t, bool parallel, p_ccfg_node par_parent, 
	    p_thread_body tb, p_ccfg_marker companion = 0);
    virtual ~ccfg_marker();

    p_tree_node node() const { return tn; }
    ccfg_node_kind kind() const { return CCFG_MARKER; }

    /* Set my companion marker. Useful for linking begin nodes with their
     * corresponding end nodes.
     */
    void set_companion(p_ccfg_marker other);
    p_ccfg_marker companion() const { return _companion; }

    virtual p_tree_instr last_instr() const;
    virtual const char *name() const { return "marker"; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

protected:
    p_tree_node tn;
    p_ccfg_marker _companion;
};


class ccfg_begin : public ccfg_marker { D_CLASS(ccfg_begin)
public:
    ccfg_begin(p_tree_node t, bool parallel, p_ccfg_node par_parent,
	    p_thread_body tb, p_ccfg_marker companion = 0);
    virtual ~ccfg_begin();

    ccfg_node_kind kind() const { return CCFG_BEGIN; }

    virtual void print(ostream& fp=cout) const;
    virtual const char *name() const { return "begin"; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

};


class ccfg_cobegin : public ccfg_begin { D_CLASS(ccfg_coend)
public:
    ccfg_cobegin(p_tree_node t, p_ccfg_node par_parent, p_thread_body tb, 
	    p_ccfg_marker companion = 0);
    virtual ~ccfg_cobegin();

    ccfg_node_kind kind() const { return CCFG_COBEGIN; }

    virtual void print(ostream& fp=cout) const;
    virtual const char *name() const { return "cobegin"; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

};




class ccfg_end : public ccfg_marker { D_CLASS(ccfg_end)
  public:
    ccfg_end(p_tree_node t, bool parallel, p_ccfg_node par_parent,
	    p_thread_body tb, p_ccfg_marker companion = 0);
    ~ccfg_end();

    ccfg_node_kind kind() const { return CCFG_END; }

    virtual void print(ostream& fp=cout) const;
    virtual const char *name() const { return "end"; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

};


class ccfg_coend : public ccfg_end { D_CLASS(ccfg_coend)
public:
    ccfg_coend(p_tree_node t, p_ccfg_node par_parent, p_thread_body tb, 
	    p_ccfg_marker companion = 0);
    ~ccfg_coend();

    ccfg_node_kind kind() const	{ return CCFG_COEND; }

    virtual void print(ostream& fp=cout) const;
    virtual const char *name() const { return "coend"; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

};



/**
LABEL nodes are used for the implicit labels associated with high-level AST
nodes.  The result is that only nodes associated with labels will have
multiple predecessors -- in other words, any node with multiple
predecessors will be empty.  That is a nice property for solving data flow
problems.
*/
class ccfg_label : public ccfg_marker { D_CLASS(ccfg_label)
public:
    ccfg_label(p_tree_node t, p_label_sym label, bool parallel, 
	    p_ccfg_node par_parent, p_thread_body tb);
    ~ccfg_label();

    ccfg_node_kind kind() const { return CCFG_LABEL; }
    virtual p_tree_instr last_instr() const { return p_tree_instr(tn); }
    void print(ostream& fp=cout) const;
    virtual const char *name() const { return "label"; }
    p_label_sym label() const { return _label; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }


private:
    p_label_sym _label;
};



/**
The CCFG_INSTR flow graph nodes correspond to individual SUIF instructions
(not basic blocks).
*/
class ccfg_instr : public ccfg_node { D_CLASS(ccfg_instr)
    friend class ccfg;
    friend class cssame;

public:
    ccfg_instr(p_tree_instr t, bool parallel, p_ccfg_node
	    par_parent, p_thread_body tb);
    ~ccfg_instr();

    ccfg_node_kind kind() const	{ return CCFG_INSTR; }
    p_tree_instr instr()		{ return ti; }

    /* Data flow information */
    virtual void findRefs();

    virtual p_tree_instr last_instr() const { return ti; }

    void print(ostream& fp=cout) const;
    virtual const char *name() const { return "instr"; }

    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }


private:
    p_tree_instr ti;
};



typedef tree_node_list_e tnle;  /* just shorthand for this declaration */

/**
The BLOCK flow graph nodes represent basic blocks.  Their boundaries are
identified by the starting and ending tree_instr nodes.  Only other
tree_instrs should be in the block -- no high-level tree_nodes.
*/
class ccfg_block : public ccfg_node { D_CLASS(ccfg_block)
    friend class ccfg;
    friend class cssame;

public:
    ccfg_block(tree_node_list *tn, p_tree_instr t1, p_tree_instr t2, bool
	    parallel, p_ccfg_node par_parent, p_thread_body tb);

    ~ccfg_block();

    ccfg_node_kind kind() const                { return CCFG_BLOCK; }

    p_tree_instr in_head() const         { return ti1; }
    p_tree_instr in_tail() const         { return ti2; }
    void set_in_head(p_tree_instr ti)    { ti1 = ti; } 
    void set_in_tail(p_tree_instr ti)    { ti2 = ti; } 

    tnle *first_non_label() const;
    tnle *last_non_cti() const;
    p_tree_instr last_exec() const;
    p_tree_instr first_exec() const;

    virtual p_tree_instr last_instr() const { return last_exec(); }

    /* Data flow information */
    virtual void findRefs();

    void print(ostream& fp=cout) const;
    virtual const char *name() const { return "block"; }


    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

private:
    tree_node_list *tnl; 
    p_tree_instr ti1;
    p_tree_instr ti2;

protected:
    void extend(p_tree_instr t);
};



/**
The test part of a tree_for node is generated automatically when the
tree_for is expanded, yet it must still be represented in flow graphs. The
TEST flow graph nodes are used for this special purpose.
*/
class ccfg_test : public ccfg_node { D_CLASS(ccfg_test)
    friend class ccfg;
    friend class cssame;

public:
    ccfg_test(p_tree_for t, bool parallel, p_ccfg_node par_parent,
	    p_thread_body tb);
    ~ccfg_test();

    virtual const char *name() const { return "test"; }
    ccfg_node_kind kind() const { return CCFG_TEST; }
    p_tree_for for_loop() const { return tf; }
    virtual p_tree_instr last_instr() const;

    void print(ostream& fp=cout) const;

    /* Data flow information */
    virtual void findRefs();


    /** Conversion operators for pointer wrappers */
    inline operator p_ccfg_node() const {
	return p_ccfg_node((ccfg_node *)this);
    }

private:
    p_tree_for tf;
};

/* Stream insertion operands */
inline ostream& operator<<(ostream& s, const ccfg_begin& n) { 
    n.print(s); return s;
}

inline ostream& operator<<(ostream& s, const ccfg_end& n) {
    n.print(s); return s;
}

inline ostream& operator<<(ostream& s, const ccfg_label& n) {
    n.print(s); return s;
}

inline ostream& operator<<(ostream& s, const ccfg_instr& n) {
    n.print(s); return s;
}

inline ostream& operator<<(ostream& s, const ccfg_block& n) {
    n.print(s); return s;
}

inline ostream& operator<<(ostream& s, const ccfg_test& n) {
    n.print(s); return s;
}

#endif /* CCFG_NODE_H */
