/**
$Id: util.cc,v 2.18 1999/03/13 19:12:50 diego Exp $
*/
/*  Control Flow Graph Utility Implementations */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    Copyright (c) 1995 The President and Fellows of Harvard University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "CSSAME"

#include <suif.h>
#include <par.h>
#include <odc-util.h>

#include "ccfg.h"

static void *get_annote_for(suif_object *obj, char *an);
static void set_annote_for(suif_object *obj, char *an, void *value);


/**
Retrieve the node corresponding to the position of a label in a control
flow graph.  This function will return NULL if the flow graph has not yet
been constructed.
*/
p_ccfg_label
label_ccfg_node (p_label_sym l)
{
    return (ccfg_label *)get_annote_for(l.ptr(), k_ccfg_node);
}


/**
Retrieve the nodes corresponding to the beginning or end of an AST node in
a control flow graph.  For instruction nodes, the result is the flow graph
node that holds the instruction; for other nodes, it is the begin/end
marker node.  These functions will return NULL if the flow graph has not
yet been constructed.
*/
p_ccfg_begin
ccfg_begin_node(p_tree_node n)
{
    char *an_name;

    if (is_cobegin(n) || is_parloop(n)) {
	an_name = k_ccfg_cobegin;
    } else if (n->is_instr()) {
	an_name = k_ccfg_node;
    } else {
	an_name = k_ccfg_begin;
    }

    return (ccfg_begin *)get_annote_for(n.ptr(), an_name);
}


p_ccfg_end
ccfg_end_node(p_tree_node n)
{
    char *an_name;

    if (is_cobegin(n) || is_parloop(n)) {
	an_name = k_ccfg_coend;
    } else if (n->is_instr()) {
	an_name = k_ccfg_node;
    } else {
	an_name = k_ccfg_end;
    }

    return (ccfg_end *)get_annote_for(n.ptr(), an_name);
}


/**
Retrieve the "test" and "top" nodes for a FOR loop in a control flow graph.
The separate node for the invisible "top" label is only used when the graph
is built with nodes for individual instructions instead of basic blocks.
These functions will return NULL if the flow graph has not yet been
constructed.
*/
p_ccfg_test
ccfg_test_node(p_tree_for n)
{
    return (ccfg_test *)get_annote_for(n.ptr(), k_ccfg_test);
}


p_ccfg_label
ccfg_toplab(p_tree_for n)
{
    return (ccfg_label *)get_annote_for(n.ptr(), k_ccfg_toplab);
}



/**
Generate vcg format file for viewing with xvcg
*/
void
generate_vcg(FILE *f, ccfg *ccfg)
{
    /* Use minbackward layout algorithm in an attempt to make all backedges  */
    /* true backedges. Doesn't always work, but the back algorithm for this. */
    /* Change these parameters as needed, but they have worked fine so far.  */
    fprintf(f,"graph: { title: \"CCFG_GRAPH\"\n");
    fprintf(f,"\n");    
    fprintf(f,"x: 30\n");
    fprintf(f,"y: 30\n");
    fprintf(f,"height: 800\n");
    fprintf(f,"width: 500\n");
    fprintf(f,"stretch: 60\n");
    fprintf(f,"shrink: 100\n");
    fprintf(f,"layoutalgorithm: minbackward\n");
    fprintf(f,"node.borderwidth: 3\n");
    fprintf(f,"node.color: white\n");
    fprintf(f,"node.textcolor: black\n");
    fprintf(f,"node.bordercolor: black\n");
    fprintf(f,"edge.color: black\n");
    fprintf(f,"\n");

    /* Put down nodes */
    int num = ccfg->num_nodes();	
    int i; 
    for (i=0; i<num; i++)
	fprintf(f,"node: { title:\"%d\" label:\"%d\" }\n", \
		ccfg->node(i)->number(),ccfg->node(i)->number());
    fprintf(f,"\n");

    /* Put down edges and backedges */
    for (i=0; i<num; i++) {
	ccfg_node_list_iter ccfgnli(ccfg->node(i)->preds());
	ccfg_node *itnd;
	while (!ccfgnli.is_empty()) {
	    itnd = ccfgnli.step();
	    if (itnd->number()<ccfg->node(i)->number())
		fprintf(f,"edge: {sourcename:\"%d\" targetname:\"%d\" }\n",\
			itnd->number(), ccfg->node(i)->number());
	    else
		fprintf(f,"backedge: {sourcename:\"%d\" targetname:\"%d\" }\n"\
			,itnd->number(), ccfg->node(i)->number());
    }
    }

    /* Wrap up at the end */
    fprintf(f,"\n");
    fprintf(f, "}\n");
}


/*
 * ccfg_node_instr_iter - iterator over ccfg_block instructions
 */

/* Constructor */

ccfg_node_instr_iter::ccfg_node_instr_iter(ccfg_node *nod, bool reverse,
	bool exec)
{
    if (nod->is_block()) {
	ccfg_block *blk = (ccfg_block *)nod; 
	/* If the 'exec' flag is set to true, we will skip any nop
	 * instructions at the beginning or at the end of the node
	 */
	if (exec) {
	    first = blk->first_exec()->list_e();
	    last = blk->last_exec()->list_e();
	} else {
	    first = blk->in_head()->list_e();
	    last = blk->in_tail()->list_e();
	}
    } else if (nod->is_instr()) {
	ccfg_instr *ins = (ccfg_instr *)nod; 
	first = ins->instr()->list_e(); 
	last = ins->instr()->list_e(); 
    } else {
	assert_msg(false, ("ccfg_node_instr_iter -- Unimplemented node type")); 
    }
    sentinel = reverse ? first : last; 
    rev = reverse; 
    reset(); 
}

/* Reset the iter to the beginning of the basic block */

void
ccfg_node_instr_iter::reset()
{
    cur = NULL;
    nxt = rev ? last : first;
}

/* Step the iter to the next instruction */

tree_instr *
ccfg_node_instr_iter::step()
{
    /* Return NULL if at end of basic block */
    if (cur == sentinel)
	return NULL;

    cur = nxt;
    nxt = rev ? nxt->prev() : nxt->next();
    return (tree_instr *)cur->contents;
}

/* Look at the next instruction, but don't step the iter */

tree_instr *
ccfg_node_instr_iter::peek()
{
    /* Return NULL if at end of basic block */
    if (cur == sentinel)
	return NULL;

    return (tree_instr *)nxt->contents;
}

/* Am I at the end of the instructions in this basic block? */

bool
ccfg_node_instr_iter::is_empty()
{
    return (cur == sentinel);
}

/**
 ** ------------- Helper functions ------------- 
 **/

/* occurs() -- returns true if cn is an element of cnl */ 
bool
occurs (ccfg_node *cn, ccfg_node_list *cnl)
{
    ccfg_node_list_iter pred_iter(cnl); 
    while (!pred_iter.is_empty()) 
	if (pred_iter.step() == cn) 
	    return true; 
    return false; 
}

/**
Returns the first tree instruction node from the given tree_node list.
*/
p_tree_instr 
first_ti(tree_node_list *tnl)
{
    D_STACK(first_ti);
    D_SELFTEST_HEADER(500, "first_ti");

    D_SELFTEST(500) {
	cout << "Looking for the first instruction in:\n";
	tree_node_list_iter iter(tnl);
	while (!iter.is_empty()) { iter.step()->print(); }
	fflush(stdout);
	cout << endl << endl;
    }

    tree_node_list_iter iter(tnl);
    while (iter.is_empty() == false) {
	tree_node *node = iter.step();
	if (node->is_instr()) {
	    D_SELFTEST(500) {
		cout << "Found it in tree_node:\n";
		node->print();
		fflush(stdout);
	    }
	    D_SELFTEST_FOOTER(500);
	    return (tree_instr *)node;
	} else if (node->is_block()) {
	    return first_ti( ((tree_block *)node)->body());
	} else if (node->is_loop()) {
	    return first_ti( ((tree_loop *)node)->body());
	} else if (node->is_for()) {
	    return first_ti( ((tree_for *)node)->body());
	} else if (node->is_if()) {
	    tree_if *ti = (tree_if *)node;
	    if (!ti->then_part()->is_empty()) {
		return first_ti(ti->then_part());
	    } else {
		return first_ti(ti->else_part());
	    }
	}
    }

    D_SELFTEST(500) { cout << "Did not find any tree_instr\n"; }
    D_SELFTEST_FOOTER(500);

    return NULL;
}


/**
Returns the last tree instruction node from the given tree_node list.
*/
p_tree_instr 
last_ti(tree_node_list *tnl)
{
    D_STACK(last_ti);
    D_SELFTEST_HEADER(500, "last_ti");

    D_SELFTEST(500) {
	cout << "Looking for the last instruction in:\n";
	tree_node_list_iter iter(tnl);
	while (!iter.is_empty()) { iter.step()->print(); }
	fflush(stdout);
	cout << endl << endl;
    }

    int j;
    for (j = tnl->count() - 1; j >= 0; j--) {
	tree_node *node = (*tnl)[j];
	if (node->is_instr()) {
	    D_SELFTEST(500) {
		cout << "Found it in tree_node:\n";
		node->print();
		fflush(stdout);
	    }
	    D_SELFTEST_FOOTER(500);

	    return (tree_instr *)node;
	} else if (node->is_block()) {
	    return last_ti( ((tree_block *)node)->body());
	} else if (node->is_loop()) {
	    return last_ti( ((tree_loop *)node)->body());
	} else if (node->is_for()) {
	    return last_ti( ((tree_for *)node)->body());
	} else if (node->is_if()) {
	    tree_if *ti = (tree_if *)node;
	    if (!ti->else_part()->is_empty()) {
		return last_ti(ti->else_part());
	    } else {
		return last_ti(ti->then_part());
	    }
	}
    }

    D_SELFTEST(500) { cout << "Did not find any tree_instr\n"; }
    D_SELFTEST_FOOTER(500);

    return NULL;
}


/*---------------------------------------------------------------------------
		  Predicates to test for special instructions
---------------------------------------------------------------------------*/
/**
Return true if the given instruction is a null operation (nop, mrk).
*/
bool
is_nop(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    if_ops op = ti->instr()->opcode();
    return (op == io_nop || op == io_mrk);
}


/**
Return true if the given instruction is any kind of control transfer.
*/
bool
is_cti(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    return (is_ubr(ti) || is_cbr(ti) || is_mbr(ti) || is_ret(ti));
}

/**
Return true if the instruction is an unconditional jump.
*/
bool
is_ubr(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    return (ti->instr()->opcode() == io_jmp);
}

/**
Return true if the instruction is a conditional jump.
*/
bool
is_cbr(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    if_ops op = ti->instr()->opcode();
    return (op == io_btrue || op == io_bfalse);
}

/**
Return true if the instruction is a multi-way branch (mbr)
*/
bool
is_mbr(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    return (ti->instr()->opcode() == io_mbr);
}

/**
Return true if the instruction is a return instruction.
*/
bool
is_ret(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    return (ti->instr()->opcode() == io_ret);
}


/**
Return true if the instruction is a label.
*/
bool
is_label(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    return (ti->instr()->opcode() == io_lab);
}


/**
Return true if the instruction is 'fall-through'. A fall-through
instruction will allow the instruction following it to execute. The non
fall-through instructions are 'ret', 'jmp' and 'mbr'.
*/
bool
is_fall_through(p_tree_node tn)
{
    return (!is_ret(tn) && !is_ubr(tn) && !is_mbr(tn));
}

/**
Return true if the given instruction is a function call.
*/
bool
is_call(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) { return false; }

    p_tree_instr ti = tn;
    return (ti->instr()->opcode() == io_cal);
}



/**
Return 'true' if the given statement is executable (ie, not a label or a
no-op)
*/
bool
is_exec(p_tree_node tn)
{
    return (!is_nop(tn) && !is_label(tn));
}


/**
Return the label symbol of a label instruction.
*/
p_label_sym
get_label(p_tree_instr ti)
{
    assert(is_label(ti));
    return (((in_lab *)(ti->instr()))->label());
}

/**
Return the target sym_node of a branch instruction.
*/
p_sym_node
get_target(p_tree_instr ti)
{
    assert(is_cti(ti));
    return (((in_bj *)(ti->instr()))->target());
}


/**
Return true if the given tree_node is allowed inside a ccfg_block node. The
node must be an instruction, not a cobegin, not a parloop and not a label.
*/
bool
is_allowedInBlock(p_tree_node tn)
{
    return (tn && tn->is_instr() && !is_cobegin(tn) && !is_parloop(tn) && 
	    !is_label(tn));
}


/**
Return true if the given tree_node must be at the start of a ccfg_block
node. Node starters are:

1- Statements that contain conflicting uses.

2- Head synchronization statements (wait)

3- Mutex synchronization statements (lock/unlock). Note that these are
   also node finalizers. So, these statements must be the only
   statements inside a basic block.
*/
bool
is_startOfBlock(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) return false;

    bool hasConflictingUses = false;
    p_tree_instr ti = tn;
    instruction *in = ti->instr();
    set_varuse *ti_refs = ::get_varuses(ti.ptr());

    /* Traverse all the references in the instruction looking for
     * conflicting uses.
     */
    if (ti_refs) {
	set_varuse::iterator i;
	for (i = ti_refs->begin(); i != ti_refs->end(); i++) {
	    p_varuse use = *i;
	    if (use->conflicts().size() > 0) {
		hasConflictingUses = true;
		break;
	    }
	}
    }

    return (hasConflictingUses || is_head_sync(in) || is_mutex_sync(in));
}


/**
Return true if the given tree_node must be at the end of a ccfg_block
node. Node finalizers are:

1- Statements that contain conflicting definitions.

2- Tail synchronization statements (set)

3- Mutex synchronization statements (lock/unlock). Note that these are
   also node starters. So these statements must be the only
   statements inside a basic block.
*/
bool
is_endOfBlock(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) return false;

    bool hasConflictingDefs = false;
    p_tree_instr ti = tn;
    instruction *in = ti->instr();
    set_vardef *ti_refs = ::get_vardefs(ti.ptr());

    /* Traverse all the references in the instruction looking for
     * conflicting defintions.
     */
    if (ti_refs) {
	set_vardef::iterator i;
	for (i = ti_refs->begin(); i != ti_refs->end(); i++) {
	    p_vardef def = *i;
	    if (def->conflicts().size() > 0) {
		hasConflictingDefs = true;
		break;
	    }
	}
    }

    return (hasConflictingDefs || is_tail_sync(in) || is_mutex_sync(in) || 
	    is_cti(ti) || is_label(ti));
}


/*---------------------------------------------------------------------------
	Functions to access CCFG and CSSAME annotations for SUIF objects

Notice that functions returning collections will never return a NULL
pointer, they will return an empty collection and annotate the object with
the newly created collection. This avoids extra checks in the client code.
It might lead to memory leaks if the annotations are not properly disposed
of by the various destructors.
---------------------------------------------------------------------------*/
p_ccfg
get_ccfg(p_tree_block block)
{
    return (ccfg *)get_annote_for(block.ptr(), k_ccfg);
}
    

set_varref *
get_varrefs(suif_object *obj, bool create_if_null)
{
    set_varref *set = (set_varref *)get_annote_for(obj, k_refs);
    if (!set && create_if_null) {
	set = new set_varref;
	set_annote_for(obj, k_refs, (void *)set);
    }

    return set;
}


set_vardef *
get_vardefs(suif_object *obj)
{
    set_vardef *set = (set_vardef *)get_annote_for(obj, k_defs);
    if (!set) {
	set = new set_vardef;
	set_annote_for(obj, k_defs, (void *)set);
    }

    return set;
}


set_varuse *
get_varuses(suif_object *obj)
{
    set_varuse *set = (set_varuse *)get_annote_for(obj, k_uses);
    if (!set) {
	set = new set_varuse;
	set_annote_for(obj, k_uses, (void *)set);
    }

    return set;
}


set_phiterm *
get_phiterms(suif_object *obj)
{
    set_phiterm *set = (set_phiterm *)get_annote_for(obj, k_phi_terms);
    if (!set) {
	set = new set_phiterm;
	set_annote_for(obj, k_phi_terms, (void *)set);
    }
    
    return set;
}

set_phiterm *
get_piterms(suif_object *obj)
{
    set_phiterm *set = (set_phiterm *)get_annote_for(obj, k_pi_terms);
    if (!set) {
	set = new set_phiterm;
	set_annote_for(obj, k_pi_terms, (void *)set);
    }

    return set;
}


list_conflict *
get_conflicts(suif_object *obj, bool create_if_null)
{
    list_conflict *list = (list_conflict *)get_annote_for(obj, k_conflicts);
    if (!list && create_if_null) {
	list = new list_conflict;
	set_annote_for(obj, k_conflicts, (void *)list);
    }

    return list;
}


vector_vardef *
get_reaching_defs(suif_object *obj)
{
    vector_vardef *vector = (vector_vardef *)get_annote_for(obj, k_reaching_defs);
    if (!vector) {
	cout << "Warning. Could not find reaching defs for object.\n";
	vector = new vector_vardef;
	set_annote_for(obj, k_reaching_defs, (void *)vector);
    }

    return vector;
}

vector_varuse *
get_reached_uses(suif_object *obj)
{
    vector_varuse *set = (vector_varuse *)get_annote_for(obj, k_reached_uses);
    if (!set) {
	set = new vector_varuse;
	set_annote_for(obj, k_reached_uses, (void *)set);
    }

    return set;
}


p_vardef 
get_currdef(p_var_sym var)
{
    return (vardef *)var->peek_annote(k_currdef);
}

mutex_struct *
get_mutex_struct(p_var_sym var)
{
    assert_msg(is_mutex_sync(var), ("Only synchronization variables can have "
		"an associated mutex structure\n"));
    return (mutex_struct *)var->peek_annote(k_mutex_struct);
}


set_ccfg_node *
get_entry_set(p_var_sym var)
{
    assert_msg(is_mutex_sync(var), ("Only synchronization variables can have "
		"an associated mutex structure\n"));
    return (set_ccfg_node *)var->peek_annote(k_entry_set);
}


set_ccfg_node *
get_exit_set(p_var_sym var)
{
    assert_msg(is_mutex_sync(var), ("Only synchronization variables can have "
		"an associated mutex structure\n"));
    return (set_ccfg_node *)var->peek_annote(k_exit_set);
}

bool
is_mutex_sync(p_var_sym var)
{
    return (var->peek_annote(k_mutex_sync) != NULL);
}

set_lockref *
get_lockrefs(p_var_sym var)
{
    set_lockref *set = (set_lockref *)var->peek_annote(k_lock_refs);
    if (!set) {
	set = new set_lockref;
	set_annote_for(var.ptr(), k_lock_refs, (void *)set);
    }

    return set;
}


set_unlockref *
get_unlockrefs(p_var_sym var)
{
    set_unlockref *set = (set_unlockref *)var->peek_annote(k_unlock_refs);
    if (!set) {
	set = new set_unlockref;
	set_annote_for(var.ptr(), k_unlock_refs, (void *)set);
    }

    return set;
}


p_ccfg_node
get_node(p_tree_node n)
{
    return (ccfg_node *)get_annote_for(n.ptr(), k_ccfg_node);
}

void *
get_annote_for(suif_object *obj, char *an)
{
    if (obj->is_tree_obj()) {
	/* Special case for tree_instr's. Annotations are actually stored
	 * in the instruction.
	 */
	tree_node *tn = (tree_node *)obj;
	if (tn->is_instr()) { obj = ((tree_instr *)tn)->instr(); }
    }

    return obj->peek_annote(an);
}



/*---------------------------------------------------------------------------
	 Functions to set CCFG and CSSAME annotations onto SUIF objects
---------------------------------------------------------------------------*/
void
set_ccfg(p_tree_block block, p_ccfg graph)
{
    set_annote_for(block.ptr(), k_ccfg, (void *)graph.ptr());
}


void
set_varrefs(suif_object *obj, set_varref *refs)
{
    set_annote_for(obj, k_refs, (void *)refs);
}

void
set_reaching_defs(suif_object *obj, vector_vardef *rdefs)
{
    set_annote_for(obj, k_reaching_defs, (void *)rdefs);
}

void
set_reached_uses(suif_object *obj, vector_varuse *ruses)
{
    set_annote_for(obj, k_reached_uses, (void *)ruses);
}

void
set_mutex_struct(suif_object *obj, mutex_struct *mxstruct)
{
    set_annote_for(obj, k_mutex_struct, (void *)mxstruct);
}

void
set_entry_set(suif_object *obj, set_ccfg_node *npart)
{
    set_annote_for(obj, k_entry_set, (void *)npart);
}

void
set_exit_set(suif_object *obj, set_ccfg_node *xpart)
{
    set_annote_for(obj, k_exit_set, (void *)xpart);
}

void
set_conflicts(suif_object *obj, list_conflict *conf)
{
    set_annote_for(obj, k_conflicts, (void *)conf);
}

void
set_begin_node(suif_object *obj, p_ccfg_begin node)
{
    set_annote_for(obj, k_ccfg_begin, (void *)node.ptr());
}

void
set_cobegin_node(suif_object *obj, p_ccfg_cobegin node)
{
    set_annote_for(obj, k_ccfg_cobegin, (void *)node.ptr());
}

void
set_end_node(suif_object *obj, p_ccfg_end node)
{
    set_annote_for(obj, k_ccfg_end, (void *)node.ptr());
}

void
set_coend_node(suif_object *obj, p_ccfg_coend node)
{
    set_annote_for(obj, k_ccfg_coend, (void *)node.ptr());
}

void
set_node(suif_object *obj, p_ccfg_node node)
{
    set_annote_for(obj, k_ccfg_node, (void *)node.ptr());
}

void
set_test_node(suif_object *obj, p_ccfg_test node)
{
    set_annote_for(obj, k_ccfg_test, (void *)node.ptr());
}

void
set_currdef(p_var_sym var, p_vardef def)
{
    var->get_annote(k_currdef);
    var->append_annote(k_currdef, (void *)def.ptr());
}


void
set_annote_for(suif_object *obj, char *an, void *value)
{
    if (obj->is_tree_obj()) {
	/* Special case for tree_instr's. Annotations are actually stored
	 * in the instruction.
	 */
	tree_node *tn = (tree_node *)obj;
	if (tn->is_instr()) { obj = ((tree_instr *)tn)->instr(); }
    }

    if (obj->peek_annote(an)) { obj->get_annote(an); }
    obj->append_annote(an, value);
}
