/**
$Id: varref.cc,v 2.29 1999/03/13 19:12:51 diego Exp $
*/
#include <algo.h>

#include <suif_copyright.h>

#define _MODULE_ "CSSAME"

#include <string.h>

#include <suif.h>
#include <par.h>
#include <odc-util.h>
#include <d_lib.h>

#include "ccfg.h"
#include "cssame.h"

/**
Constructor.
*/
varref::varref(p_var_sym var, instruction *instr, p_ccfg_node node)
{ 
    D_SELFTEST_HEADER(310, "varref::varref");
    
    D_SELFTEST(310) {
	cout << "Building a new reference object for variable '" << var->name();
	cout << "' (" << var << ")\n";
    }

    _var		= var; 
    _instr		= instr; 
    _lineno		= source_line_num(_instr);
    _node		= node; 
    _chain		= NULL;
    _save_chain		= NULL;
    _mark		= NULL;
    _has_been_cloned	= false;
    _is_clone		= false;
    
    /* Append this reference to the set of references for its node and graph */
    if (_node) {
	_node->refs().insert(this);
	_node->parent()->refs().insert(this);
    }

    D_SELFTEST_FOOTER(310);
}

/**
Return a clone to this reference
*/
p_varref
varref::clone(instruction *call_site)
{
    _has_been_cloned = true;
    p_varref clone = this->make_clone(call_site);
    clone->_is_clone = true;
    clone->_original = this;

    return clone;
}


/**
Add this reference to the various collections attached to SUIF objects.
This is done here so that we can update annotations in one spot. It makes
it easier to add new annotations later on. New annotation names should be
declared in ccfg.h and init.cc
*/
void
varref::updateAnnotations() const
{
    D_SELFTEST_HEADER(320, "varref::updateAnnotations");

    D_SELFTEST(320) {
	cout << "Adding annotations to variable " << _var
	    << " (kind = " << this->ckind() << ")\n\n";
    }

    /* k_refs: References made in the statement and variable. */
    if (_instr) { ::get_varrefs(_instr)->insert((varref *)this); }
    ::get_varrefs(_var.ptr())->insert((varref *)this);

    D_SELFTEST(321) {
	cout << "\nReferences for variable '" << _var << "'\n";
	set_varref *refs = ::get_varrefs(_var.ptr());
	copy(refs->begin(), refs->end(), ostream_iterator<p_varref>(cout,"\n"));
    }

    if (this->isD()) {
	/* k_defs: Definitions made in the statement and to the variable. */
	if (_instr) { ::get_vardefs(_instr)->insert((vardef *)this); }
	::get_vardefs(_var.ptr())->insert((vardef *)this);
    }

    if (this->isU()) {
	/* k_uses: Uses in the statement and variable. */
	if (_instr) { ::get_varuses(_instr)->insert((varuse *)this); }
	::get_varuses(_var.ptr())->insert((varuse *)this);
    }

    if (this->isPHI()) {
	/* k_phi_terms: Phi terms in the statement and variable */
	if (_instr) { ::get_phiterms(_instr)->insert((phiterm *)this); }
	::get_phiterms(_var.ptr())->insert((phiterm *)this);
    }

    if (this->isPI()) {
	/* k_pi_terms: Pi terms in the statement and variable */
	if (_instr) { ::get_piterms(_instr)->insert((phiterm *)this); }
	::get_piterms(_var.ptr())->insert((phiterm *)this);
    }

    if (this->isSync()) {
	/* k_sync_var: Variable is a synchronization variable */
	_var->get_annote(k_sync_var);
	_var->append_annote(k_sync_var, (void *)this);
    }

    if (this->isMutexSync()) {
	/* k_mutex_sync: This variable is a mutex synch variable */
	_var->get_annote(k_mutex_sync);
	_var->append_annote(k_mutex_sync, (void *)this);
    }

    if (this->isLock()) {
	/* k_lock_refs: List of lock references to the variable */
	::get_lockrefs(_var)->insert((lockref *)this);
    }

    if (this->isUnlock()) {
	/* k_unlock_refs: List of unlock references to the variable */
	::get_unlockrefs(_var)->insert((unlockref *)this);
    }

    D_SELFTEST(320) {
	cout << "Annotations for variable " << _var << ":";
	_var->print_annotes(); fflush(stdout);
	cout << endl << endl;
	if (_instr) {
	    cout << "Annotations for statement " << _instr->number() << ":";
	    _instr->print_annotes(); fflush(stdout);
	    cout << endl;
	}
    }

    D_SELFTEST_FOOTER(320);
}



/**
Remove the variable reference object from memory;
*/
varref::~varref()
{
    D_SELFTEST_HEADER(280, "varref::~varref");
    D_SELFTEST(280) {
	cout << "Removing varref for '" << _var->name() << "'"
	    << " (this = " << this << ")\n";
    }

    D_SELFTEST_FOOTER(280);
}



/**
Print the variable reference to the given stream
*/
void
varref::print(ostream& f) const
{
    f << _var->name() << " " << this->ckind() << " at line " << this->lineno();

    if (_chain && this->isU()) { f << " -> "; _chain->print(f); }

    if (_node) { f << " [#" << _node->number() << "]"; }

    if (_is_clone) { f << " [clone]"; }
}

ostream& operator<<(ostream& os, const varref& ref)
{
    ref.print(os);

    return os;
}


/**
Print a compact version of the variable reference to the given stream
*/
void
varref::print_brief(ostream& f) const
{
    f << this->ckind() << "_" << _var->name() << "(" << this->lineno() << ")";

    if (_chain && this->isU()) {
	f << " -> "; _chain->print_brief(f);
    }

    if (_is_clone) { f << " [c]"; }
}


//***************************************************************************
//			 Methods for class phiterm
//***************************************************************************
/**
Class constructor. Attach the $\phi$ or $\pi$ term to a variable and node.
Initialize the list of immediate reached uses to NULL. Attach the term to
the first instruction of the node.
*/
phiterm::phiterm(p_var_sym var, p_ccfg_node node, RefKind kind)
    : vardef(var, NULL, node)
{
    D_SELFTEST_HEADER(310, "phiterm::phiterm");

    D_SELFTEST(310) { 
	cout << "New " << ((kind == PHI) ? "PHI" : "PI")
	    << " term for variable '" << var->name() << "' at node "; 
	node->print(); cout << endl;
	cout << endl;
    }

    _kind = kind;
    _removed = false;

    /* Attach the term to the first instruction of the given CCFG node */
    if (node->is_begin() || node->is_end() || node->is_label()) {
	p_tree_node tn = p_ccfg_marker(node)->node();

	D_SELFTEST(310) { 
	    cout << "Attaching term to tree_node:\n";
	    tn->print();
	    fflush(stdout);
	}

	if (tn->is_instr()) {
	    _instr = p_tree_instr(tn)->instr();
	    _lineno = source_line_num(_instr);
	} else if (tn->is_loop()) {
	    if (node->is_begin()) {
		_instr = first_ti(p_tree_loop(tn)->body())->instr();
		_lineno = source_line_num(_instr);
	    } else {
		_instr = last_ti(p_tree_loop(tn)->test())->instr();
		_lineno = source_line_num(_instr);
	    }
	} else if (tn->is_for()) {
	    if (node->is_begin()) {
		_instr = first_ti(p_tree_for(tn)->lb_list())->instr();
		_lineno = source_line_num(_instr);
	    } else {
		_instr = last_ti(p_tree_for(tn)->body())->instr();
		_lineno = source_line_num(_instr) + 1;
	    }
	} else if (tn->is_if()) {
	    if (node->is_begin()) {
		_instr = first_ti(p_tree_if(tn)->header())->instr();
		_lineno = source_line_num(_instr);
	    } else {
		p_tree_if tif = tn;
		if (tif->else_part()->is_empty() == false) {
		    _instr = last_ti(tif->else_part())->instr();
		} else {
		    _instr = last_ti(tif->then_part())->instr();
		}
		_lineno = source_line_num(_instr) + 1;
	    }
	} else if (tn->is_block() || tn->is_proc()) {
	    if (node->is_begin()) {
		_instr = first_ti(p_tree_block(tn)->body())->instr();
		_lineno = source_line_num(_instr);
	    } else {
		_instr = last_ti(p_tree_block(tn)->body())->instr();
		_lineno = source_line_num(_instr);
	    }
	} else {
	    cout << "Unknown tree node kind " << tn->kind()
		<< " at CCFG marker node " << node->number() << endl;
	    assert(false);
	}
    } else if (node->is_instr()) {
	_instr = p_ccfg_instr(node)->instr()->instr();
	_lineno = source_line_num(_instr);
    } else if (node->is_block()) {
	_instr = p_ccfg_block(node)->in_head()->instr();
	_lineno = source_line_num(_instr);
    } else if (node->is_test()) {
	_instr = first_ti(p_ccfg_test(node)->for_loop()->lb_list())->instr();
	_lineno = source_line_num(_instr);
    } else if (node->is_cobegin()) {
	_instr = p_ccfg_cobegin(node)->last_instr()->instr();
	_lineno = source_line_num(_instr);
    } else if (node->is_coend()) {
	_instr = p_ccfg_coend(node)->last_instr()->instr();
	_lineno = source_line_num(_instr);
    } else {
	cout << "Unknown node kind " << node->kind()
	    << ". This can't be good.\n";
	assert(false);
    }

    if (!_instr) {
	cout << endl << __FILE__ << "(" << __LINE__ << "): "
	    << "Warning. Could not find a tree_instr for reference\n"
	    << *this << endl << "in CCFG node: "; node->print(); cout << endl;
	_lineno = 0;
    } else {
	D_SELFTEST(310) {
	    cout << "Attaching term to instruction:\n";
	    _instr->print();
	    fflush(stdout);
	}
    }

    this->updateAnnotations();

    /* Append this phi/pi to the set of phis/pis for its node and graph */
    if (_node) {
	if (this->isPHI()) {
	    _node->phiterms().insert(this);
	    _node->parent()->phiterms().insert(this);
	} else if (this->isPI()) {
	    _node->piterms().insert(this);
	    _node->parent()->piterms().insert(this);
	} else {
	    assert(false);
	}
    }

    D_SELFTEST_FOOTER(310);
}



/**
Remove the PHI term from memory.
*/
phiterm::~phiterm()
{
    D_SELFTEST_HEADER(280, "phiterm::~phiterm");
    D_SELFTEST(280) {
	cout << "Removing phiterm for '" << _var->name() << "'\n";
    }

    D_SELFTEST_FOOTER(280);
}


/**
Print the phi term to the given stream
*/
void
phiterm::print(ostream& f) const
{
    f << this->ckind() << "_" << _var->name() << "(" << this->lineno() << ")";

    f << " = " << this->ckind() << "(";

    vector_vardef::const_iterator i;
    unsigned pos;
    for (i = _phi_chain.begin(), pos = 1; i != _phi_chain.end(); i++, pos++) {
	p_vardef arg = *i;
	if (arg) {
	    arg->print_brief(f);
	    if (pos < _phi_chain.size()) { f << ", "; }
	}
    }
    f << ")";

    if (_node) { f << " [#" << _node->number() << "]"; }

    if (_removed) { f << " [REMOVED]"; }
}

ostream& operator<<(ostream& os, const phiterm& ref)
{
    ref.print(os);

    return os;
}


void
phiterm::print_brief(ostream& f) const
{
    this->varref::print_brief(f);

    if (_node) { f << " [#" << _node->number() << "]"; }

    if (_removed) { f << " [REMOVED]"; }
}


//***************************************************************************
//			 Methods for class vardef
//***************************************************************************
/**
Class constructor
*/
vardef::vardef(p_var_sym var, instruction *in, p_ccfg_node node)
    : varref(var, in, node)
{
    D_SELFTEST_HEADER(310, "vardef::vardef");

    this->updateAnnotations();

    /* Append this definition to the set of defs for its node and graph */
    if (_node) {
	_node->defs().insert(this);
	_node->parent()->defs().insert(this);
    }

    D_SELFTEST_FOOTER(310);
}

p_varref vardef::make_clone(instruction *call_site) const
{
    return new vardef(this->var(), call_site, ::get_node(call_site->parent()));
}


/**
Class destructor. Removes this definition from the list of definitions for
the associated variable and statement. Also free up space taken up by the
list of immediate uses reached by this definition.
*/
vardef::~vardef()
{
    D_SELFTEST_HEADER(280, "vardef::~vardef");
    D_SELFTEST(280) { cout << "Removing vardef for '" << _var->name() << "'\n"; }

    D_SELFTEST_FOOTER(280);
}


//***************************************************************************
//			 Methods for class varuse
//***************************************************************************
/**
Build a new variable use object.
*/
varuse::varuse(p_var_sym var, instruction *in, p_ccfg_node node)
    : varref(var, in, node)
{
    D_SELFTEST_HEADER(310, "varuse::varuse");

    this->updateAnnotations();

    /* Append this use to the set of uses for its node and graph */
    if (_node) {
	_node->uses().insert(this);
	_node->parent()->uses().insert(this);
    }

    D_SELFTEST_FOOTER(310);
}

p_varref varuse::make_clone(instruction *call_site) const
{
    return new varuse(this->var(), call_site, ::get_node(call_site->parent()));
}


/**
Remove the use from memory.
*/
varuse::~varuse()
{
    D_SELFTEST_HEADER(280, "varuse::~varuse");
    D_SELFTEST(280) { cout << "Removing varuse for '" << _var->name() << "'\n"; }

    D_SELFTEST_FOOTER(280);
}


//***************************************************************************
//			Methods for class syncref
//***************************************************************************
/**
Build a new synchronization reference.
*/
syncref::syncref(p_var_sym var, instruction *in, p_ccfg_node node)
    : varref(var, in, node)
{
    D_SELFTEST_HEADER(310, "syncref::syncref");

    if_ops op = in->opcode();
    assert(op == io_cal);

    /* Retrieve the name of the function being called */
    _name = ::func_symbol(in)->name();

    D_SELFTEST(310) { cout << "Synchronization operation: " << _name << "\n"; }

    D_SELFTEST_FOOTER(310);
}

/**
Print the synchronization reference to the given stream
*/
void
syncref::print(ostream& f) const
{
    f << _name << "(" << _var->name() << ") at line " << this->lineno();
    f << " (" << this->ckind() << ")";
    if (_node) { f << " [#" << _node->number() << "]"; }
    if (_is_clone) { f << " [clone]"; }
}

ostream& operator<<(ostream& os, const syncref& ref)
{
    ref.print(os);
    return os;
}


//***************************************************************************
//		       Methods for class head_syncref
//***************************************************************************
/**
Build a synchronization reference for a wait() operation
*/
head_syncref::head_syncref(p_var_sym var, instruction *in, p_ccfg_node node)
    : syncref(var, in, node)
{
    D_SELFTEST_HEADER(310, "head_syncref::head_syncref");

    this->updateAnnotations();

    D_SELFTEST_FOOTER(310);
}

p_varref head_syncref::make_clone(instruction *call_site) const
{
    head_syncref *r;
    r = new head_syncref(this->var(),call_site,::get_node(call_site->parent()));
    r->_name = this->name();
    return r;
}


/**
Remove the wait() synchronization reference from memory.
*/
head_syncref::~head_syncref()
{
    D_SELFTEST_HEADER(280, "head_syncref::~head_syncref");

    D_SELFTEST(280) {
	cout << "Removing head_syncref for '" << _var->name() << "'\n";
    }

    D_SELFTEST_FOOTER(280);
}


//***************************************************************************
//		       Methods for class tail_syncref
//***************************************************************************
/**
Build a synchronization reference for a set() operation
*/
tail_syncref::tail_syncref(p_var_sym var, instruction *in, p_ccfg_node node)
    : syncref(var, in, node)
{
    D_SELFTEST_HEADER(310, "tail_syncref::tail_syncref");

    this->updateAnnotations();

    D_SELFTEST_FOOTER(310);
}


p_varref tail_syncref::make_clone(instruction *call_site) const
{
    tail_syncref *r;
    r = new tail_syncref(this->var(),call_site,::get_node(call_site->parent()));
    r->_name = this->name();
    return r;
}

/**
Remove the synchronization reference from memory.
*/
tail_syncref::~tail_syncref()
{
    D_SELFTEST_HEADER(280, "tail_syncref::~tail_syncref");

    D_SELFTEST(280) {
	cout << "Removing tail_syncref for '" << _var->name() << "'\n";
    }

    D_SELFTEST_FOOTER(280);
}



//***************************************************************************
//			   Methods for class lockref
//***************************************************************************
/**
Build a new lock reference
*/
lockref::lockref(p_var_sym var, instruction *in, p_ccfg_node node)
    : mutex_syncref(var, in, node)
{
    D_SELFTEST_HEADER(310, "lockref::lockref");

    this->updateAnnotations();

    D_SELFTEST_FOOTER(310);
}

p_varref lockref::make_clone(instruction *call_site) const
{
    lockref *r;
    r = new lockref(this->var(), call_site, ::get_node(call_site->parent()));
    r->_name = this->name();
    return r;
}

/**
Remove the synchronization reference from memory.
*/
lockref::~lockref()
{
    D_SELFTEST_HEADER(280, "lockref::~lockref");

    D_SELFTEST(280) {
	cout << "Removing lockref for '" << _var->name() << "'\n";
    }

    D_SELFTEST_FOOTER(280);
}


//***************************************************************************
//			   Methods for class unlockref
//***************************************************************************
/**
Build a new unlock reference
*/
unlockref::unlockref(p_var_sym var, instruction *in, p_ccfg_node node)
    : mutex_syncref(var, in, node)
{
    D_SELFTEST_HEADER(310, "unlockref::unlockref");

    this->updateAnnotations();

    D_SELFTEST_FOOTER(310);
}

p_varref unlockref::make_clone(instruction *call_site) const 
{
    unlockref *r;
    r = new unlockref(this->var(), call_site, ::get_node(call_site->parent()));
    r->_name = this->name();
    return r;
}

/**
Remove the synchronization reference from memory.
*/
unlockref::~unlockref()
{
    D_SELFTEST_HEADER(280, "unlockref::~unlockref");

    D_SELFTEST(280) {
	cout << "Removing unlockref for '" << _var->name() << "'\n";
    }

    D_SELFTEST_FOOTER(280);
}


//***************************************************************************
//		       Methods for class conflict
//***************************************************************************
/**
Constructor. Mark the statements for the two references with the conflict.
*/
conflict::conflict(p_varref tail, p_varref head) : _tail(tail), _head(head)
{
    list_conflict *conf;

    /* Append the conflict to the tail reference statement */
    conf = ::get_conflicts(_tail->instr());
    if (conf == NULL) {
	conf = new list_conflict;
	set_conflicts(_tail->instr(), conf);
    }
    conf->push_back(*this);

    /* Append the conflict to the head reference statement */
    conf = ::get_conflicts(_head->instr());
    if (conf == NULL) {
	conf = new list_conflict;
	set_conflicts(_head->instr(), conf);
    }
    conf->push_back(*this);

    /* Tell the reference objects that they participate in a memory conflict */
    _tail->conflicts().push_back(*this);
    _head->conflicts().push_back(*this);
}

/**
Destructor. Remove the conflict annotations.
*/
conflict::~conflict()
{
}


/**
Print the conflict to the given stream
*/
void
conflict::print(ostream& f) const
{
    _tail->print(); f << " conflicts with ";  _head->print(); f << endl;
}

ostream& operator<<(ostream& os, const conflict& conflict)
{
    conflict.print(os);
    return os;
}



/*---------------------------------------------------------------------------
				 Free functions
---------------------------------------------------------------------------*/
/**
Return true if the given instruction is a call to one of the given
function names.
*/
static bool
is_in_list(instruction *instr, char **names)
{
    if_ops op = instr->opcode();
    if (op != io_cal) return false;

    unsigned i = 0;
    char *name = ::func_symbol(instr)->name();
    while (names[i] != NULL) {
	if (strcmp(name, names[i]) == 0) { return true; }
	i++;
    }

    return false;
}


/**
Return true if the given instruction is a call to a synchronization
function that is supposed to be at the beginning of basic blocks (eg,
wait())
*/
bool
is_head_sync(instruction *instr)
{
    
    static char *_head_sync_names[] = { "_wait", NULL };
    return is_in_list(instr, _head_sync_names);
}



/**
Return true if the given instruction is a call to a synchronization
function that is supposed to be at the end of basic blocks (eg,
_Set())
*/
bool
is_tail_sync(instruction *instr)
{
    static char *_tail_sync_names[] = { "_set", NULL };
    return is_in_list(instr, _tail_sync_names);
}

/**
Return true if the given instruction is a call to a mutex synchronization 
function.
*/
bool
is_mutex_sync(instruction *instr)
{
    static char *_mutex_sync_names[] = { "suif_lock", "suif_unlock", NULL };
    return is_in_list(instr, _mutex_sync_names);
}



/**
Return true if the given instruction is a call to suif_lock()
*/
bool
is_lock_sync(instruction *instr)
{
    static char *_lock_sync_names[] = { "suif_lock", NULL };
    return is_in_list(instr, _lock_sync_names);
}


/**
Return true if the given instruction is a call to suif_unlock()
*/
bool
is_unlock_sync(instruction *instr)
{
    static char *_unlock_sync_names[] = { "suif_unlock", NULL };
    return is_in_list(instr, _unlock_sync_names);
}


/**
Return true if the given instruction is a call to one of the side-effect-free
functions. These functions do not modify any of the local variables of the
function even if they are passed pointers to them.

This is a dangerous assumption because we are identifying them by name, we
are not actually doing IPA. This should be a switch to the compiler.
bool
has_side_effects(instruction *call)
{
    static char *_known_names[] = {
	"printf", "acos", "acosf", "asin", "asinf", "atan", "atan2f", "atanf",
	"ceil", "ceilf", "cos", "cosf", "drand48", "erand48", "fabs", "fabsf",
	"floor", "floorf", "fmod", "fmodf", "log", "log10", "log10f", "logf",
	"lrand48", "modf", "pow", "powf", "rand", "random", "sin", "sinf",
	"sqrt", "sqrtf", "srand", "srand48", "srandom", "tan", NULL
    };

    return !is_in_list(call, _known_names);
}
*/
