/**
$Id: varref.h,v 2.21 1999/03/13 19:12:52 diego Exp $
*/
#ifndef __VAR_REF_H
#define __VAR_REF_H

#include <set>
#include <list>
#include <iostream.h>

#include <d_lib.h>

using namespace std;

enum RefKind { D, U, PHI, PI, HSYNC, TSYNC, USYNC, MUTEXSYNC, LOCK, UNLOCK };

/**
Identification predicates
*/
bool is_head_sync(instruction *instr);
bool is_tail_sync(instruction *instr);
bool is_mutex_sync(instruction *instr);
bool is_lock_sync(instruction *instr);
bool is_unlock_sync(instruction *instr);
/* bool has_side_effects(instruction *call); */

/**
Reference conflicts are tuples of references to the same memory location
that are in conflict:

\begin{enumerate}
\item	The references are made by concurrent threads.
\item	At least one of them is a write operation.
\end{enumerate}
*/
class conflict { D_CLASS(conflict)
public:
    conflict(p_varref tail, p_varref head);
    conflict(const conflict& r) { _tail = r.tail(); _head = r.head(); }
    ~conflict();

    conflict& operator=(const conflict& r) {
	_tail = r.tail(); _head = r.head(); return *this;
    }

    bool operator==(const conflict& r) const {
	return (_tail.ptr() == r.tail().ptr() && _head.ptr() == r.head().ptr());
    }

    bool operator<(const conflict& r) const {
	return ((void *)this) < ((void *)&r);
    }

    const p_varref& tail() const { return _tail; }
    const p_varref& head() const { return _head; }

    void print(ostream& f=cout) const;

protected:
    p_varref _tail;
    p_varref _head;
};


/**
Base class for variable references. A variable reference represents
operations on variables. These operations can be user-defined (definitions
or uses) or they can be introduced by the compiler (phi terms, pi terms,
synchronization references).
*/
class varref { D_CLASS(varref)
    friend class ccfg;
    friend class cssame;
    
public:
    varref(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~varref();

    p_var_sym var() const		{ return _var; }
    instruction *instr() const		{ return _instr; }
    p_ccfg_node node() const		{ return _node; }

    virtual RefKind kind() const = 0;
    virtual bool isD() const		{ return false; }
    virtual bool isU() const		{ return false; }
    virtual bool isPHI() const		{ return false; }
    virtual bool isPI() const		{ return false; }
    virtual bool isHSync() const	{ return false; }
    virtual bool isTSync() const	{ return false; }
    virtual bool isUSync() const	{ return false; }
    virtual bool isMutexSync() const	{ return false; }
    virtual bool isLock() const		{ return false; }
    virtual bool isUnlock() const	{ return false; }
    virtual bool isSync() const		{ return false; }

    virtual char *ckind() const = 0;
    list_conflict& conflicts() { return _conflicts; }
    bool are_conflicts_computed() const { return _are_conflicts_computed; }
    void set_conflicts_computed() { _are_conflicts_computed = true; }
    int lineno() const { return _lineno; }

    /** clone() makes a deep copy of the varref object and sets a pointer
     * to the cloned reference. Sometimes it's useful to know that we are
     * dealing with a clone. Clones are mainly used at function call sites.
     *
     * The called function is analyzed and all the references to variables
     * visible at the call site are cloned at the call site.
     */
    p_varref clone(instruction *call_site);
    bool is_clone() const { return _is_clone; }
    bool has_been_cloned() const { return _has_been_cloned; }
    p_varref original() const { return _original; }
     
    /** chain() is a link from a use of a variable at this reference to the
     * immediate reaching definition or phi-term.
     */
    p_vardef chain() const { return _chain; }

    /** Reaching definitions. Generated by cssame::computeReachingDefs() */
    vector_vardef& reachingDefs() { return _rdefs; }

    /* Reached uses information. Generated by cssame::computeReachingDefs() */
    vector_varuse& reachedUses() { return _ruses; }

    virtual void print(ostream& f=cout) const;
    virtual void print_brief(ostream& f=cout) const;

    /* Comparison operators used to insert varref objects (or wrapped
     * pointers to varref objects) into sorted collections.
     */
    bool operator<(const varref& other) const {
	return this->lineno() < other.lineno();
    }

    bool operator==(const varref& other) const { return this == &other; }

    /* Associate this variable reference to a given node. */
    void set_node(p_ccfg_node node) { _node = node; }

protected:
    p_var_sym _var;
    instruction *_instr;
    unsigned _lineno;
    p_ccfg_node _node;
    list_conflict _conflicts;
    bool _are_conflicts_computed;
    vector_vardef _rdefs;
    vector_varuse _ruses;
    p_vardef _chain;
    p_vardef _save_chain;
    p_varuse _mark;
    p_varref _confRef;		/// Reference that conflicts with me.
    bool _has_been_cloned;	/// The reference has been cloned.
    bool _is_clone;		/// The reference is a clone.
    p_varref _original;		/// Original reference (for clones).

    /** Make a deep copy of the current variable reference. */
    virtual p_varref make_clone(instruction *call_site) const = 0;

    void set_chain(p_vardef def) { _chain = def; }

    /** save_chain() is a temporary placeholder to save the old reaching
     * definition when a new definition or phi-term is reached.
     */
    p_vardef save_chain() const { return _save_chain; }
    void set_save_chain(p_vardef def) { _save_chain = def; }

    /* Marks set by cssame::followChain() when computing reaching definitions */
    bool isMarkedBy(const p_varuse& u) { return _mark.ptr() == u.ptr(); }
    void markedBy(p_varuse u) { _mark = u; }

    /* Methods to update the annotations to SUIF objects */
    void updateAnnotations() const;
};


/**
Variable definitions. A variable definition can be a $\phi$ term, a $\pi$ term
or an actual definition in the source program. Every definition knows about the
uses immediately affected by it. This information is collected during the
construction of the CSSAME form (cssame::searchFUDChains and
cssame::placePiFunctions).
*/
class vardef : public varref { D_CLASS(vardef)
public:
    vardef(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~vardef();

    vector_varuse& immediateUses() { return _immediateUses; }
    virtual RefKind kind() const { return D; }
    virtual char *ckind() const	 { return "D"; }
    virtual bool isD() const { return true; }

protected:
    /* List of immediate reached uses. This list is actually built when the
     * CSSAME information is computed. The methods that build the FUD
     * chains for $\phi$ and $\pi$ terms are responsible for adding the
     * uses that are immediately affected by this definition.
     *
     * This list is used to prune the CSSAME information later on. When
     * $\pi$ or $\phi$ terms are found to be superfluous they are removed
     * from the CSSAME structure. To properly update the CSSAME
     * information, the uses immediately reached by this definition must be
     * linked to whatever definition reaches this $\phi$ or $\pi$ term. Of
     * course, this process should never be performed on real definitions.
     */
    vector_varuse _immediateUses;

    virtual p_varref make_clone(instruction *call_site) const;
};


/**
Variable uses. These represent read operations on variables.
*/
class varuse : public varref { D_CLASS(varuse)
public:
    varuse(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~varuse();

    virtual RefKind kind() const { return U; }
    virtual char *ckind() const { return "U"; }
    virtual bool isU() const { return true; }

protected:
    virtual p_varref make_clone(instruction *call_site) const;
};



/**
$\phi$ and $\pi$ terms. $\phi$ and $\pi$ are a special type of variable
definition that model multiple reaching definitions to a single variable
use via multiple control paths ($\phi$ terms) or concurrent definitions
($\pi$ terms). Since these definitions are not part of the program, they
attach themselves to the first instruction of the node which contains them.
*/
class phiterm : public vardef { D_CLASS(phiterm)
public:
    phiterm(p_var_sym var, p_ccfg_node node, RefKind kind = PHI);
    virtual ~phiterm();

    /** phi_chain() is a vector of links from a phi-term at this reference
     * to the reaching definitions along each CFG predecessor.
     */
    vector_vardef& phi_chain() { return _phi_chain; }

    virtual void print(ostream& f=cout) const;
    virtual char *ckind() const	{ return (_kind == PHI) ? "Phi" : "Pi"; }
    virtual RefKind kind() const { return _kind; }
    virtual bool isPHI() const { return _kind == PHI; }
    virtual bool isPI() const { return _kind == PI; }
    virtual bool isD() const { return false; }

    /** Mark a $\phi$ term as removed when it is superfluous and you want
     * to ignore it when traversing the SSA web.
     */
    bool removed() const { return _removed; }
    void remove() { _removed = true; }

    virtual void print_brief(ostream& f=cout) const;

protected:
    RefKind _kind;
    vector_vardef _phi_chain;
    bool _removed;	/// true if this term has been removed from the graph.
};


/**
Base synchronization reference class. Synchronization references represent
calls to synch operations in the program.
*/
class syncref : public varref { D_CLASS(syncref)
public:
    syncref(p_var_sym var, instruction *in, p_ccfg_node node);

    char *name() const { return _name; }
    virtual void print(ostream& f=cout) const;

    virtual bool isSync() const	{ return true; }

protected:
    char *_name;
};


/**
Head synchronization references represent one of the operations for
directed synchronization operations like set/wait. A head synch reference
represents the wait part.
*/
class head_syncref : public syncref { D_CLASS(head_syncref)
public:
    head_syncref(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~head_syncref();

    virtual bool isHSync() const { return true; }
    virtual char *ckind() const { return "HSync"; }
    virtual RefKind kind() const { return HSYNC; }

protected:
    virtual p_varref make_clone(instruction *call_site) const;
};


/**
Tail synchronization references represent one of the operations for
directed synchronization operations like set/wait. A tail synch reference
represents the set part.
*/
class tail_syncref : public syncref { D_CLASS(tail_syncref)
public:
    tail_syncref(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~tail_syncref();

    virtual bool isTSync() const { return true; }
    virtual char *ckind() const { return "TSync"; }
    virtual RefKind kind() const { return TSYNC; }

protected:
    virtual p_varref make_clone(instruction *call_site) const;
};


/**
Undirected synchronization operations do not enforce execution ordering.
They are typically used to guarantee mutual exclusion.
*/
class undir_syncref : public syncref { D_CLASS(undir_syncref)
protected:
public:
    undir_syncref(p_var_sym var, instruction *in, p_ccfg_node node) :
	syncref(var, in, node) {}
    virtual ~undir_syncref() {}

    virtual bool isUSync() const { return true; }
};


/**
Mutual exclusion synchronization operations.
*/
class mutex_syncref : public undir_syncref { D_CLASS(mutex_syncref)
public:
    mutex_syncref(p_var_sym var, instruction *in, p_ccfg_node node) :
	undir_syncref(var, in, node) {}
    virtual ~mutex_syncref() {}

    virtual bool isMutexSync() const { return true; }
    virtual bool isLock() const	{ return false; }
    virtual bool isUnlock() const { return false; }
};


/**
Lock operations.
*/
class lockref : public mutex_syncref { D_CLASS(lockref)
public:
    lockref(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~lockref();

    virtual bool isLock() const	{ return true; }
    virtual char *ckind() const { return "Lock"; }
    virtual RefKind kind() const { return LOCK; }

protected:
    virtual p_varref make_clone(instruction *call_site) const;
};


/**
Unlock operations.
*/
class unlockref : public mutex_syncref { D_CLASS(unlockref)
public:
    unlockref(p_var_sym var, instruction *in, p_ccfg_node node);
    virtual ~unlockref();

    virtual bool isUnlock() const { return true; }
    virtual char *ckind() const { return "Unlock"; }
    virtual RefKind kind() const { return UNLOCK; }

protected:
    virtual p_varref make_clone(instruction *call_site) const;
};


/* Free functions */
ostream& operator<<(ostream& os, const conflict& conflict);
ostream& operator<<(ostream& os, const varref& ref);
ostream& operator<<(ostream& os, const phiterm& ref);
ostream& operator<<(ostream& os, const syncref& ref);

#endif	// __VAR_REF_H
