/**
$Id: find-cobegin.cc,v 2.1 1999/02/19 20:06:44 diego Exp $

Routines to look for cobegin statements in the program. Cobegins are
represented by switch() statements which are tricky to detect in SUIF
because SUIF has a very peculiar way of generating code for switch
statements. If there are less than 4 cases, it converts the switch to an
if-tree. Otherwise, it generates code to check the bounds of the
variable and a multiway branch.
*/
#include <string.h>
#include <suif.h>
#include <suif_copyright.h>

#include "findpar.h"

DECLARE_LIST_CLASS(label_sym_list, label_sym *);

static boolean is_cobegin(tree_node *t);
static boolean isIfTree(tree_node *t);
static label_sym_list *findThreadsInIfTree(tree_node *t);
static label_sym_list *findThreadsInMbr(tree_node *t);
static label_sym_list *findCobeginThreads(tree_node *t);
static void replaceSwitch(tree_node *t, label_sym_list *labelList);

static label_sym *defLabel;


//---------------------------------------------------------------------------
// findCobegin()
// Analyze the given tree_node looking for a cobegin. If found, modify the
// statement to look like this:
//
//   mbr .__ODC_cobegin, low 1, L:main.L3, (L:main.L4, L:main.L5)
//   lab L:main.L4
//   ...
//   jmp L:main.L3
//   lab L:main.L5
//   ...
//   jmp L:main.L3
//---------------------------------------------------------------------------
void 
find_cobegin(tree_node *t, void *)
{
    if (is_cobegin(t)) {
	label_sym_list *labelList = findCobeginThreads(t);
	replaceSwitch(t, labelList);

    }
}


//---------------------------------------------------------------------------
// is_cobegin()
// Returns TRUE if the given tree_node is a cobegin statement. A cobegin
// statement always begins with an assignment statement from __ODC_cobegin
// to a SUIF temporary variable.
//---------------------------------------------------------------------------
boolean
is_cobegin(tree_node *t)
{
    if (t->is_instr()) {
	instruction *instr = ((tree_instr *)t)->instr();
	if (instr->opcode() == io_cpy) {
	    operand src = ((in_rrr *)instr)->src1_op();
	    if (src.is_symbol() && 
		strcmp(src.symbol()->name(), COBEGIN) == 0) {
		return TRUE;
	    }
	}
    }

    return FALSE;
}

//---------------------------------------------------------------------------
// isIfTree()
// Returns TRUE if the given tree_node is the start of a switch statement
// represented by an if-tree. ie, the switch statement contains less than 4
// cases.
//
//	switch (__ODC_cobegin) {
//		case 1: ...; break;
//		case 2: ...; break;
//	}
//
// Looks like this (in pseudo-SUIF):
//
//   31: cpy main.suif_tmp1 = .__ODC_cobegin
//   32: btrue  main.suif_tmp1 == 1, L:main.L4
//   35: btrue  main.suif_tmp1 == 2, L:main.L5
//   38: jmp L:main.L3
//   40: lab L:main.L4
//       ...
//   46: jmp L:main.L3
//   47: lab L:main.L5
//       ...
//   53: jmp L:main.L3
//   54: lab L:main.L3
//
// If we add two more cases, the switch statement looks like this:
//
//   31: cpy main.suif_tmp1 = .__ODC_cobegin
//   32: btrue  main.suif_tmp1 < 1, L:main.L8
//   35: btrue  main.suif_tmp1 > 4, L:main.L8
//   38: mbr main.suif_tmp1, low 1, L:main.L3, (L:main.L4, L:main.L5, L:main.L6, L:main.L7)
//   39: lab L:main.L8
//   40: jmp L:main.L3
//   42: lab L:main.L4
//	 ...
//   48: jmp L:main.L3
//   49: lab L:main.L5
//	 ...
//   55: jmp L:main.L3
//   56: lab L:main.L6
//	 ...
//   62: jmp L:main.L3
//   63: lab L:main.L7
//	 ...
//   69: jmp L:main.L3
//   70: lab L:main.L3
//
// To determine which case we are handling, we check the first btrue
// statement. If the comparison operator is '<', the switch statement is
// represented by an mbr => The threads are the different cases in the mbr.
// Otherwise, the switch statement contains less than 4 cases and each
// btrue contains a jump to the body of the corresponding thread.
//
// WARNING:	This is a gross hack and seems to be highly version
// 		dependent. I don't know what is going to happen with the
// 		next version of SUIF. If they decide to generate different
// 		code for switch statements, this function will need to be
// 		re-implemented.
//---------------------------------------------------------------------------
boolean
isIfTree(tree_node *t)
{
    tree_node_list_iter iter(t->parent());
    iter.set(t->list_e());

    /* Skip over the assignment statement. */
    iter.step();

    /* Grab the next instruction which should be a btrue branch. If the
     * comparison is '==' then the switch is represented with an if-tree,
     * otherwise it is the bounds check prior to an mbr.
     */
    tree_node *tn;
    instruction *instr;

    tn = iter.step();
    assert(tn->is_instr());

    instr = ((tree_instr *)tn)->instr();
    assert(instr->opcode() == io_btrue);

    if (((in_bj *)instr)->src_op().instr()->opcode() == io_seq) {
	return TRUE;
    } else {
	return FALSE;
    }
}

//---------------------------------------------------------------------------
// findThreadsInIfTree()
// Return a list of tree_node_list. Each element is a list of tree_nodes
// representing the statements in the thread.
//
//	switch (__ODC_cobegin) {
//		case 1: ...; break;
//		case 2: ...; break;
//	}
//
// Looks like this (in pseudo-SUIF):
//
//   31: cpy main.suif_tmp1 = .__ODC_cobegin
//   32: btrue  main.suif_tmp1 == 1, L:main.L4
//   35: btrue  main.suif_tmp1 == 2, L:main.L5
//   38: jmp L:main.L3
//   40: lab L:main.L4
//       ...
//   46: jmp L:main.L3
//   47: lab L:main.L5
//       ...
//   53: jmp L:main.L3
//   54: lab L:main.L3
//---------------------------------------------------------------------------
label_sym_list *
findThreadsInIfTree(tree_node *t)
{
    /* Collect labels for each of the threads and the default label. Each
     * thread is entered by one of the btrue statements at the beginning of
     * the construct. The end of the thread is delimited by a jump to the
     * default label. The default label is jumped to right after all the
     * btrues that jump to the different threads.
     */
    unsigned numThreads;
    boolean foundAllThreads;
    tree_node_list_iter iter(t->parent());
    label_sym_list *labelList = new label_sym_list;

    /* Skip over the initial assignment to suif_tmp. */
    iter.set(t->list_e());
    iter.step();

    numThreads = 0;
    foundAllThreads = FALSE;
    while (foundAllThreads == FALSE) {
	/* We should not run out of nodes in the program */
	assert(iter.is_empty() == FALSE);

	/* Get the next node. It should be an instruction node. More
	 * specifically, it should be a btrue or a jmp.
	 */
	tree_node *tn = iter.step();
	assert(tn->is_instr());
	
	instruction *instr = ((tree_instr *)tn)->instr();
	assert(instr->opcode() == io_jmp || instr->opcode() == io_btrue);

	/* If we get here, we are only dealing with jmp or btrue
	 * instructions. If it's a jmp we save the label as the default
	 * label for the switch.
	 */
	if (instr->opcode() == io_jmp) {
	    defLabel = ((in_bj *)instr)->target();
	    foundAllThreads = TRUE;
	} else {	/* instr->opcode() == btrue */
	    numThreads++;
	    label_sym *label = ((in_bj *)instr)->target();
	    labelList->append(label);
	}
    }

    return labelList;
}


//---------------------------------------------------------------------------
// findTreadsInMbr()
// Return a list of tree_node_list. Each element is a list of tree_nodes
// representing the statements in the thread.
//
// The following switch statement:
//
//	switch (__ODC_cobegin) {
//		case 1: ...; break;
//		case 2: ...; break;
//		case 3: ...; break;
//		case 4: ...; break;
//	}
//
// Looks like this (in pseudo-SUIF):
//
//   31: cpy main.suif_tmp1 = .__ODC_cobegin
//   32: btrue  main.suif_tmp1 < 1, L:main.L8
//   35: btrue  main.suif_tmp1 > 4, L:main.L8
//   38: mbr main.suif_tmp1, low 1, L:main.L3, (L:main.L4, L:main.L5, L:main.L6, L:main.L7)
//   39: lab L:main.L8
//   40: jmp L:main.L3
//   42: lab L:main.L4
//	 ...
//   48: jmp L:main.L3
//   49: lab L:main.L5
//	 ...
//   55: jmp L:main.L3
//   56: lab L:main.L6
//	 ...
//   62: jmp L:main.L3
//   63: lab L:main.L7
//	 ...
//   69: jmp L:main.L3
//   70: lab L:main.L3
//---------------------------------------------------------------------------
label_sym_list *
findThreadsInMbr(tree_node *t)
{
    unsigned numThreads, i;
    tree_node_list_iter iter(t->parent());
    label_sym_list *labelList;

    /* Skip over the initial assignment to suif_tmp. */
    iter.set(t->list_e());
    iter.step();

    /* Skip over the two btrue statements that check the bounds of the
     * switch variable.
     */
    iter.step();
    iter.step();

    /* The next tree_node should be an mbr instruction */
    tree_node *tn = iter.step();
    assert(tn->is_instr());

    instruction *instr = ((tree_instr *)tn)->instr();
    assert(instr->opcode() == io_mbr);

    /* Now we find out what the default label is and collect the label for
     * each thread.
     */
    in_mbr *mbr = (in_mbr *)instr;
    defLabel = mbr->default_lab();
    numThreads = mbr->num_labs();

    labelList = new label_sym_list;
    for (i = 0; i < numThreads; i++) {
	labelList->append(mbr->label(i));
    }

    return labelList;
}

//---------------------------------------------------------------------------
// findCobeginThreads()
// Build a list of threads starting at the given cobegin node.
//---------------------------------------------------------------------------
label_sym_list *
findCobeginThreads(tree_node *t)
{
    if (isIfTree(t)) {
	return findThreadsInIfTree(t);
    } else {
	return findThreadsInMbr(t);
    }
}

//---------------------------------------------------------------------------
// replaceSwitch()
// Replace the header of the switch by a single mbr to each thread.
//---------------------------------------------------------------------------
void
replaceSwitch(tree_node *t, label_sym_list *labelList)
{
    tree_node_list *prog;
    tree_node_list_iter progIter;
    operand cobegin_op;
    in_mbr *mbr_i;
    tree_instr *mbr_t;
    unsigned i;

    /* Replace the cobegin by a single mbr instruction */
    cobegin_op = ((in_rrr *)((tree_instr *)t)->instr())->src1_op();
    mbr_i = new in_mbr(cobegin_op, 0, labelList->count(), defLabel);
    mbr_i->append_annote(k_cobegin);

    /* Set all the labels in the new mbr instruction before adding it
     * to the program.
     */
    for (i = 0; i < labelList->count(); i++) {
	mbr_i->set_label(i, (*labelList)[i]);
    }

    /* Add the new mbr instruction before the original switch statement */
    mbr_t = new tree_instr(mbr_i);
    assert(mbr_t);
    prog = t->parent();
    prog->insert_before(mbr_t, t->list_e());

    /* Remove the header of the original switch statement from the program.
     * Remove all tree nodes from the first node indicated by 't' to the
     * label for the first thread body indicated by the first label of the
     * new mbr instruction just inserted.
     */
    progIter.set(t->list_e());
    boolean removedAll = FALSE;
    label_sym *firstThreadLabel = mbr_i->label(0);
    while (removedAll == FALSE) {
	tree_node *tn = progIter.step();

	if (tn->is_instr()) {
	    instruction *instr = ((tree_instr *)tn)->instr();
	    if (instr->opcode() == io_lab &&
		((in_lab *)instr)->label() == firstThreadLabel) {
		removedAll = TRUE;
		continue;
	    }
	}

	prog->remove(tn->list_e());
    }
}
