/**
$Id: Error.h,v 1.5 1998/12/31 04:22:55 diego Exp $

Error handling classes. Each error type is a separate class that contains all
the information pertaining the error. Each error type knows how to display
itself on stdout and what action to take after the error has occurred.
Errors are supposed to be thrown at the error point and caught somewhere
above the function that encountered the problem. If the error is not caught
it will propagate all the way back to main().

All the error types inherit from the base class Error which only knows in
which source file and line number the error occurred but not its nature.
There are two virtual methods that should be defined by each error class:

1-  action(): This method is responsible for recovering from the error.
    The default recovery action is to call ::exit(-1) (Defined in class
    Error).

2-  show(): This method shows a message explaining the error.
*/
#ifndef __OCC_ERROR_H
#define __OCC_ERROR_H

#include <suif.h>

/**
Base class for error messages. It contains the compiler's source file and
line number where the error was triggered.
*/
class Error {
    const char *_file;
    int _line;

    virtual void printHeader() const {
	cout << endl << "\"" << _file << "\", line " << _line << ": ";
    }

    public:
    Error(const char *f, int l) : _file(f), _line(l) {
	this->printHeader();
    }

    virtual void action() const { ::exit(-1); }
    virtual void show() const {
	cout  << "Unkown error code. This is not good." << endl;
    }
};



/**
Missing environment variables.
*/
class _E_EnvVarMissing : public Error {
    const String _var;

    public:
    _E_EnvVarMissing(const char *f, int l, const String& var) : 
	Error(f, l), _var(var) { }

    virtual void show() const {
	cout << "Environment variable " << _var << " is not set. Aborting." 
	     << endl;
    }
};


/**
Failed conversion process. One of the external passes needed to convert the
program failed.
*/
class _E_Conversion : public Error {
    const SourceFile *_file;
    const String _cmd;

    public:
    _E_Conversion(const char *f, int l, const SourceFile *file, 
	    const String& cmd) : Error(f, l), _file(file), _cmd(cmd) {}

    virtual void show() const {
	cout << "Transformation failed" << endl;
	cout << *_file;
	cout << "Command: " << _cmd << endl;
    }
};


/**
System call failure.
*/
class _E_SystemCall : public Error {
    int _errno;
    const String _syscall;

    public:
    _E_SystemCall(const char *f, int l, int errno, const String& syscall) :
	Error(f, l), _errno(errno), _syscall(syscall) {}

    virtual void show() const {
	cout << "System error:  " << ::strerror(_errno) << endl;
	cout << "While calling: " << _syscall <<endl;
    }
};


/**
The user requested an unrecognized DFA pass name.
*/
class _E_UnknownDFAPass : public Error {
    const String _name;

    public:
    _E_UnknownDFAPass(const char *f, int l, const String& name) : 
	Error(f, l), _name(name) {}
    
    virtual void show() const {
	cout << "Unrecognized DFA pass name '" << _name << "'." << endl;
    }
};


class _E_UnknownOPTPass : public Error {
    const String _name;

    public:
    _E_UnknownOPTPass(const char *f, int l, const String& name) : 
	Error(f, l), _name(name) {}
    
    virtual void show() const {
	cout << "Unrecognized OPT pass name '" << _name << "'." << endl;
    }
};


/**
Abstract super-class for transformation errors.
*/
class _E_Transformation : public Error {
    protected:
    const FileState *_state;

    public:
    _E_Transformation(const char *f, int l, const FileState *s) : 
	Error(f, l), _state(s) {}

    virtual void show() const {
	cout << ((_state != NULL) ? _state->id() : "[NULL state]") << ": ";
    }
};


/**
Unrecognized file extension. A file transformation failed because it didn't
recognize the file extension.
*/
class _E_UnknownExtension : public _E_Transformation {
    const String _ext;

    public:
    _E_UnknownExtension(const char *f, int l, const FileState *s,
	    const String& e) : _E_Transformation(f, l, s), _ext(e) {}

    virtual void show() const {
	this->_E_Transformation::show();
	cout << "Unrecognized file extension '" << _ext << "'." << endl;
    }
};

class _E_CantSetFSE : public _E_Transformation {
    public:
    _E_CantSetFSE(const char *f, int l, const FileState *s) :
	_E_Transformation(f, l, s) {}

    virtual void show() const {
	this->_E_Transformation::show();
	cout << "Only dfaState objects can have an associated fse" << endl;
    }
};

class _E_CantFindCSSAME : public Error {
    tree_proc *_tp;

    public:
    _E_CantFindCSSAME(const char *f, int l, tree_proc *tp) : 
	Error(f, l), _tp(tp) {}

    virtual void show() const {
	cout << "Cannot find CSSAME form for procedure " 
	    << _tp->proc()->name() << endl;
    }
};
#endif	// __OCC_ERROR_H
