/**
$Id: FileState.cc,v 1.10 1999/03/02 18:38:41 diego Exp $

File states. Each input file goes through a series of states as the
compiler transforms it. File states can be internal or external. An
external file state represents a disk file that is processed with an
external SUIF pass. For instance, C source files are parsed with 'scc',
.spd files are analyzed with 'porky', etc.

Internal states are transformations done internally by the compiler itself.
These include data flow analysis, which generate a flow graph and a CSSAME
representation for the program. Optimization passes are also represented.

State transitions are done by the file itself. Each state knows what its
next state should be. The state object within the file changes itself from
object to object as the compilation progresses
*/
#include <list>

#include <suif.h>
#include <cg.h>
#include <par.h>
#include <ccfg.h>
#include <cssame.h>
#include <odc-util.h>
#include <str.h>

#include "compiler.h"

using namespace std;

/**
Execute an external command to perform the state change. This method builds
a command line out of the given cmd and flags. It throws an _E_Conversion
exception if the command fails.
*/
void
ExternalState::execute(const String& cmd, const String& flags)
{
    if (_file->ext() != _ext) {
	throw _E_UnknownExtension(__FILE__, __LINE__, this, _file->ext());
    }

    String shell(String("cd ") + _file->dirname() + " ; " + cmd + " " + flags);

    if (OCC->verbose()) {
	cout << "% " << shell << endl;
    }

    int ec = ::system(shell);
    if (ec != 0) {
	throw _E_Conversion(__FILE__, __LINE__, _file, shell);
    }

    /* Add the current file to the list of temporary files to be removed at
     * the end of the compilation process. We only do this if this state is
     * not the first or last state in the transformation process. If the
     * current file extension is either the original extension or the last
     * extension requested by the user, then we do nothing.
     */
    if (_ext == _file->origext() || _ext == OCC->stop_at()) {
	/* Do nothing. Files at the start and end of the state chain are
	 * never removed.
	 */
    } else {
	OCC->addTmpFile(_file->name());
    }
}

void
FileState::set_fse(file_set_entry *fse)
{ 
    fse = fse;
    throw _E_CantSetFSE(__FILE__, __LINE__, this);
}


/**
State change method for source files. Parse the source file by calling
'scc'.
*/
FileState *
SourceState::change()
{
    this->execute("scc", String("-.spd ") + OCC->defines() + OCC->incdirs() + 
	    _file->basename() + _file->ext());

    return(new spdState(_file));
}


/**
State change method for .spd files. Run 'porky' to do simple clean-up of
the IL.
*/
FileState *
spdState::change()
{
    this->execute("porky", String("-ucf-opt ") + _file->basename() + 
	    _file->ext() + " " + _file->basename() + ".spx");

    return(new spxState(_file));
}


/**
State change for .spx files. Run 'findpar' to find cobegin/coend and
parloop constructs.
*/
FileState *
spxState::change()
{
    this->execute("findpar", _file->basename() + _file->ext() + " " + 
	    _file->basename() + ".od1");

    return(new dfaState(_file));
}


/**
State change for .odc files. Run 'genpar' to generate the SUIF
annotations used to generate parallel code.
*/
FileState *
odcState::change()
{
    this->execute("genpar", _file->basename() + _file->ext() + " " + 
	    _file->basename() + ".pa1");

    return(new pa1State(_file));
}


/**
State change for .pa1 files. Run SUIF's 'pgen' to generate the actual
parallel code.
*/
FileState *
pa1State::change()
{
    this->execute("pgen", _file->basename() + _file->ext() + " " + 
	    _file->basename() + ".pa2");

    return(new pa2State(_file));
}


/**
State change for .pa2 files. Run 'fixpgen' to clean up some of the
oddities generated by 'pgen' and fix label names (there is a bug in pgen
that causes it to generate wrong label names sometimes).

This is the last transformation in the chain. The remaining classes
parState and oState are 'final' states that need no further conversion.
They only need to be linked together. The linking process is done after
all these individual transformations.
*/
FileState *
pa2State::change()
{
    this->execute("fixpgen", _file->basename() + _file->ext() + " " + 
	    _file->basename() + ".par");

    return(new parState(_file));
}
