/**
$Id: FileState.h,v 1.7 1999/03/13 19:12:52 diego Exp $

File states. Each input file goes through a series of states as the
compiler transforms it. File states can be internal or external. An
external file state represents a disk file that is processed with an
external SUIF pass. For instance, C source files are parsed with 'scc',
.spd files are analyzed with 'porky', etc.

Internal states are transformations done internally by the compiler itself.
These include data flow analysis, which generate a flow graph and a CSSAME
representation for the program. Optimization passes are also represented.

State transitions are done by the file itself. Each state knows what its
next state should be. The state object within the file changes itself from
object to object as the compilation progresses
*/
#ifndef __FILE_STATE_H
#define __FILE_STATE_H

class file_set_entry;

/**
Abstract super-class.
*/
class FileState {
    friend class SourceFile;

    protected:
    const String _ext;
    SourceFile *_file;

    public:
    FileState(SourceFile *file, const char *ext) : 
	_file(file), _ext(ext) {}

    /** State change method. Must be implemented by derived classes */
    virtual FileState *change() = 0;

    /** Termination test. If this state is the last in the chain return
     * true. There are two conditions that can trigger this:
     * 
     * 1- If the current file extension is the one the user wanted to stop
     *    the process at.
     * 
     * 2- Or, if this is a final state. Final states unconditionally return
     *    true (ie, they override this default method). At this time only
     *    parState and oState do this.
     */
    virtual bool done() const { return (_file->ext() == OCC->stop_at()); }

    /** Identification string. */
    virtual const char *id() const { return "None"; }

    /** File extension associated with this state. */
    const String &ext() const { return _ext; }
    
    /** Source file associated with this state. */
    SourceFile *file() const { return _file; }

    /** Set the SUIF file_set_entry associated with this state. Only
     * dfaState objects should have an fse.
     */
    virtual void set_fse(file_set_entry *fse);

    /** Identification predicates. Each derived class should overwrite the
     * appropriate one.
     */
    virtual bool isExternal() const { return false; }
    virtual bool isSource() const { return false; }
    virtual bool isSPD() const { return false; }
    virtual bool isSPX() const { return false; }
    virtual bool isODC() const { return false; }
    virtual bool isPA1() const { return false; }
    virtual bool isPA2() const { return false; }
    virtual bool isPAR() const { return false; }
    virtual bool isO() const { return false; }

    virtual bool isInternal() const { return false; }
    virtual bool isDFA() const { return false; }
    virtual bool isOPT() const { return false; }
};


/**
External states. These model transformation done by different SUIF passes.
*/
class ExternalState : public FileState {
    protected:
    void execute(const String& cmd, const String& flags);

    public:
    ExternalState(SourceFile *file, const char *ext) : 
	FileState(file, ext) {}
    virtual const char *id() const { return "External"; }
    virtual bool isExternal() const { return true; }
    virtual FileState *change() = 0;
};


class SourceState : public ExternalState {
    public:
    SourceState(SourceFile *file) : ExternalState(file, ".c") {}
    virtual const char *id() const { return "SourceState"; }
    virtual bool isSource() const { return true; }
    virtual FileState *change();
};

class spdState : public ExternalState {
    public:
    spdState(SourceFile *file) : ExternalState(file, ".spd") {}
    virtual const char *id() const { return "spdState"; }
    virtual bool isSPD() const { return true; }
    virtual FileState *change();
};

class spxState : public ExternalState {
    public:
    spxState(SourceFile *file) : ExternalState(file, ".spx") {}
    virtual const char *id() const { return "spxState"; }
    virtual bool isSPX() const { return true; }
    virtual FileState *change();
};

class odcState : public ExternalState {
    public:
    odcState(SourceFile *file) : ExternalState(file, ".odc") {}
    virtual const char *id() const { return "odcState"; }
    virtual bool isODC() const { return true; }
    virtual FileState *change();
};

class pa1State : public ExternalState {
    public:
    pa1State(SourceFile *file) : ExternalState(file, ".pa1") {}
    virtual const char *id() const { return "pa1State"; }
    virtual bool isPA1() const { return true; }
    virtual FileState *change();
};

class pa2State : public ExternalState {
    public:
    pa2State(SourceFile *file) : ExternalState(file, ".pa2") {}
    virtual const char *id() const { return "pa2State"; }
    virtual bool isPA2() const { return true; }
    virtual FileState *change();
};

class parState : public ExternalState {
    public:
    parState(SourceFile *file) : ExternalState(file, ".par") {}
    virtual const char *id() const { return "parState"; }
    virtual bool isPAR() const { return true; }
    virtual FileState *change() { return this; }
    virtual bool done() const { return true; }
};

class oState : public ExternalState {
    public:
    oState(SourceFile *file) : ExternalState(file, ".o") {}
    virtual const char *id() const { return "oState"; }
    virtual bool isO() const { return true; }
    virtual FileState *change() { return this; }
    virtual bool done() const { return true; }
};


/**
Internal states. These model analyses and optimizations done inside the
compiler. All the internal transformations are performed on .od1 files, so
they all keep the same extension.
*/
class InternalState : public FileState {
    public:
    InternalState(SourceFile *file) : FileState(file, ".od1") {}

    virtual const char *id() const { return "Internal"; }
    virtual bool isInternal() const { return true; }
    virtual FileState *change() = 0;
};

class dfaState : public InternalState { D_CLASS(dfaState)
public:
    dfaState(SourceFile *file);

    virtual const char *id() const { return "dfaState"; }
    virtual bool isDFA() const { return true; }
    virtual FileState *change();
    virtual void set_fse(file_set_entry *fse) { _fse = fse; }

    file_set_entry *fse() const { return _fse; }
    map<p_tree_proc, set_varref *>& varref_map() { return _varref_map; }
    map<p_tree_proc, list_conflict *>& conflict_map() { return _conflict_map; }
    map<p_tree_proc, p_ccfg>& ccfg_map() { return _ccfg_map; }
    map<p_tree_proc, vector_tree_par *>& tree_par_map() { return _tree_par_map;}
    cg *call_graph() { return _call_graph; }

protected:
    file_set_entry *_fse;
    cg *_call_graph;
    map<p_tree_proc, set_varref *> _varref_map;
    map<p_tree_proc, list_conflict *> _conflict_map;
    map<p_tree_proc, p_ccfg> _ccfg_map;
    map<p_tree_proc, vector_tree_par *> _tree_par_map;

    void runDFA();
    void runOPT();

    void buildDFAInfo();
    void removeDFAInfo();

    void build_threads();
    void find_refs();
    void find_confs();
    void build_ccfgs();

    void remove_threads();
    void remove_refs();
    void remove_confs();
    void remove_ccfgs();
};

#endif	// __FILE_STATE_H
