/**
$Id: Odyssey.cc,v 1.17 1999/02/23 23:47:22 diego Exp $

Odyssey compiler class. This class represents the Odyssey compiler. It
can analyze and optimize explicitly parallel C programs.
*/
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>

#include <suif.h>
#include <par.h>
#include <ccfg.h>
#include <cssame.h>
#include <odc-util.h>
#include <str-regexp.h>
#include <str.h>
#include <opt.h>

#include <d_lib.h>

#include "compiler.h"

/**
Constructor.
*/
Odyssey::Odyssey(int argc, char **argv)
{
    /* Initialize data members */
    _argc = argc;
    _argv = argv;
    _progname = _argv[0];
    _stop_at = "";
    _compile_only = false;
    _defines = "";
    _debug = false;
    _help = false;
    _incdirs = "";
    _keep_inter = false;
    _libdirs = "";
    _libs = "";
    _modify_output_name = false;
    _output_name_suffix = "";
    _output_name = "a.out";
    _optlevel = "";
    _verbose = 0;
    _version = false;
    _dfa_list = "";

    /* Set default include and library directories based on environment
     * variables ODYSSEY, SUIFHOME and MACHINE. Maybe we should set these
     * defaults based on some configuration file. But this should be enough
     * for now.
     */
    _ODYSSEY  = this->getenv("ODYSSEY");
    _SUIFHOME = this->getenv("SUIFHOME");
    _MACHINE  = this->getenv("MACHINE");

    _incdirs = "-I" + _ODYSSEY + "/include ";

    if (_MACHINE == "i386-linux" ) {
	_libdirs = "-L" + _SUIFHOME + "/" + _MACHINE + "/lib ";
	_libs = "-lruntime_pthreads -lpthread ";
    } else if (_MACHINE == "mips-sgi-irix5.3") {
	_libdirs = "-L" + _SUIFHOME + "/" + _MACHINE + "/lib ";
	_libs = "-lruntime_pthreads -lpthread ";
    } else {
	cout << "Architecture " << _MACHINE << " not supported yet. ";
	cout << "No binary will be generated" << endl;
    }
}


/**
Destructor.
*/
Odyssey::~Odyssey()
{
    if (_keep_inter == false) {
	if (_verbose > 0) {
	    cout << "Removing temporary files: ";
	}

	set<String>::iterator f;
	for (f = _tmpFiles.begin(); f != _tmpFiles.end(); f++) {
	    if (_verbose > 0) {
		cout << *f << " ";
	    }
	    if (::unlink(*f) != 0) {
		cout << "Warning: Could not remove temporary file " << *f 
		     << endl;
	    }
	}

	if (_verbose > 0) {
	    cout << endl;
	}
    }
}



/**
Custom environment query method. If the given environment variable doesn't
exist, it throws an exception.
*/
String
Odyssey::getenv(const char *varname)
{
    char *value = ::getenv(varname);
    if (value == NULL) {
	throw _E_EnvVarMissing(__FILE__, __LINE__, varname);
    }

    return String(value);
}


/**
Execute the compiler.
*/
int
Odyssey::run()
{
    this->parseCmdLine();

    D_SELFTEST(1) { cout << *this; }

    if (_help) { this->showHelp(); return 1; }

    if (_version) { this->showVersion(); return 1; }

    if (_files.empty()) { this->showHelp(); return 1; }


    try { 
	/* Transform all the input files to .odc files */
	this->parse();

	/* Analyze and optimize the .odc files */
	this->initSUIF();
	this->transform();
	this->exitSUIF();

	/* Generate parallel code */
	this->gencode();
    }
    catch (_E_Transformation& e) { e.show(); return -1; }


    /* Link all the files into the final executable */
    try { 
	this->link();
    }
    catch (Error& e) { e.show(); return -1; }


    /* Return UNIX style error code */
    return 0;
}



/**
Parse input files, convert them into .od1 files so that we can
transform/analyze them.
*/
void
Odyssey::parse()
{
    /* Process each file in the list. Note that we have to do some tricks
     * here. We cannot process each file using our own iterator because it
     * gets in the way of how SUIF handles files. SUIF wants us to build a
     * file_set with all the files that we want to process and iterate
     * through them using its own internal methods.
     * 
     * To get around this problem, we process every file up to the .odc
     * state. Then we take all the .odc files and use the SUIF iterators to
     * analyze/optimize them internally. Of course, none of this is needed
     * if the user asked to stop before the .odc state (which is when we
     * can apply our analysis and optimization passes).
     */
    list<SourceFile>::iterator f;
    for (f = _files.begin(); f != _files.end(); f++) {
	if (this->verbose() || _files.size() > 1) {
	    cout << f->origname() << ":" << endl;
	}

	while (!f->done() && !f->state()->isDFA()) { 
	    f->transform();
	}
    }
}


/**
Analyze and optimize the .od1 files. We have to build a SUIF file_set and
run all the passes using SUIF's iterators. As the .od1 files are analyzed
and optimized, they are written out to .odc files.
*/
void
Odyssey::transform()
{
    this->buildListOfDFAPasses();
    this->buildListOfOPTPasses();
    this->buildSUIFfileset();	/* Always build after DFA and OPT lists. */

    fileset->reset_iter();
    list<SourceFile>::iterator f;
    for (f = _files.begin(); f != _files.end(); f++) {
	if (f->state()->isDFA()) {
	    file_set_entry *fse = fileset->next_file();
	    f->state()->set_fse(fse);
	    f->transform();
	}
    }
}


/**
Generate parallel annotations and parallel code from the .odc files.
*/
void
Odyssey::gencode()
{
    list<SourceFile>::iterator f;
    for (f = _files.begin(); f != _files.end(); f++) {
	while (!f->done()) {
	    f->transform();
	}
    }
}


/**
Gather all the files that are in the .od1 state and add them to the SUIF's
fileset so that we can apply our internal transformations.
*/
void
Odyssey::buildSUIFfileset()
{
    list<SourceFile>::iterator f;
    for (f = _files.begin(); f != _files.end(); f++) {
	if (f->state()->isDFA()) {
	    String oldname(f->name());
	    String name(f->basename());

	    if (_modify_output_name) { 
		name += _output_name_suffix;
	    }

	    String newname(f->dirname() + "/" + name + ".odc");

	    fileset->add_file(oldname, newname);
	}
    }
}



/**
The '-F' argument is a comma-separated list of names. Each name corresponds
to one of the classes derived from dfaPass. The DFA passes will be
performed in the order given on the command line.

To incorporate new DFA passes, do the following:
1- Subclass the new pass from dfaPass.
2- Register the new pass in this method.
*/
void
Odyssey::buildListOfDFAPasses()
{
    String *array;
    String p(_dfa_list);

    if (p == "") {
	return;
    }

    int nfields = p.Split(array, ",");
    if (nfields == 0) {
	/* Only one DFA pass specified */
	nfields = 1;
	array = new String[nfields];
	array[0] = p;
    }

    for (int i = 0; i < nfields; i++) {
	dfa *pass;

	if (array[i] == "cssame") {
	    pass = new cssamePass(_verbose);
	} else {
	    throw _E_UnknownDFAPass(__FILE__, __LINE__, array[i]);
	}

	_dfaPasses.push_back(pass);
    }
}



/**
The '-P' argument is a comma-separated list of names. Each name corresponds
to one of the classes derived from optPass. The OPT passes will be
performed in the order given on the command line.

To incorporate new OPT passes, do the following:
1- Subclass the new pass from optPass in the directory OptLib.
2- Register the new pass in this method.
*/
void
Odyssey::buildListOfOPTPasses()
{
    String *array;
    String p(_opt_list);

    if (p == "") {
	return;
    }

    int nfields = p.Split(array, ",");
    if (nfields == 0) {
	/* Only one OPT pass specified */
	nfields = 1;
	array = new String[nfields];
	array[0] = p;
    }

    for (int i = 0; i < nfields; i++) {
	opt *pass;

	/* Determine whether we need to display stats */
	bool showStats;
	int  pos = array[i].Index("=");
	if (pos != -1 && pos < array[i].Len() - 1) {
	    showStats = (array[i][pos + 1] != '0') ;
	} else {
	    showStats = false;
	}

	RegExp re;

	if (array[i] == (re = RegExp("^dce=?"))) {
	    pass = new dce(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^licm=?"))) {
	    pass = new licm(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^mbl=?"))) {
	    pass = new mbl(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^rlicm=?"))) {
	    pass = new rlicm(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^gcp=?"))) {
	    pass = new gcp(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^cse=?"))) {
	    pass = new cse(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^copy_prop=?"))) {
	    pass = new copy_prop(showStats, _verbose);
	} else if (array[i] == (re = RegExp("^forward_prop=?"))) {
	    pass = new forward_prop(showStats, _verbose);
	} else {
	    throw _E_UnknownOPTPass(__FILE__, __LINE__, array[i]);
	}


	/** If the user specified the -n switch, modify the basename to
	 * reflect all the optimization passes that will be performed on
	 * the file.
	 */
	if (_modify_output_name) {
	    _output_name_suffix += String("-") + array[i];
	}

	_optPasses.push_back(pass);
    }
}



/**
Helper function to initialize SUIF.
*/
void
Odyssey::initSUIF()
{
    if (OCC->verbose()) { cout << "Initializing SUIF... "; }

    LIBRARY(par, init_par, exit_par);
    LIBRARY(cssame, init_cssame, exit_cssame);
    LIBRARY(useful, init_useful, exit_useful);
    LIBRARY(cg, init_cg, exit_cg);

    ::init_suif(_argc, _argv);

    if (OCC->verbose()) { cout << "Done" << endl; }
}


/**
Helper function to tell SUIF that we are done with the current file.
*/
void
Odyssey::exitSUIF()
{
    if (OCC->verbose()) { cout << "Shutting down SUIF... "; }

    ::exit_suif();

    if (OCC->verbose()) { cout << "Done" << endl; }
}



/**
Link all the object files into the final executable.
*/
bool
Odyssey::link()
{
    /* Check if we have to generate binaries. If the user asked to stop at
     * something other than '.o' then we just return. This method is needed
     * when the user wants to generate .o files or an executable.
     */
    if (_stop_at == "" || _stop_at == ".o") {
	/* Do nothing. Continue with the linking phase. */
    } else {
	return true;
    }

    /* Build the command line with all the .par and feed them, together
     * with all the appropriate CFLAGS, to scc. Notice that we always
     * 'link' (even when the user specified -c). This link stage is
     * actually a call to the native compiler to generate .o files and the
     * final executable.
     */
    String files;
    list<SourceFile>::iterator f;
    for (f = _files.begin(); f != _files.end(); f++) {
	/* Rename each file with extension .par to have an extension .spd so
	 * that 'scc' can recognize it. 
	 */
	String name;

	if (f->ext() == ".par") {
	    String oldname(f->name());
	    String newname(f->dirname() + "/" + f->basename() + ".spd");
	    int ec = ::rename(oldname, newname);
	    if (ec != 0) {
		throw _E_SystemCall(__FILE__, __LINE__, errno, 
			String("::rename(") + oldname + ", " + newname + ")");
	    }
	    name = newname;

	    /* Add the new file name to the set of temporary files */
	    this->addTmpFile(newname);
	} else {
	    name = f->name();
	}

	files += name + " ";
    }

    /* Modify the name of the executable if the user used the -n switch. */
    if (_modify_output_name) {
	_output_name += _output_name_suffix;
    }

    /* Grab all the CFLAGS */
    String CFLAGS;
    if (_compile_only)		 CFLAGS += "-c ";
    if (_debug)			 CFLAGS += "-g ";
    if (_libdirs != "")		 CFLAGS += _libdirs + " ";
    if (_libs != "" )		 CFLAGS += _libs + " ";
    if (_output_name != "a.out") CFLAGS += String(" -o ") + _output_name + " ";
    if (!_optlevel > 0)		 CFLAGS += String("-O") + _optlevel + " ";

    /* Build the command line and launch the compiler */
    String shellCmd(String("scc ") + CFLAGS + files);
    if (_verbose > 0) {
	cout << "Executing: " << shellCmd << endl;
    }
    int ec = ::system(shellCmd);
    if (ec != 0) {
	throw _E_SystemCall(__FILE__, __LINE__, errno, 
		String("::system(") + shellCmd + ")");
    }

    return true;
}



/**
Display the compiler state on the given stream.
*/
ostream &
Odyssey::print(ostream &os) const
{
    os << "Compiler state:" << endl << endl;

    os << "Compiler executable name:       " << _progname << endl;
    os << "Number of arguments left:       " << _argc << endl;
    os << "Files to compile:               ";
    list<SourceFile>::const_iterator f;
    for (f = _files.begin(); f != _files.end(); f++) {
	os << f->origname() << " ";
    }
    os << endl << endl;

    os << "Flags:" << endl;
    os << "Compile only (-c):              " << _compile_only << endl;
    os << "Debug (-g):                     " << _debug << endl;
    os << "Help (-h):                      " << _help << endl;
    os << "Keep intermediate files (-k):   " << _keep_inter << endl;
    os << "Verbose operation (-v):         " << _verbose << endl;
    os << "Version (-V):                   " << _version << endl;
    os << endl;

    os << "Directory paths and symbols" << endl;
    os << "Defines (-D):                   " << _defines << endl;
    os << "Include directories (-I):       " << _incdirs << endl;
    os << "Library directories (-L):       " << _libdirs << endl;
    os << "Libraries (-l):                 " << _libs << endl;
    os << endl;

    os << "Miscellaneous options"<< endl;
    os << "Stop at (-.):                   " << _stop_at << endl;
    os << "Name of executable (-o):        " << _output_name << endl;
    os << "Optimization level:             " << _optlevel << endl;

    return os;
}


ostream& operator<<(ostream& os, const Odyssey &compiler)
{
    return compiler.print(os);
}


/**
Parse the command line
*/
void
Odyssey::parseCmdLine()
{
    extern char *optarg;
    extern int optind;
    int c;

    if (_argc == 1) {
	_help = true;
    }

    while ((c = getopt(_argc, _argv, ".:cD:F:ghI:kL:l:no:O:P:v:V")) != -1) {
	switch (c) {
	    case '.': _stop_at = String(".") + optarg; break;
	    case 'c': _compile_only = true; break;
	    case 'D': _defines += String("-D") + optarg + " "; break;
	    case 'F': _dfa_list = optarg; break;
	    case 'g': _debug = true; break;
	    case 'h': _help = true; break;
	    case 'I': _incdirs += String("-I") + optarg + " "; break;
	    case 'k': _keep_inter = true; break;
	    case 'L': _libdirs += String("-L") + optarg + " "; break;
	    case 'l': _libs += String("-l") + optarg + " "; break;
	    case 'n': _modify_output_name = true; break;
	    case 'o': _output_name = optarg; break;
	    case 'O': _optlevel = ::atoi(optarg); break;
	    case 'P': _opt_list = optarg; break;
	    case 'v': _verbose = ::atoi(optarg); _help = (_verbose == 0); break;
	    case 'V': _version = true; break;
	}
    }

    for (; optind < _argc; optind++) {
	SourceFile *f = new SourceFile(_argv[optind]);
	_files.push_back(*f);
    }
}



/**
Shows a help message with all the options.
*/
void
Odyssey::showHelp()
{
    this->showVersion();

    const char *help = "\n"
    "usage: %s <options> <files>\n\n"

    "Options: (Default values shown in [])\n"
    "-.<ext>:   Stop compiling at extension <ext>. []\n"
    "-c:        Compile only, do not generate executable. [false]\n"
    "-D<name>:  Define C pre-processor symbol <name>. []\n"
    "-F<list>:  Comma-separated list of DFA passes to run. []\n"
    "-g:        Generate debugging information. [false]\n"
    "-h:        Show this help message and exit.\n"
    "-I<dir>:   Add <dir> to include directory path. []\n"
    "-k:        Keep intermediate files. [false]\n"
    "-L<dir>:   Add <dir> to library directory path. []\n"
    "-l<lib>:   Link executable with library file <lib>. []\n"
    "-n:        Modify output name with optimizations done. [false]\n"
    "-o<name>:  Name of executable. [a.out]\n"
    "-O<num>:   Set sequential optimization level to <num>. [0]\n"
    "-P<list>:  Comma-separated list of OPT passes to run. To display\n"
    "           statistics about each pass, append '=num' to the pass name\n"
    "           (0 <= num < 1). Known passes are:\n"
    "\n"
    "           licm           Lock Independent Code Motion\n"
    "           rlicm          Relaxed LICM\n"
    "           mbl            Mutex Body Localization\n"
    "           dce            Dead Code Elimination\n"
    "           gcp            Global Constant Propagation\n"
    "           cse            Common Subexpression Elimination\n"
    "           copy_prop      Copy Propagation\n"
    "           forward_prop   Forward Propagation\n"
    "\n"
    "-v<num>:   Set verbosity level to 'num' [0]\n"
    "-V:        Show compiler version and exit.\n";

    printf(help, _progname);
}


/**
Shows the compiler version.
*/
void
Odyssey::showVersion()
{
    extern char *prog_ver_string, *prog_who_string, *prog_suif_string;

    cout << "Odyssey compiler v" << prog_ver_string 
	 << " -- Based on SUIF v" << prog_suif_string << endl;
    cout << prog_who_string << endl;
}
