/**
$Id: SourceFile.cc,v 1.6 1999/02/09 18:03:42 diego Exp $
*/
#include <stdlib.h>
#include <unistd.h>
#include <libgen.h>

#include <iostream.h>
#include <list>

#include <suif.h>
#include <par.h>
#include <ccfg.h>
#include <cssame.h>
#include <str.h>

#include "compiler.h"

using namespace std;

/**
Constructor.
*/
SourceFile::SourceFile(char *name)
{
    _name = name;
    _basename = ::basename(name);
    _dirname = ::dirname(name);

    /* Separate the extension from the file name */
    String *array;
    int nf = _basename.Split(array, '.');
    if (nf == 0) {
	/* No '.' in the file, extension is empty */
	_ext = "";
    } else {
	/* Found at least one '.' in the file, set the extension to the
	 * very last field found by String::Split(). Then set the basename
	 * to all the other fields left in the array.
	 */
	_ext = String(".") + array[nf - 1];
	_basename = "";
	for (int i = 0; i < nf - 1; i++) {
	    _basename += array[i];
	}
    }

    /* Initialize my state based on the initial file extension */
    if (_ext == ".c") {
	_state = new SourceState(this);
    } else if (_ext == ".spd") {
	_state = new spdState(this);
    } else if (_ext == ".spx") {
	_state = new spxState(this);
    } else if (_ext == ".od1") {
	_state = new dfaState(this);
    } else if (_ext == ".odc") {
	_state = new odcState(this);
    } else if (_ext == ".pa1") {
	_state = new pa1State(this);
    } else if (_ext == ".pa2") {
	_state = new pa2State(this);
    } else if (_ext == ".par") {
	_state = new parState(this);
    } else if (_ext == ".o") {
	_state = new oState(this);
    } else {
	throw _E_UnknownExtension(__FILE__, __LINE__, NULL, _ext);
    }

    if (OCC->verbose()) { cout << *this; }
}


/**
Destructor. Get rid of the file's state.
*/
SourceFile::~SourceFile()
{
    delete _state;
}


/**
Modifies the base name of the file by appending a string with the names of
all the optimization passes performed on the file.
*/
void
SourceFile::modify_basename()
{ 
    _basename += OCC->output_name_suffix();
}


/**
Output the object to the given stream
*/
ostream&
SourceFile::print(ostream &os) const
{
    os << "Source file: " << _name << " -> " << _dirname << " / " << _basename
       << " (" << this->ext() << ")" << endl;

    return os;
}


ostream&
operator<<(ostream &os, const SourceFile& file)
{
    return file.print(os);
}


/** 
Determine whether this file has been fully processed.
*/
bool
SourceFile::done() const
{ 
    return _state->done();
}

/**
Cause a state transition. Tell the current state of the file to progress to
the next state. Re-set the state to be the new state returned by the state
transition method.
*/
void
SourceFile::transform()
{
    if (_state->done()) {
	return;
    }

    FileState *newState = _state->change();
    delete _state;
    _state = newState;
}

/**
Return the file extension associated with the current file state.
*/
const String&
SourceFile::ext() const
{ 
    return _state->ext();
}


/**
Make a deep copy of the given file.
*/
void
SourceFile::copyFrom(const SourceFile& orig)
{
    _name = orig.origname();
    _dirname = orig.dirname();
    _basename = orig.basename();
    _ext = orig.origext();

    /* The state from the original file will be pointing at the
     * original file. Re-set it so that it points here instead.
     */
    _state = orig.state();
    if (_state) {
	_state->_file = this;
    }
}
