/**
$Id: dfaState.cc,v 1.5 1999/03/15 21:28:10 diego Exp $
*/
#include <list>

#include <suif.h>
#include <cg.h>
#include <par.h>
#include <ccfg.h>
#include <cssame.h>
#include <odc-util.h>
#include <str.h>

#include "compiler.h"

using namespace std;

/**
Private helpers to build CCFGs, find references, conflicts and thread
objects on the procedures in the program.
*/
static void _mark_pure_functions(global_symtab *the_symtab);
static void _unmark_pure_functions(global_symtab *the_symtab);

static void _build_threads_for_proc(cg_node *node, void *x);
static void _remove_threads_from_proc(cg_node *node, void *x);

static void _find_refs_in_proc(cg_node *node, void *x);
static void _remove_refs_from_proc(cg_node *node, void *x);

static void _find_confs_in_proc(cg_node *node, void *x);
static void _remove_confs_from_proc(cg_node *node, void *x);

static void _build_ccfg_for_proc(cg_node *node, void *x);
static void _remove_ccfg_for_proc(cg_node *node, void *x);



/*---------------------------------------------------------------------------
				Internal states
---------------------------------------------------------------------------*/
/** 
Set the SUIF's file set entry to NULL. This should be set to the proper
file_set_entry by Odyssey::transform() before doing the state change.
*/
dfaState::dfaState(SourceFile *file) : InternalState(file)
{
    _fse = NULL;
}


/** 
Perform the DFA and OPT passes specified in the command line. Because of
SUIF's file processing semantics, we must do all the analyses and
optimizations on one file at a time. This will get in the way if we ever
want to do cross-file IPA. We'll deal with this problem when the time
comes.
*/
FileState *
dfaState::change()
{
    D_SELFTEST_HEADER(10, "dfaState::change");

    assert(_fse);

    /* Build the call graph for the program */
    _call_graph = new cg;

    if (OCC->verbose() > 1) {
	cout << "\nCall graph for file '" << _file->name() << "'\n";
	_call_graph->print(); fflush(stdout);
	cout << endl;
    }

    /* Read all the procedures into memory. */
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	proc_sym *psym = _call_graph->node(i)->suif_proc();
	psym->read_proc();
    }

    /* Run all the requested DFA and OPT passes */
    if (!OCC->dfaPasses().empty() || !OCC->optPasses().empty()) {
	/* Mark pure functions in the global symbol table */
	::_mark_pure_functions(_fse->parent()->globals());

	this->buildDFAInfo();
	this->runDFA();
	this->runOPT();
	this->removeDFAInfo();

	/* Remove pure function marks from the global symbol table */
	::_unmark_pure_functions(_fse->parent()->globals());
    }

    /* Write the procedures out to the .odc file */
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	proc_sym *psym = _call_graph->node(i)->suif_proc();
	psym->write_proc(_fse);
	psym->flush_proc();
    }

    OCC->addTmpFile(_file->name());

    /** If the user wants the output name modified, append the string of
     * optimization passes to the basename of the file. This will make it
     * easier to tell which transformations were done on the file.
     */
    if (OCC->modify_output_name()) { _file->modify_basename(); }

    D_SELFTEST_FOOTER(10);

    return(new odcState(_file));
}



/**
Run the DFA passes on all the procedures in the program. This method
assumes that dataflow information has been collected already.
*/
void
dfaState::runDFA()
{
    list<dfa *>::iterator dfaIter;
    list<dfa *>& p = OCC->dfaPasses();

    if (p.empty()) {
	return;
    }

    /* Now run all the passes on each procedure */
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	cg_node *node = _call_graph->node(i);
	tree_proc *tp = node->suif_proc()->block();

	if (node != _call_graph->main_node() && node->preds()->is_empty()) {
	    warning_line(tp, "Function %s may not be invoked.\n", 
		    tp->proc()->name());
	}


	for (dfaIter = p.begin(); dfaIter != p.end(); dfaIter++) {
	    dfa *pass = *dfaIter;

	    if (OCC->verbose()) { 
		cout << "[[" << setfill('-') << setw(78) << "-" << setfill(' ') 
		    << setw(0) << endl;
		cout << "Running data flow pass '" << pass->id() << "'"
		    << " on procedure '" << tp->proc()->name() << "'\n";
		cout << setfill('-') << setw(80) << "-" << setfill(' ') 
		    << setw(0) << endl;
	    }

	    pass->run(tp);

	    if (OCC->verbose()) {
		cout << "]]" << setfill('-') << setw(78) << "-" << setfill(' ') 
		    << setw(0) << endl << endl << endl;
	    }
	}
    }
}



/**
Run the OPT passes on the given procedure. This method assumes that data
flow information has been collected already.
*/
void
dfaState::runOPT()
{
    list<opt *>::iterator optIter;
    list<opt *>& o = OCC->optPasses();

    if (o.empty()) {
	return;
    }

    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	cg_node *node = _call_graph->node(i);
	tree_proc *tp = node->suif_proc()->block();

	if (node != _call_graph->main_node() && node->preds()->is_empty()) {
	    warning_line(tp, "Function %s may not be invoked.\n", 
		    tp->proc()->name());
	}

	for (optIter = o.begin(); optIter != o.end(); optIter++) {
	    opt *pass = *optIter;

	    if (OCC->verbose()) {
		cout << "[[" << setfill('-') << setw(78) << "-" << setfill(' ') 
		    << setw(0) << endl;
		cout << tp->proc()->name() << ": Running optimization pass '" 
		    << pass->id() << "'\n";
		cout << setfill('-') << setw(80) << "-" << setfill(' ') 
		    << setw(0) << endl;
	    }

	    bool made_progress = pass->run(tp);

	    if (OCC->verbose()) {
		cout << endl;
		cout << tp->proc()->name() << ": Procedure was" <<
		    ((made_progress) ? "" : " NOT") << " transformed\n";
		cout << "]]" << setfill('-') << setw(78) << "-" << setfill(' ') 
		    << setw(0) << endl << endl << endl;
	    }

	    /* Remove all the dataflow information. 
	     * 
	     * [BUG] We should only remove those parts that have changed, but
	     * this will do for now.
	     */
	    if (made_progress) {
		this->removeDFAInfo();
		this->buildDFAInfo();
	    }
	}
    }
}


void
dfaState::buildDFAInfo()
{
    this->build_threads();
    this->find_refs();
    this->find_confs();
    this->build_ccfgs();
}


void
dfaState::removeDFAInfo()
{
    this->remove_ccfgs();
    /*
    this->remove_confs();
    this->remove_refs();
    this->remove_threads();
    */
}


/*---------------------------------------------------------------------------
       Local functions to build thread and tree_par objects for all 
		      the procedures in the program.
---------------------------------------------------------------------------*/
void
dfaState::build_threads()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_build_threads_for_proc(_call_graph->node(i), (void *)this);
    }
}

static void _build_threads_for_tn(tree_node *tn, void *x);

static void
_build_threads_for_proc(cg_node *node, void *x)
{
    D_STACK(_build_threads_for_proc);
    D_SELFTEST_HEADER(20, "_build_threads_for_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    D_SELFTEST(20) {
	cout << "Looking for threads in procedure: '" << psym->name()
	    << "'.\n";
    }

    if (!psym->is_in_memory()) { psym->read_proc(); }
    tree_proc *tp = psym->block();

    pair<tree_proc *, dfaState *> arg = make_pair(tp, state);
    tp->body()->map(_build_threads_for_tn, (void *)&arg);

    D_SELFTEST(20) {
	map<p_tree_proc, vector_tree_par *>::iterator iter;
	iter = state->tree_par_map().find(tp);
	if (iter == state->tree_par_map().end()) {
	    cout << "The procedure does not contain any parallel structures\n";
	} else {
	    cout << "Parallel structures found:\n";
	    vector_tree_par *tps = (*iter).second;
	    vector_tree_par::iterator i;
	    for (i = tps->begin(); i != tps->end(); i++) {
		(*i)->print();
		cout << endl;
	    }
	}
    }

    D_SELFTEST_FOOTER(20);
}


static void
_build_threads_for_tn(tree_node *tn, void *x)
{
    pair<tree_proc *, dfaState *> *arg = (pair<tree_proc *, dfaState *> *)x;
    tree_proc *tp = arg->first;
    dfaState *state = arg->second;

    D_SELFTEST(22) {
	cout << "Analyzing tree_node:\n";
	tn->print(); fflush(stdout);
	cout << endl;
    }

    if (is_par(tn)) {
	tree_par *par;

	if (is_cobegin(tn)) {
	    D_SELFTEST(22) { cout << "It is a cobegin statement\n"; }
	    par = new tree_cobegin(tn);	// [MEMORY LEAK]
	} else if (is_parloop(tn)) {
	    D_SELFTEST(22) { cout << "It is a parloop statement\n"; }
	    par = new tree_parloop(tn);	// [MEMORY LEAK]
	} else {
	    error_line(-1, tn, ("Fatal. Unknown parallel structure.\n"));
	}

	if (state->tree_par_map().find(tp) == state->tree_par_map().end()) {
	    /* Create vector on first use */
	    state->tree_par_map()[tp] = new vector_tree_par;
	}

	state->tree_par_map()[tp]->push_back(par);
    }
}


void
dfaState::remove_threads()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_remove_threads_from_proc(_call_graph->node(i), (void *)this);
    }
}


static void
_remove_threads_from_proc(cg_node *node, void *x)
{
    D_STACK(_remove_threads_from_proc);
    D_SELFTEST_HEADER(710, "_remove_threads_from_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    assert_msg(psym->is_in_memory(), 
	    ("Fatal. Procedure '%s' is not loaded.\n", psym->name()));
    tree_proc *tp = psym->block();


    map<p_tree_proc, vector_tree_par *>::iterator tree_par_iter; 
    tree_par_iter = state->tree_par_map().find(tp);
    if (tree_par_iter == state->tree_par_map().end()) {
	return;
    }
   
    D_SELFTEST(710) {
	cout << "Deleting thread objects for procedure '" << psym->name()
	    << "'\n";
    }

    vector_tree_par *tree_pars = (*(tree_par_iter)).second;
    vector_tree_par::iterator i;
    for (i = tree_pars->begin(); i != tree_pars->end(); i++) {
	delete (*i).ptr();
    }
    delete tree_pars;
    state->tree_par_map().erase(tree_par_iter);

    D_SELFTEST_FOOTER(710);

}



/*---------------------------------------------------------------------------
    Mapped function to find references in all the procedures of the program
---------------------------------------------------------------------------*/
void
dfaState::find_refs()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_find_refs_in_proc(_call_graph->node(i), (void *)this);
    }
}

static void
_find_refs_in_proc(cg_node *node, void *x)
{
    D_STACK(_find_refs_in_proc);
    D_SELFTEST_HEADER(30, "_find_refs_in_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    if (!psym->is_in_memory()) { psym->read_proc(); }
    tree_proc *tp = psym->block();

    D_SELFTEST(30) {
	cout << "Looking for references in function '" << psym->name() << "'\n";
    }

    set_varref *refs = ::find_memory_refs(tp);
    state->varref_map()[tp] = refs;

    D_SELFTEST(30) {
	cout << "References found:\n";
	copy(refs->begin(), refs->end(),ostream_iterator<p_varref>(cout, "\n"));
	cout << endl;
    }

    D_SELFTEST_FOOTER(30);
}

void
dfaState::remove_refs()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_remove_refs_from_proc(_call_graph->node(i), (void *)this);
    }
}


static void
_remove_refs_from_proc(cg_node *node, void *x)
{
    D_STACK(_remove_refs_from_proc);
    D_SELFTEST_HEADER(720, "_remove_refs_from_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    assert_msg(psym->is_in_memory(), 
	    ("Fatal. Procedure '%s' is not loaded.\n", psym->name()));
    tree_proc *tp = psym->block();


    map<p_tree_proc, set_varref *>::iterator ref_iter; 
    ref_iter = state->varref_map().find(tp);
    if (ref_iter == state->varref_map().end()) {
	return;
    }
   
    D_SELFTEST(720) {
	cout << "Deleting memory references for procedure '" << psym->name()
	    << "'\n";
    }

    set_varref *refs = (*(ref_iter)).second;
    for (set_varref::iterator i = refs->begin(); i != refs->end(); i++) {
	delete (*i).ptr();
    }
    delete refs;
    state->varref_map().erase(ref_iter);

    D_SELFTEST_FOOTER(720);
}



/*---------------------------------------------------------------------------
     Mapped function to find conflicts in all the procedures of the program
---------------------------------------------------------------------------*/
void
dfaState::find_confs()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_find_confs_in_proc(_call_graph->node(i), (void *)this);
    }
}

static void
_find_confs_in_proc(cg_node *node, void *x)
{
    D_STACK(_find_confs_in_proc);
    D_SELFTEST_HEADER(40, "_find_confs_in_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    if (!psym->is_in_memory()) { psym->read_proc(); }
    tree_proc *tp = psym->block();

    D_SELFTEST(40) {
	cout << "Looking for conflicts in function '" << psym->name() << "'\n";
    }

    list_conflict *confs = ::find_memory_conflicts(tp);
    state->conflict_map()[tp] = confs;

    D_SELFTEST(40) {
	cout << "Conflicts found:\n";
	copy(confs->begin(),confs->end(),ostream_iterator<conflict>(cout,"\n"));
	cout << endl;
    }

    D_SELFTEST_FOOTER(40);
}


void
dfaState::remove_confs()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_remove_confs_from_proc(_call_graph->node(i), (void *)this);
    }
}


static void
_remove_confs_from_proc(cg_node *node, void *x)
{
    D_STACK(_remove_confs_from_proc);
    D_SELFTEST_HEADER(730, "_remove_confs_from_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    assert_msg(psym->is_in_memory(), 
	    ("Fatal. Procedure '%s' is not loaded.\n", psym->name()));
    tree_proc *tp = psym->block();


    map<p_tree_proc, list_conflict *>::iterator conf_iter; 
    conf_iter = state->conflict_map().find(tp);
    if (conf_iter == state->conflict_map().end()) {
	return;
    }
   
    D_SELFTEST(730) {
	cout << "Deleting memory conflicts for procedure '" << psym->name()
	    << "'\n";
    }

    list_conflict *confs = (*(conf_iter)).second;
    delete confs;
    state->conflict_map().erase(conf_iter);

    D_SELFTEST_FOOTER(730);
}




/*---------------------------------------------------------------------------
     Mapped function to build the CCFG for each procedure in the call graph
---------------------------------------------------------------------------*/
void
dfaState::build_ccfgs()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_build_ccfg_for_proc(_call_graph->node(i), (void *)this);
    }
}

static void
_build_ccfg_for_proc(cg_node *node, void *x)
{
    D_STACK(_build_ccfg_for_proc);
    D_SELFTEST_HEADER(50, "_build_ccfg_for_proc");

    dfaState *state = (dfaState *)x;

    proc_sym *psym = node->suif_proc();
    if (!psym->is_in_memory()) { psym->read_proc(); }
    tree_proc *tp = psym->block();

    D_SELFTEST(50) { 
	cout << "Building CCFG for function '" << psym->name() << "'\n";
    }

    assert_msg(state->varref_map().find(tp) != state->varref_map().end(),
	    ("Can't build CCFG for procedure '%s'. Find refs first!\n",
	     psym->name()));

    assert_msg(state->conflict_map().find(tp) != state->conflict_map().end(),
	    ("Can't build CCFG for procedure '%s'. Find conflicts first!\n",
	     psym->name()));

    p_ccfg graph = new ccfg(tp);

    /* Clean up the graph and find dominator information used by the CSSAME
     * form when jumping from graph to graph.
     */
    graph->remove_unreachable_blocks();
    graph->find_dominators();
    graph->find_postdominators();
    graph->find_df();
    graph->findRefs();

    state->ccfg_map()[tp] = graph;

    D_SELFTEST(51) {
	cout << "CCFG for function '" << psym->name() << "':\n";
	graph->print();
    }

    D_SELFTEST_FOOTER(50);
}



void
dfaState::remove_ccfgs()
{
    for (unsigned i = 0; i < _call_graph->num_nodes(); i++) {
	_remove_ccfg_for_proc(_call_graph->node(i), (void *)this);
    }
}

static void
_remove_ccfg_for_proc(cg_node *node, void *x)
{
    D_STACK(_remove_ccfg_for_proc);
    D_SELFTEST_HEADER(740, "_remove_ccfg_for_proc");

    dfaState *state = (dfaState *)x;
    proc_sym *psym = node->suif_proc();

    assert_msg(psym->is_in_memory(), 
	    ("Fatal. Procedure '%s' is not loaded.\n", psym->name()));

    tree_proc *tp = psym->block();

    map<p_tree_proc, p_ccfg>::iterator ccfg_iter; 
    ccfg_iter = state->ccfg_map().find(tp);
    if (ccfg_iter == state->ccfg_map().end()) {
	return;
    }
   
    D_SELFTEST(740) {
	cout << "Deleting CCFG for procedure '" << psym->name() << "'\n";
    }

    p_ccfg ccfg = (*(ccfg_iter)).second;
    delete ccfg.ptr();
    state->ccfg_map().erase(ccfg_iter);

    D_SELFTEST_FOOTER(740);
}


/**
Local function to mark pure functions. A pure function is guaranteed not to
reference any globals or have side effects (ie, its value is purely
defined by its arguments). Adapted from the 'c_lib_info' program that comes
with SUIF1.
*/
static void 
_mark_pure_functions(global_symtab *the_symtab)
{
    D_STACK(_mark_pure_functions);
    D_SELFTEST_HEADER(80, "_mark_pure_functions");

    static char *pure_function_names[] = {
	/* Known synchronization functions. */
	"_set", "_wait", "suif_lock", "suif_unlock", "suif_global_barrier",
	"suif_get_my_id",

	/* Known standard functions */
	"printf", "fflush", "exit", "abort", "fprintf", "putc", "fputc",
	"putchar", "atoi", "atof", "atol", "atod", "sigpause",

	/* These are all from the standard math library, as described
	 * in ANSI/ISO 9899-1990, section 7.5.
	 */
	"acos", "acosf", "asin", "asinf", "atan", "atan2f", "atanf",
	"ceil", "ceilf", "cos", "cosf", "drand48", "erand48", "fabs", "fabsf",
	"floor", "floorf", "fmod", "fmodf", "ldexp", "log", "log10", "log10f",
	"logf", "lrand48", "modf", "pow", "powf", "rand", "random", "sin",
	"sinf", "sqrt", "sqrtf", "srand", "srand48", "srandom", "tan",
	NULL
    };

    sym_node_list_iter sym_iter(the_symtab->symbols());
    while (!sym_iter.is_empty()) {
	sym_node *this_symbol = sym_iter.step();
	D_SELFTEST(85) {
	    cout << "-> Testing if symbol '" << this_symbol->name()
		<< "' is a pure function name.\n";
	}
	if (this_symbol->is_proc()) {
	    char **follow = &(pure_function_names[0]);
	    while (*follow != NULL) {
		if (strcmp(this_symbol->name(), *follow) == 0) {
		    this_symbol->append_annote(k_pure_function);
		    D_SELFTEST(80) {
			cout << "-> Found pure function '";
			this_symbol->print(); cout << "'.\n";
		    }
		    break;
		}
		++follow;
	    }
	}
    }

    D_SELFTEST_FOOTER(80);
}


/**
Local function to remove pure function marks.
*/
static void 
_unmark_pure_functions(global_symtab *the_symtab)
{
    sym_node_list_iter sym_iter(the_symtab->symbols());
    while (!sym_iter.is_empty()) {
	sym_node *this_symbol = sym_iter.step();
	if (this_symbol->is_proc()) {
	    this_symbol->get_annote(k_pure_function);
	}
    }
}
