/**
$Id: CompilerPass.h,v 2.9 1999/03/05 14:50:44 diego Exp $
*/
#ifndef __COMPILER_PASS_H
#define __COMPILER_PASS_H

#include <list>
#include <set>
#include <map>

#include <iostream.h>

#include <suif.h>
#include <builder.h>
#include <useful.h>
#include <cssame.h>
#include <d_lib.h>
#include <str.h>
#include <wrappers.h>

using namespace std;

/* Collections used */
typedef list<p_tree_instr> list_tree_instr;

/* Forward declarations needed to declare pointer wrappers */
class CompilerPass;
class dfa;
class cssamePass;
class dce;
class gcp;
class licm;
class rlicm;
class mbl;

typedef pWrapper<CompilerPass *> p_CompilerPass;
typedef pWrapper<dfa *> p_dfa;
typedef pWrapper<cssamePass *> p_cssamePass;
typedef pWrapper<dce *> p_dce;
typedef pWrapper<gcp *> p_gcp;
typedef pWrapper<licm *> p_licm;
typedef pWrapper<rlicm *> p_rlicm;
typedef pWrapper<mbl *> p_mbl;


/**
Abstract super class for compiler passes. There are two main kinds of
compiler passes: data flow analysis (dfa) passes that compute data flow
properties of the program and optimization (opt) passes that apply
optimizing transformations to the program.
*/
class CompilerPass { D_CLASS(CompilerPass)
public:
    CompilerPass(unsigned verbose = 0) : _verbose(verbose) {}
    virtual bool run(p_tree_proc tp) = 0;
    virtual const char *id() const { return "CompilerPass"; }

protected:
    unsigned _verbose;
};



/*---------------------------------------------------------------------------
				Analysis Passes
---------------------------------------------------------------------------*/
class dfa : public CompilerPass { D_CLASS(dfa)
public:
    dfa(unsigned verbose = 0) : CompilerPass(verbose) {}
    virtual bool run(p_tree_proc tp) = 0;
    virtual const char *id() const { return "dfa"; }
};


class cssamePass : public dfa { D_CLASS(cssamePass)
public:
    cssamePass(unsigned verbose = 0) : dfa(verbose) {}
    virtual bool run(p_tree_proc tp);
    virtual const char *id() const { return "cssame"; }
};



/*---------------------------------------------------------------------------
			      Optimization Passes
---------------------------------------------------------------------------*/
/**
Base optimization class. Every sub-class must implement the protected
method opt::execute() which when invoked should perform the corresponding
optimization. 

The compiler calls the public method opt::run(tp) which sets up everything
needed by the optimization (building the CSSAME form, etc), calls the main
optimization algorithm (opt::execute()) and cleans up afterward.

It is not necessary to overload the public opt::run(tp) method, the three
phases opt::setup(), opt::execute() and opt::cleanup() can be overloaded if a
subclass needs to do something different.
*/
class opt : public CompilerPass { D_CLASS(opt)
public:
    opt(bool showStats, unsigned verbose = 0) : 
	CompilerPass(verbose), _ssa(0), _tp(0), _showStats(showStats) {}

    bool showStats() const	{ return _showStats; }
    p_tree_proc proc() const	{ return _tp; }
    p_cssame ssa() const	{ return _ssa; }

    virtual bool run(p_tree_proc tp) { 
	this->setup(tp);
	bool made_progress = this->execute();
	this->printStats(cout);
	this->cleanup();

	return made_progress;
    }

    virtual const char *id() const { return "opt"; }
    virtual void printStats(ostream& s = cout) const = 0;

protected:
    p_cssame _ssa;
    p_tree_proc _tp;
    bool _showStats;

    virtual void setup(p_tree_proc tp);
    virtual bool execute() = 0;
    virtual void cleanup();
};



/*---------------------------------------------------------------------------
		 Classes for collecting optimization statistics
---------------------------------------------------------------------------*/
/**
Abstract super-class. An optimization statistic object holds information
about the transformations done to the program by the associated
optimization object.

The hierarchy for statistics classes is similar to the hierarchy for
optimization passes. Each statistics object contains a reference to its
associated optimization object.
*/
class OptStat {
public:
    OptStat(const opt& pass, bool active = false) : 
	_pass(pass), _active(active) {}

    /* Show the collected statistics on the given stream */
    virtual void print(ostream& s = cout) const {
	if (!_active) { return; }
	s << endl << "No statistics collected by '" << id() << "'\n\n";
    }

    /* Return an identification string */
    virtual const char *id() const = 0;

    /* Enable/disable stats collection */
    void enable() { _active = true; }
    void disable() { _active = false; }
    bool active() const { return _active; }

    /* Access the associated optimization pass */
    const opt& pass() const { return _pass; }

protected:
    const opt& _pass;
    bool _active;
};



/*---------------------------------------------------------------------------
	   Classes for optimizations involving mutex synchronization
---------------------------------------------------------------------------*/
/**
Base class. Collects basic information about locks and critical sections
used in the program.
*/
class MutexStat : public OptStat {
public:
    MutexStat(const opt& pass, bool active = false) : OptStat(pass, active) {}
    virtual void print(ostream& s = cout) const;
    virtual const char *id() const { return "MutexStat"; }
};

#endif	// __COMPILER_PASS_H
