/**
$Id: dce.cc,v 2.9 1999/03/05 14:50:45 diego Exp $
*/
#include "dce.h"

char *k_live;

static void computePrelive(tree_node *tn, void *x);


/**
Parallel Dead Code Elimination algorithm. Based on the sequential
algorithm from 'Efficiently Computing Static Single Assignment Form and
the Control Dependence Graph', R. Cytron et. al., ACM TOPLAS v13, No.4,
Oct '91, pp 451-490.
*/
bool
dce::execute()
{
    D_SELFTEST_HEADER(1, "dce::execute");

    p_ccfg graph = _ssa->graph();
    list_tree_instr worklist;
    list_tree_instr prelive;

    /* Incorporate the k_live annotation into SUIF's annotation manager */
    k_live = lexicon->enter("live")->sp;

    _ssa->computeReachingDefs();
    graph->find_reverse_df();
    graph->tproc()->map(computePrelive, (void *)&prelive);

    list_tree_instr::iterator stmt_iter = prelive.begin();
    for (; stmt_iter != prelive.end(); stmt_iter++) {
	p_tree_instr stmt = *stmt_iter;
	stmt->instr()->append_annote(k_live, (void *)stmt.ptr());
	worklist.push_back(stmt);
    }

    while (!worklist.empty()) {
	p_tree_instr stmt = worklist.back(); worklist.pop_back();

	 D_SELFTEST(2) {
	    cout << "\n\nProcessing statement\n";
	    stmt->print();
	    cout << "\nChecking reaching definitions\n";
	}

	/* If this statement is live, then any reaching definitions to this
	 * statement should also be live and added to the work list.
	 */
	set_vardef *rdefs = get_vardefs(stmt.ptr());
	set_vardef::iterator rdefs_iter = rdefs->begin();
	for (; rdefs_iter != rdefs->end(); rdefs_iter++) {
	    instruction *instr = (*rdefs_iter)->instr();
	    if (!instr->peek_annote(k_live)) {
		instr->append_annote(k_live, (void *)stmt.ptr());
		worklist.push_back(instr->parent());

		D_SELFTEST(2) {
		    cout << "Marking live reaching definition\n";
		    instr->parent()->print();
		}
	    }
	}

	/* If this statement is live, then any control dependence
	 * predecessor for this statement should also be marked live. The
	 * control dependence relation is computed as the dominance
	 * frontier for the reverse flow graph. A node Y is control
	 * dependent on another node X iff X is in RDF(Y).
	 *
	 * We traverse all the nodes in our reverse dominance frontier, if
	 * the last statement of each node is not live, we mark it live.
	 */
	D_SELFTEST(2) {
	    cout << "\nChecking control dependence predecessors\n";
	}
	p_ccfg_node owner = get_node(stmt);
	assert(owner);
	bit_set *rdf = graph->reverse_dom_frontier(owner);
	bit_set_iter rdf_iter(rdf);
	while (!rdf_iter.is_empty()) {
	    int cdp = rdf_iter.step();
	    p_ccfg_node cdp_node = graph->node(cdp);
	    p_tree_instr last = cdp_node->last_instr();

	    D_SELFTEST(2) {
		cout << "Checking last statement of node " << cdp << endl;
		last->print();
	    }

	    if (!last->instr()->peek_annote(k_live)) {
		last->instr()->append_annote(k_live, (void *)last.ptr());
		worklist.push_back(last);

		D_SELFTEST(2) {
		    cout << "Making previous statement live\n";
		}
	    }
	}
    }

    D_SELFTEST_FOOTER(1);

    return true;	/* [HACK] Should only return true if it really made
			   any progress. */
}



/**
This function is mapped over all the tree nodes in the program. It marks
live instructions assumed to affect program output, such as an I/O
statement, an assignment to a reference parameter or a call to a routine
that may have side effects (check Cytron's paper for details).
*/
void
computePrelive(tree_node *tn, void *x)
{
    D_STACK(computePrelive);
    D_SELFTEST_HEADER(1, "computePrelive");

    list_tree_instr *prelive = (list_tree_instr *)x;
    assert(prelive);

    if (!tn->is_instr()) {
	return;
    }

    instruction *instr = ((tree_instr *)tn)->instr();
    if_ops opcode = instr->opcode();

    /* Check if the instruction is a function call or a non-void return */
    switch (opcode) {
	case io_cal:
	    prelive->push_back(instr->parent());

	    D_SELFTEST(2) {
		cout << "Added function call to prelive set\n";
		instr->print();
		cout << endl;
	    }
	    break;

	case io_ret:
	    if (!((in_rrr *)instr)->src1_op().is_null()) {
		prelive->push_back(instr->parent());

		D_SELFTEST(2) {
		    cout << "Added non-void return statement to prelive set\n";
		    instr->print();
		    cout << endl;
		}
	    }
	    break;
    }

    /* Check if the instruction defines a variable outside the current
     * scope (ie, it defines a global variable or a reference parameter).
     */
    set_vardef *defs = get_vardefs(instr);
    if (defs && !defs->empty()) {
	set_vardef::iterator defs_iter = defs->begin();
	for (; defs_iter != defs->end(); defs_iter++) {
	    p_vardef def = *defs_iter;

	    /* Check if the variable is declared outside the scope of the
	     * procedure containing the instruction. We first walk up the
	     * ownership tree looking for the symbol table for the
	     * procedure that contains the instruction.
	     *
	     * Then we lookup the variable in the symbol table hierarchy.
	     * If the variable is found in an ancestor symbol table, we
	     * mark the statement as live.
	     */
	    base_symtab *symtab;
	    symtab = instr->parent()->proc()->block()->symtab()->parent();
	    var_sym *var = symtab->lookup_var(def->var()->name());
	    if (var) {
		prelive->push_back(instr->parent());

		D_SELFTEST(2) {
		    cout << "Added definition of global variable " << 
			var->name() << " to prelive set\n";
		    instr->print();
		    cout << endl;
		}
	    }
	}
    }
}

