/**
$Id: gcp.cc,v 2.9 1999/03/05 14:50:46 diego Exp $
*/

#include "gcp.h"

/**
Constructor for GCP objects.
*/
gcp::gcp(bool showStats, unsigned verbose)
     : opt(showStats, verbose), _stat(gcp::stats(*this, showStats))
{
}


/**
Performs Global Constant Propagation and constant folding on the given
CSSAME object. Adapted from the SUIF source code: 
SUIF/src/basesuif/porky/constants.cc and SUIF/src/basesuif/porky/fold.cc
*/
bool
gcp::execute()
{
    D_SELFTEST_HEADER(510, "gcp::execute");

    D_SELFTEST(510) { cout << "Pass 1: Compute reaching definitions\n"; }
    _ssa->computeReachingDefs();

    _made_progress = false;

    D_SELFTEST(510) { cout << "Pass 2: Propagate constants\n"; }
    this->propagate_constants(_ssa->graph()->tproc());

    D_SELFTEST(510) { cout << "Pass 3: Fold constants\n"; }
    this->fold_all_constants(_ssa->graph()->tproc());

    D_SELFTEST_FOOTER(510);

    return _made_progress;
}



/*---------------------------------------------------------------------------
		       Methods for statistics collection
---------------------------------------------------------------------------*/
void
gcp::printStats(ostream& s) const
{
    _stat.print(s);
}



void
gcp::stats::add_constant_replacement_for(p_var_sym var)
{
    if (!_active) { return; }

    if (_constant_vars.find(var) == _constant_vars.end()) {
	_constant_vars[var] = 1;
    } else {
	_constant_vars[var]++;
    }
}

void
gcp::stats::add_folded_expr(unsigned lineno)
{
    if (!_active) { return; }

    _folded_exprs.insert(lineno);
    _num_folded++;
}


void
gcp::stats::print(ostream &s) const
{
    if (!_active) { return; }

    s << "Results of constant propagation for procedure '" <<
	_pass.ssa()->graph()->tproc()->proc()->name() << "'" << endl << endl;

    s << setw(25) << setiosflags(ios::left) << "Variable"
      << setw(25) << setiosflags(ios::left) << "# of uses"
      << setw(25) << setiosflags(ios::left) << "# of const replacements"
      << endl;
    s << setw(80) << setfill('-') << "-" << setw(0) << setfill(' ') << endl;

    s << endl;

    map<p_var_sym, unsigned>::const_iterator i;
    for (i = _constant_vars.begin(); i != _constant_vars.end(); i++) {
	pair<p_var_sym, unsigned> p = *i;
	p_var_sym var = p.first;
	unsigned num_const_repl = p.second;
	unsigned num_uses = get_varuses(var.ptr())->size();
	double percent = ((double)num_const_repl / (double)num_uses) * 100.0;

	s << setw(25) << setiosflags(ios::left) << var->name()
	  << setw(25) << setiosflags(ios::left) << num_uses
	  << setw(1) << setiosflags(ios::left) << num_const_repl
	  << setiosflags(ios::fixed|ios::left) << setprecision(0) 
	  << " (" << percent << "%)" << endl;
    }

    s << endl << endl;

    s << "Folding simplified a total of " << _num_folded << " expressions.\n";
    s << endl;

    s << "Line numbers with folded expressions:\n";
    set<unsigned>::iterator j;
    for (j = _folded_exprs.begin(); j != _folded_exprs.end(); j++) {
	s << *j << " ";
    }
    s << endl;
}
