/**
$Id: licm.cc,v 2.26 1999/03/13 19:12:53 diego Exp $

Lock Independent Code Motion optimization.
*/
#include <algo.h>

#include "licm.h"

using namespace std;

/**
Build a new LICM object
*/
licm::licm(bool showStats, unsigned verbose)
    : opt(showStats, verbose), _stat(licm::licmStat(*this, showStats))
{
}


/**
Lock Independent Code Motion (LICM). This function implements the LICM
algorithm as described in the Technical Report 'Analysis and Optimization
of Explicitly Parallel Programs', TR 98-11, University of Alberta
(ftp://ftp.cs.ualberta.ca/pub/TechReports/1998/TR98-11/TR98-11.ps).
*/
bool
licm::execute()
{
    /* Steve rolling his head on the keyboard after watching me yell at the
     * computer in a particularly intense debugging session:
     *
     * sdfghbderwQAZCFDFDXUJK':?JFG
     * GHBNJRDEWSERT6YUOIK
     */
    D_SELFTEST_HEADER(1, "licm::execute");

    /** Incorporate the moved annotations into SUIF's annotation manager */
    k_to_premutex = lexicon->enter("to_premutex")->sp;
    k_to_postmutex = lexicon->enter("to_postmutex")->sp;
    k_move = lexicon->enter("move")->sp;

    _made_progress = false;

    /** Analyze all the synchronization variables used in the program. */
    set_var_sym::iterator mutex_iter = _ssa->mutexVars().begin();
    for (; mutex_iter != _ssa->mutexVars().end(); mutex_iter++) {
	p_var_sym var = *mutex_iter;

	D_SELFTEST(1) {
	    cout << "Processing lock variable '" << var->name() << "'\n";
	}
	   

	/** For each lock variable, analyze all its mutex bodies looking for
	 * lock independent code. We only analyze mutex bodies contained in
	 * the procedure that we are optimizing.
	 */
	mutex_struct *mxstruct = get_mutex_struct(var);
	mutex_struct::iterator interval_iter = mxstruct->begin();
	for (; interval_iter != mxstruct->end(); interval_iter++) {
	    p_mutex_body mb = *interval_iter;

	    D_SELFTEST(1) {
		static int tmp = 1;
		cout << "Mutex body " << tmp++ << ": " << mb << endl;
	    }

	    if (_tp != mb->entry()->parent()->tproc()) {
		continue;	/* Only process our own mutex bodies */
	    }

	    this->doLICM(mb);

	    D_SELFTEST(1) { cout << endl; }
	}
    }

    return _made_progress;
}



/**
Do LICM on the given mutex body.
*/
void
licm::doLICM(p_mutex_body mb)
{
    D_SELFTEST_HEADER(600, "licm::doLICM");

    tree_node_list_e *stmt_e, *lock, *unlock;
    p_tree_node stmt;

    lock = mb->entry_ti()->list_e();
    unlock = mb->exit_ti()->list_e();

    D_SELFTEST(600) {
	cout << "Lock statement for mutex body is:\n";
	lock->contents->print(); fflush(stdout);
	cout << "\nUnlock statement for mutex body is:\n";
	unlock->contents->print(); fflush(stdout);
	cout << endl;
	cout << "The mutex body is:\n";
	mb->printBody();
	cout << endl << endl;
    }


    /** Forward search looking for statements to move to the premutex node.
     * This phase might also find statements that can be moved to the
     * postmutex node.
     */

    D_SELFTEST(600) { cout << "Checking statements in forward order\n"; }
    for (stmt_e = lock->next(); stmt_e != unlock; stmt_e = stmt_e->next()) {
	stmt = stmt_e->contents;

	D_SELFTEST(601) {
	    cout << "About to consider statement:\n";
	    stmt->print(); fflush(stdout);
	    cout << endl;
	}

	if (stmt->peek_annote(k_move) == NULL && is_exec(stmt)) {
	    /** Only try to move executable statements that haven't been
	     * moved yet. We don't actually move statements while we
	     * traverse the mutex body. We do all the movements at the end.
	     */
	    this->tryToMove(stmt, mb);
	}
    }

    /**
    Backward search looking for statements to move to the postmutex node. This
    is needed because the previous pass might not have moved some down-movable
    statements that were blocking each other.
    */
    D_SELFTEST(600) { cout << "\n\nChecking statements in reverse order\n"; }
    for (stmt_e = unlock->prev(); stmt_e != lock; stmt_e = stmt_e->prev()) {
	stmt = stmt_e->contents;

	D_SELFTEST(601) {
	    cout << "\nAbout to consider statement:\n";
	    stmt->print(); fflush(stdout);
	    cout << endl;
	}

	if (stmt->peek_annote(k_move) == NULL && is_exec(stmt)) {
	    this->tryToMove(stmt_e->contents, mb);
	}
    }

    /* Once all the statements have been processed, we do the actual
     * movement.
     */
    for (stmt_e = lock->next(); stmt_e != unlock; stmt_e = stmt_e->next()) {
	stmt = stmt_e->contents;
	if (stmt->peek_annote(k_move)) {
	    this->moveStmt(stmt, mb);
	    _stat.addHoistedStmt(mb);
	    _made_progress = true;
	}
    }

    D_SELFTEST(600) { 
	cout << endl << endl << "Mutex body after moving statements" << endl;
	mb->printBody();
    }

    /* if the mutex body is empty, remove it */
    bool mbEmpty = true;
    for (stmt_e = lock->next(); stmt_e != unlock; stmt_e = stmt_e->next()) {
	stmt = stmt_e->contents;
	if (is_exec(stmt)) {
	    mbEmpty = false; 
	    break;
	}
    }

    if (mbEmpty) {
	D_SELFTEST(600) { 
	    cout << "Mutex body is empty. Removing lock/unlock statements.\n";
	}
	tree_node_list *tnl = lock->contents->parent();
	tnl->remove(lock);
	tnl->remove(unlock);

	_stat.addEmptyMB(mb);
    }

    D_SELFTEST_FOOTER(600);
}



/**
Tries to move the given statement to the pre or postmutex nodes for the
given mutex body.
*/
void
licm::tryToMove(p_tree_node stmt, p_mutex_body mb)
{
    D_STACK(tryToMove);
    D_SELFTEST_HEADER(610, "licm::tryToMove");

    int where = this->isMovable(stmt, mb, NULL);

    if (where == NOT_MOVABLE) {
	D_SELFTEST(610) { cout << "Statement is not movable.\n\n"; }
	D_SELFTEST_FOOTER(610);
	_stat.addNonMovableStmt(mb);
	return;
    }

    /* Statement can be moved. Annotate statement with possible destination. */
    stmt->append_annote(k_move, (void *)stmt.ptr());

    if (where & PREMUTEX) {
	D_SELFTEST(610) {
	    cout << "Statement can move to the premutex node.\n\n";
	}
	stmt->append_annote(k_to_premutex, (void *)stmt.ptr());
    }
    
    if (where & POSTMUTEX) {
	D_SELFTEST(610) {
	    cout << "Statement can move to the postmutex node.\n\n";
	}
	stmt->append_annote(k_to_postmutex, (void *)stmt.ptr());
    }

    _stat.addMovableStmt(mb);

    D_SELFTEST_FOOTER(610);
}




/**
Determines whether the given statement can be moved outside the given mutex
body. It returns a location where the statement can be placed. The third
argument is used when checking statements enclosed in control structures
(loops, ifs, etc). A statement is movable if:

\begin{enumerate}
\item it's lock independent,
\item its node dominates the exit node of the mutex body, and
\item it doesn't have any data dependencies with statements before/after it.
\end{enumerate}

When a statement is inside a control structure, these conditions should be
altered slightly to filter out the enclosing control structure. Since the
statement can never be moved outside the control structure, what we do is
try to move the whole structure outside the mutex body. For this to work,
all the statements inside the control structure must comply with the
following conditions:

\begin{enumerate}
\item They are all lock independent,
\item the node holding the entry point to the control structure dominates the
      exit node for the mutex body, and
\item they don't have any data dependencies with statements before/after
      the enclosing control structure.
\end{enumerate}

The third argument indicates whether the statement being checked is inside
a control structure or not. If this argument is NULL, the statement is not
enclosed in any control structure. Otherwise, this argument is a pointer to
the outermost control structure containing it (this handles the case of
nested control structures).
*/
int
licm::isMovable(p_tree_node stmt, p_mutex_body mb, p_tree_node enclCtrl)
{
    /* Non-executable statements can be moved anywhere really. */
    if (!is_exec(stmt)) { return (PREMUTEX | POSTMUTEX); }

    D_SELFTEST(3) {
	cout << "Checking statement #";
	if (stmt->is_instr()) {
	    cout << p_tree_instr(stmt)->instr()->number();
	} else {
	    cout << stmt->number();
	}
	cout << endl;
    }

    D_SELFTEST(3) { cout << endl; stmt->print(); fflush(stdout); cout << endl; }

    if (!stmt->is_instr()) {
	D_SELFTEST(3) { cout << "\nStatement is a control structure\n"; }
	if (!enclCtrl) { enclCtrl = stmt; }
	return this->isCtrlStructMovable(stmt, mb, enclCtrl);
    }

    assert(stmt->is_instr());
    p_tree_instr instr = stmt;

    /** If the statement is inside a control structure, we use the control
     * structure's entry node as the reference node. Otherwise, we use the
     * statement's node.
     */
    p_ccfg_node node;
    if (enclCtrl) {
	node = ccfg_begin_node(enclCtrl);
    } else {
	node = get_node(stmt);
    }
    assert(node);

    /* Check if the mobility conditions are met. */
    int dest = NOT_MOVABLE;
    if (node->dominates(mb->exit()) && this->isLI(instr)) {

	_stat.addLIStmt(mb);

	if ( ! this->hasDependencies(instr, mb, enclCtrl, UP)) {
	    dest |= PREMUTEX;
	}

	if ( ! this->hasDependencies(instr, mb, enclCtrl, DOWN)) {
	    dest |= POSTMUTEX;
	}
    }

    return dest;
}



/**
Determine whether a control structure is movable. A control structure can
only be moved if all the interior statements can be moved in the same
direction.
*/
int
licm::isCtrlStructMovable(p_tree_node stmt, p_mutex_body mb,
	p_tree_node enclCtrl)
{
    D_SELFTEST_HEADER(3, "isCtrlStructMovable");

    unsigned stmtCtr = 0, preCtr = 0, postCtr = 0, notMovableCtr = 0;

    /* Make sure that the statement is a control structure */
    assert(!stmt->is_instr());

    /* Check each instruction inside the control structure */
    for (unsigned i = 0; i < stmt->num_child_lists(); i++) {
	tree_node_list_iter iter(stmt->child_list_num(i));
	while (!iter.is_empty()) {
	    p_tree_node tn = iter.step();
	    stmtCtr++;

	    /* Determine if the statement is movable. If it is, increment
	     * the corresponding counter. If the statement is not movable,
	     * return immediately. Also keep track of the total number of
	     * statements.
	     *
	     * Notice that we also take provision for statements that can
	     * move both ways. If a statement can move to the premutex and
	     * the postmutex node, then we increment both counters so that
	     * later on if some statements can move in only one direction,
	     * the statements that can move in both directions contribute
	     * to the movement.
	     */
	    D_SELFTEST(1) { cout << "\n\t"; }

	    int where = this->isMovable(tn, mb, enclCtrl);

	    if (where == NOT_MOVABLE) { notMovableCtr++; break; }
	    if (where & PREMUTEX)    { preCtr++; }
	    if (where & POSTMUTEX)   { postCtr++; }
	}
    }


    /* If all the statements in the control structure can be moved in the
     * same direction, then the whole control structure can be moved.
     */
    int dest = NOT_MOVABLE;
    if (stmtCtr == preCtr) { dest |= PREMUTEX; }
    if (stmtCtr == postCtr) { dest |= POSTMUTEX; }

    D_SELFTEST(2) {
	cout << "\n\nChecked " << stmtCtr << " statements in the control "
	    << "structure\n";
	cout << notMovableCtr << " statement(s) cannot be moved.\n";
	cout << preCtr << " statement(s) can move up.\n";
	cout << postCtr << " statement(s) can move down.\n";
	cout << "Therefore, the statement can go ";
	if (dest == NOT_MOVABLE) { cout << "nowhere"; }
	if (dest & PREMUTEX) { cout << "to the premutex node "; }
	if (dest & POSTMUTEX) { cout << "to the postmutex "; }
	cout << endl << endl;
    }

    return dest;
}




/**
Determine whether the given instruction statement has dependencies with
other statements inside the mutex body. The direction argument determines
if the check is done on the statements above or the statements below the
given statement.
*/
bool
licm::hasDependencies(p_tree_instr instr, p_mutex_body mb,
	p_tree_node enclCtrl, Direction dir)
{
    p_tree_node first, last, s;
    tree_node_list_e *startAt;

    D_SELFTEST_HEADER(2, "hasDependencies");

    /* Determine where to start checking. If the statement is inside a
     * control structure, then we have to start at the entry point of the
     * control structure, otherwise we start at the statement above/below us.
     */
    startAt = (enclCtrl) ? enclCtrl->list_e() : instr->list_e();
    first = (dir == UP) ? startAt->prev()->contents : startAt->next()->contents;


    /* Determine where to stop checking. If we are going up, we stop at the
     * lock instruction for the mutex body, otherwise we stop at the
     * unlock.
     */
    last =  (dir == UP) ? mb->entry_ti()->list_e()->contents
			: mb->exit_ti()->list_e()->contents;


    D_SELFTEST(2) { 
	cout << "Checking data dependencies with statements "
	    << ((dir == UP) ? "above" : "below") << endl;
    }

    D_SELFTEST(3) {
	cout << "First statement to check:\n";
	first->print(stdout, 8); fflush(stdout); cout << endl;

	cout << "Last statement to check:\n";
	last->print(stdout, 8); fflush(stdout); cout << endl;
    }

    for (   s = first;
	    s != last;
	    s = (dir == UP) ? s->list_e()->prev()->contents 
	    		    : s->list_e()->next()->contents
	)
    {
	/* We only care about executable statements. */
	if (!::is_exec(s)) { continue; }

	D_SELFTEST(3) { s->print(stdout, 4); fflush(stdout); }

	/* Avoid checking a statement against itself. */
	if (s == instr) {
	    D_SELFTEST(3) { cout << "\t-> Same statement. Skipping.\n"; }
	    continue;
	}

	/* If the current statement is already marked as moved, skip it. */
	if (s->peek_annote(k_move)) {
	    D_SELFTEST(3) { cout << "\t-> Already moved. Skipping.\n";}
	    continue;
	}

	/* Check if there is any dependencies between the statement we are
	 * checking (instr) and the current statement (s).
	 */
	if (this->hasDependencies(instr, s)) {
	    D_SELFTEST(2) {
		cout << "Dependencies found. Statement cannot move "
		    << ((dir == UP) ? "up" : "down") << ".\n";
	    }

	    D_SELFTEST_FOOTER(2);

	    return true;
	}
    }


    D_SELFTEST(2) {
	cout << "No dependencies found. Statement can move "
	    << ((dir == UP) ? "up" : "down") << ".\n";
    }

    D_SELFTEST_FOOTER(2);

    return false;
}


static inline bool varCompare(p_varref r1, p_varref r2) { 
    return ::strcmp(r1->var()->name(), r2->var()->name()) < 0;
}

/**
Check if instr1 defines or uses variables defined in instr2 and vice-versa.
Notice that we cannot use standard reaching-definitions information because
we need to check for anti-dependencies.
*/
bool
licm::hasDependencies(p_tree_instr i1, p_tree_instr i2)
{
    D_STACK(hasDependencies);
    D_SELFTEST_HEADER(2, "hasDependencies");

    /* Retrieve the sets of definitions and uses made by the statements. */
    set_vardef *d1 = get_vardefs(i1.ptr());
    set_varuse *u1 = get_varuses(i1.ptr());

    set_vardef *d2 = get_vardefs(i2.ptr());
    set_varuse *u2 = get_varuses(i2.ptr());

    D_SELFTEST(3) {
	cout << "\nReferences in first instruction:\n";
	i1->print(); fflush(stdout);

	cout << "\n\tDefinitions: ";
	copy(d1->begin(), d1->end(), ostream_iterator<p_vardef>(cout, " "));
	cout << endl;

	cout << "\n\tUses: ";
	copy(u1->begin(), u1->end(), ostream_iterator<p_varuse>(cout, " "));
	cout << endl << endl;
    }

    D_SELFTEST(3) {
	cout << "\nReferences in second instruction:\n";
	i2->print(); fflush(stdout);

	cout << "\n\tDefinitions: ";
	copy(d1->begin(), d1->end(), ostream_iterator<p_vardef>(cout, " "));
	cout << endl;

	cout << "\n\tUses: ";
	copy(u2->begin(), u2->end(), ostream_iterator<p_varuse>(cout, " "));
	cout << endl << endl;
    }
	

    D_SELFTEST(2) { cout << "DEF-DEF "; }
    set_varref result;
    set_intersection(d1->begin(), d1->end(), d2->begin(), d2->end(),
	    inserter(result, result.begin()), varCompare);
    if (!result.empty()) {
	D_SELFTEST(2) { cout << "(Yes)\n"; }
	return true;
    }
    D_SELFTEST(2) { cout << "(No), DEF-USE "; }

    set_intersection(d1->begin(), d1->end(), u2->begin(), u2->end(),
	    inserter(result, result.begin()), varCompare);
    if (!result.empty()) {
	D_SELFTEST(2) { cout << "(Yes)\n"; }
	return true;
    }
    D_SELFTEST(2) { cout << "(No), USE-DEF "; }

    set_intersection(d2->begin(), d2->end(), u1->begin(), u1->end(),
	    inserter(result, result.begin()), varCompare);
    if (!result.empty()) {
	D_SELFTEST(2) { cout << "(Yes)\n"; }
	return true;
    }
    D_SELFTEST(2) { cout << "(No)\n"; }

    return false;
}



/**
Determines whether a given instruction node has dependencies with any of
the statements inside a control structure.
*/
bool
licm::hasDependencies(p_tree_instr ti, p_tree_node tn)
{
    if (tn->is_instr()) {
	/* If 'tn' is just an instruction node, call the base test. */
	return this->hasDependencies(ti, p_tree_instr(tn));
    }

    /* Traverse all the instruction lists in the control structure
     * recursively comparing each individual statement to the given
     * instruction.
     */
    for (unsigned i = 0; i < tn->num_child_lists(); i++) {
	tree_node_list_iter iter(tn->child_list_num(i));
	while (!iter.is_empty()) {
	    p_tree_node s = iter.step();

	    D_SELFTEST(2) {
		s->print(stdout, 4); fflush(stdout);
		cout << "\t-> ";
	    }

	    if (this->hasDependencies(ti, s)) {
		return true;
	    }
	}
    }

    return false;
}



/**
Returns 'true' if the given statement is Lock Independent. If the object is
of type Relaxed LICM, the function will use a relaxed notion of lock
independence (see rlicm::isRelaxedLI)
*/
bool
licm::isLI(p_tree_instr ti)
{
    p_ccfg_node node = get_node(ti);
    assert(node);

    /* Trivial case, do not try to move synchronization operations */
    if (ti->instr()->peek_annote(k_mutex_sync)) {
	D_SELFTEST(2) {
	    cout << "Statement is a synchronization operation. It is not LI.\n";
	}
	return false;
    }

    /* If the node has no conflicts, the statement is lock independent */
    if (node->conflicts().empty()) {
	D_SELFTEST(2) { cout << "Statement is lock independent\n"; }
	return true;
    }

    /* If the node has conflicts, the statement may or may not be lock
     * independent. Traverse the list of conflicts checking if any of the
     * references come from the statement. If at least one conflicting
     * reference comes from this statement, the statement is not lock
     * independent.
     */
    list_conflict::iterator iter = node->conflicts().begin();
    for (; iter != node->conflicts().end(); iter++) {
	conflict conf = *iter;
	p_varref this_ref = conf.head();

	/* The head of the current conflict is the reference that comes
	 * from this node. If that reference comes from the statement that
	 * we are checking, then the statement is involved in a data
	 * conflict and it's not lock independent in the strict sense.
	 *
	 * If this LICM object is a Relaxed LICM, we will also try to see if
	 * the references to the variable comply with the relaxed lock
	 * independence conditions.
	 */
	if (this_ref->instr()->parent() == ti.ptr()) {
	    D_SELFTEST(2) { 
		cout << "Statement is not lock independent. It has a conflict "
		    << "with: "; conf.tail()->print(); fflush(stdout);
		cout << endl;
	    }

	    /* Check if we have to apply a relaxed notion of lock
	     * independence. If so, check if the three conditions for
	     * relaxed lock independence are met.
	     *
	     * Only objets of class Relaxed LICM will actually check for
	     * relaxed conditions. Normal LICM objects will just return
	     * false.
	     */
	    if (this->isRelaxedLI(this_ref)) {
		continue;	/* Check the next reference */
	    } else {
		return false;	/* The reference is not RLI. Therefore the
				   statement cannot be RLI. */
	    }
	}
    }

    D_SELFTEST(2) { cout << "Statement is lock independent\n"; }
    return true;
}


/**
Move the given statement to the premutex or postmutex node of the given
mutex body.
*/
void
licm::moveStmt(p_tree_node stmt, p_mutex_body mb)
{
    /* The statement should be marked as movable already. */
    assert(stmt->peek_annote(k_move));

    tree_node_list_e *dest;
    tree_node_list *tnl = stmt->parent();

    D_STACK(moveStmt);
    D_SELFTEST_HEADER(2, "moveStmt");

    D_SELFTEST(2) {
	cout << "Moving statement to the " 
	    << (stmt->peek_annote(k_to_premutex) ? "premutex" : "postmutex")
	    << " node:\n";
	stmt->print(); fflush(stdout);
	cout << endl;
    }

    D_SELFTEST(3) {
	cout << "Mutex body before moving the statement" << endl;
	mb->printBody();
    }


    /* Insert the statement in the spot determined by the k_move
     * annotation. Remove the movement annotations before cloning the
     * statement.
     * 
     * Notice that we always try to move to the premutex node first. This
     * implements our simple cost model. When a statement can be moved to
     * both the premutex and postmutex nodes, it's always better to move it
     * to the premutex node to avoid lock congestions.
     */
    bool canMoveUp = (stmt->get_annote(k_to_premutex) != NULL);
    stmt->get_annote(k_move);
    stmt->get_annote(k_to_postmutex);

    if (canMoveUp) {
	dest = mb->entry_ti()->list_e();
	tnl->insert_before(stmt->clone(), dest);
    } else {
	dest = mb->exit_ti()->list_e();
	tnl->insert_after(stmt->clone(), dest);
    }

    tnl->remove(stmt->list_e());

    D_SELFTEST(2) {
	cout << endl << endl << "Mutex body after moving statement" << endl;
	mb->printBody();
    }
}


/*---------------------------------------------------------------------------
		       Methods for the statistic classes
---------------------------------------------------------------------------*/
void
licm::licmStat::addHoistedStmt(const p_mutex_body& mb)
{
    if (_active) { this->incr(_hoisted_stmts, mb); }
}


void
licm::licmStat::addEmptyMB(const p_mutex_body& mb)
{
    if (_active) { this->incr(_empty_mbs, mb); }
}


void
licm::licmStat::addNonMovableStmt(const p_mutex_body& mb)
{
    if (_active) { this->incr(_non_movable_stmts, mb); }
}


void
licm::licmStat::addMovableStmt(const p_mutex_body& mb)
{
    if (_active) { this->incr(_movable_stmts, mb); }
}


void
licm::licmStat::addLIStmt(const p_mutex_body& mb)
{
    if (_active) { this->incr(_LI_stmts, mb); }
}


void 
licm::licmStat::addRLIStmt(const p_mutex_body& mb)
{
    if (_active) { this->incr(_RLI_stmts, mb); }
}


/**
Increment the counter associated with the given mutex body.
*/
void
licm::licmStat::incr(licm_counter& counter, const p_mutex_body& mb)
{
    if (counter.find(mb) == counter.end()) {
	counter[mb] = 1;
    } else {
	counter[mb]++;
    }
}

/**
Print statistics for LICM.
*/
void
licm::licmStat::print(ostream& s) const
{
    if (!_active) { return; }

    this->MutexStat::print(s);

    set_var_sym mxvars = _pass.ssa()->mutexVars();
    for (set_var_sym::iterator i = mxvars.begin(); i != mxvars.end(); i++) {
	p_var_sym var = *i;

	s << "\n\nLICM statistics for mutex structure for " << var << "\n";

	s << "\nHoisted statements\n";
	this->print_counter(_hoisted_stmts, var, s);

	s << "\nEmpty mutex bodies\n";
	this->print_counter(_empty_mbs, var, s);

	s << "\nNon-movable statements\n";
	this->print_counter(_non_movable_stmts, var, s);

	s << "\nLock Independent (LI) statements\n";
	this->print_counter(_LI_stmts, var, s);

	s << "\nRelaxed Lock Independent (RLI) statements\n";
	this->print_counter(_RLI_stmts, var, s);
    }
}


void
licm::licmStat::print_counter(const licm_counter& counter, const p_var_sym& var,
    ostream& s) const
{
    unsigned total = 0;
    mutex_struct *mxstruct = get_mutex_struct(var);
    mutex_struct::iterator j;

    for (j = mxstruct->begin(); j != mxstruct->end(); j++) {
	p_mutex_body mb = *j;
	unsigned cnt = counter.find(mb)->second;

	s << "\tMutex body "; mb->print_brief(s); s << ": " << cnt << endl;
	total += cnt;
    }
    s << "Total for mutex structure: " << total << endl;
}
