/**
$Id: mbl.cc,v 1.19 1999/03/15 21:28:11 diego Exp $

Mutex body localization. This transformation converts references to shared
variables inside a mutex body into references to variables local to the
mutex body. 

The basic idea is simple. All the modifications to shared variables inside a
mutex body are only observed locally by the thread. The only changes observed
by other threads are those definitions that reach the exit node of the thread.
The localization transformation makes these properties explicit by creating
local copies of the shared variable, making all internal modifications on the
local variable and copying back the local variable to the shared copy at the
end.

This transformation will likely increase the chances of finding lock
independent code. 
*/
#include <string.h>

#include "mbl.h"

using namespace std;

/**
Constructor for MBL objects.
*/
mbl::mbl(bool showStats, unsigned verbose)
    : opt(showStats, verbose), _stat(mbl::mblStat(*this, showStats))
{
}

/**
Show statistics about the optimization on the given stream.
*/
void
mbl::printStats(ostream& s) const
{
    _stat.print(s);
}

/**
Mutex Body Localization. This function implements the MBL algorithm as
described in the Technical Report 'Analysis and Optimization of Explicitly
Parallel Programs', TR 98-11, University of Alberta
(ftp://ftp.cs.ualberta.ca/pub/TechReports/1998/TR98-11/TR98-11.ps).
*/
bool
mbl::execute()
{
    D_SELFTEST_HEADER(1, "mbl::execute");

    _made_progress = false;

    /** Traverse all the mutex structures in the program looking for variables
     * to localize.
     */
    set_var_sym::iterator mutex_iter = _ssa->mutexVars().begin();
    for (; mutex_iter != _ssa->mutexVars().end(); mutex_iter++) {
	p_var_sym var = *mutex_iter;

	D_SELFTEST(1) {
	    cout << "Processing lock variable '" << var->name() << "'\n";
	}

	mutex_struct *mxstruct = get_mutex_struct(var);
	mutex_struct::iterator interval_iter = mxstruct->begin();
	for (; interval_iter != mxstruct->end(); interval_iter++) {
	    p_mutex_body mb = *interval_iter;

	    D_SELFTEST(1) { 
		static int tmp = 1;
		cout << "Mutex body " << tmp++ << ": " << mb << endl;
	    }

	    if (_tp != mb->entry()->parent()->tproc()) {
		continue;	/* Only process our own mutex bodies */
	    }

	    this->doMBL(mb);

	    D_SELFTEST(1) { cout << endl; }
	}
    }

    D_SELFTEST_FOOTER(1);

    return _made_progress;
}


/**
Look for variables to localize in the given mutex body.
*/
void
mbl::doMBL(p_mutex_body mb)
{
    /** 
    Determine which variables need to be localized (findVarsToLocalize) and
    perform the localization.
    */
    set_var_sym vars = this->findVarsToLocalize(mb);
    set_var_sym::iterator iter;

    for (iter = vars.begin(); iter != vars.end(); iter++) {
	this->localize((*iter), mb);
    }
}


/**
Determine which variables should be localized in the given mutex body.
*/
set_var_sym
mbl::findVarsToLocalize(p_mutex_body mb)
{
    set_var_sym vars;

    /** Traverse all the nodes in the mutex body looking for variable
     * references.
     */
    set_ccfg_node::iterator node_iter = mb->body().begin();
    for (; node_iter != mb->body().end(); node_iter++ ) {
	p_ccfg_node node = *node_iter;

	/** Determine which variables have been referenced in this node. */
	set_varref::iterator ref_iter = node->refs().begin();
	for (; ref_iter != node->refs().end(); ref_iter++) {
	    p_varref ref = *ref_iter;

	    /* Do not process clone references, only originals! */
	    if (!ref->is_clone() && (ref->isD() || ref->isU())) {
		vars.insert(ref->var());
	    }
	}
    }

    /** Once we have collected all the variables referenced in the mutex
     * body, we determine which can actually be localized.
     */
    set_var_sym::iterator iter;
    for (iter = vars.begin(); iter != vars.end(); iter++) {
	if ( ! this->isLocalizable((*iter), mb)) {
	    vars.erase(iter);
	}
    }

    return vars;
}


/**
Core localization algorithm. Determine whether a local copy of the given
variable can be created inside the given mutex body.
*/
void
mbl::localize(p_var_sym var, p_mutex_body mb)
{
    D_SELFTEST_HEADER(1, "mbl::localize");

    D_SELFTEST(1) { cout << "Localizing variable '" << var->name() << "'\n"; }

    if ( ! this->isLocalizable(var, mb)) {
	D_SELFTEST(1) { 
	    cout << "Variable '" << var->name() << "' cannot be localized.\n";
	    D_SELFTEST_FOOTER(1);
	}
	return;
    }


    /* Initialize the SUIF builder to the current procedure. */
    block::set_proc(mb->parent()->tproc().ptr());

    /* Create a local copy for the variable. We add a new variable to the
     * variable's symbol table and assign it a unique name based on the
     * original name of the variable to localize.
     */
    block local = block::new_sym(var->type(), String("_local_") + var->name());

    this->createEntryCopy(var, local, mb);
    this->modifyInteriorRefs(var, local, mb);
    this->createExitCopy(var, local, mb);

    _made_progress = true;

    D_SELFTEST_FOOTER(1);
}



/**
Helper function that determines whether or not the variable in the given
mutex body can be localized. 

For the time being we add pretty much any variable defined or used within
the mutex body that has a non-empty list of conflicts. That is, we add all
the variables that have conflicting references with other threads.

This function returns true iff:

\begin{enumerate}
\item	The variable has conflicts, and

\item	the variable is not spilled (a spilled variable is one that is static,
	its address has been taken or it's an aggregate like an array, a
	structure or a union), and

\item	the variable is not a synchronization object, and

\item	the variable should only be referenced inside mutex bodies in the same
	mutex structure. Otherwise, it is possible that the variable is
	being accessed by another thread concurrently with the references
	inside this mutex body.
\end{enumerate}
*/
bool
mbl::isLocalizable(p_var_sym var, p_mutex_body mb)
{
    D_SELFTEST_HEADER(620, "mbl::isLocalizable");

    reasons reason;

    set_varref *refs = ::get_varrefs(var.ptr());

    /* Check the individual reasons that prevent a variable from being
     * localized. We go from simpler to the more elaborate/expensive
     * checks.
     */
    if (refs->empty()) {
	reason = NO_REFERENCES;
    } else if (var->is_addr_taken() || !var->is_scalar() || var->is_param()) {
	reason = SPILLED;
    } else if (var->peek_annote(k_sync_var)) {
	reason = ITS_SYNC_VAR;
    } else if (!this->hasConflicts(var)) {
	reason = NO_CONFLICTS;
    } else if (this->hasConflictsOutsideMS(var, mb)) {
	reason = CONFLICTS_OUTSIDE_MB;
    } else {
	_stat.addLocalizableVar(var, mb);
	D_SELFTEST_FOOTER(620);
	return true;
    }

    _stat.addNonLocalizableVar(var, mb, reason);

    D_SELFTEST(620) {
	cout << "Will not localize variable '" << var->name() << "' because ";

	if (reason == NO_REFERENCES) {
	    cout << "there are no references to it.\n";
	} else if (reason == CONFLICTS_OUTSIDE_MB) {
	    cout << "there are conflicting refs outside\nits mutex body.\n";
	} else if (reason == mbl::NO_CONFLICTS) {
	    cout << "it has no conflicts.\n";
	} else if (reason == mbl::SPILLED) {
	    cout << "it is an array/structure or\n\t"
		<< "it's a function argument or its address has been taken.\n";
	} else if (reason == mbl::ITS_SYNC_VAR) {
	    cout << "it is a synchronization variable.\n";
	}
    }

    D_SELFTEST_FOOTER(620);
    return false;
}



/**
Return true if the given variable has conflicting references in the procedure.
*/
bool
mbl::hasConflicts(p_var_sym var)
{
    bool hasConflicts = false;

    set_varref *refs = ::get_varrefs(var.ptr());

    set_varref::iterator ref_iter;
    for (ref_iter = refs->begin(); ref_iter != refs->end(); ref_iter++) {
	p_varref ref = *ref_iter;
	if (!ref->is_clone() && !ref->conflicts().empty()) {
	    hasConflicts = true;
	    break;
	}
    }

    return hasConflicts;
}


/**
Return true if the given variable has conflicting references outside the
same mutex structure. That is, determine whether or not all the conflicting
references are protected by the same lock.
*/
bool
mbl::hasConflictsOutsideMS(p_var_sym var, p_mutex_body mb)
{
    D_SELFTEST_HEADER(630, "mbl::hasConflictsOutsideMS");

    set_varref *refs = ::get_varrefs(var.ptr());
    mutex_struct *mxstruct = mb->mxstruct();
    assert(mxstruct);

    D_SELFTEST(630) {
	cout << "Checking if variable '" << var->name() << "' has "
	    << "conflicting reference outside the mutex structure\n";
	cout << "for the mutex body: " << mb << endl << endl;
    }

    /** Traverse all the conflicting references for the variable. They must
     * all be inside mutex bodies inside the same mutex structure.
     */
    bool hasConflicts;
    set_varref::iterator ref_iter;
    for (ref_iter = refs->begin(); ref_iter != refs->end(); ref_iter++) {
	p_varref ref = *ref_iter;

	D_SELFTEST(630) {
	    cout << "\n\nAnalyzing reference: "; ref->print(); cout << endl;
	}

	/* Only process original references. */
	if (ref->is_clone()) {
	    D_SELFTEST(630) { cout <<"Reference is a clone. Skipping.\n";}
	    continue;
	}

	/* Ignore the reference if it has no conflicts. */
	if (ref->conflicts().empty()) {
	    D_SELFTEST(630) { cout <<"Reference has no conflicts. Skipping.\n";}
	    continue;
	}

	/* If the reference has conflicts, then the reference must be
	 * within one of the mutex bodies of the same mutex structure
	 * containing 'mb'.
	 */
	D_SELFTEST(630) {
	    cout << "Reference has conflicts.\n";
	    cout << "Checking if the reference is inside the same mutex "
		<< "structure as\n" << mb << endl;
	}

	hasConflicts = true;
	p_ccfg_node refnode = ref->node();
	mutex_struct::iterator mbiter = mxstruct->begin();
	for (; mbiter != mxstruct->end(); mbiter++) {
	    p_mutex_body mb1 = *mbiter;

	    /* If the reference is inside 'mb1', which is a sibling of
	     * 'mb', then the reference is protected.
	     */
	    if (mb1->entry()->parent() == refnode->parent() &&
		mb1->entry()->dominates(refnode) &&
		mb1->exit()->postdominates(refnode)) 
	    {
		D_SELFTEST(630) {
		    cout << "Yes. The reference is protected.\n";
		    cout << "It is inside mutex body: " << mb1 << endl;
		}
		hasConflicts = false;
		break;
	    }
	}

	if (hasConflicts) {
	    D_SELFTEST(630) {
		cout << "\nNo. The reference is not inside any mutex body "
		    << "in the same mutex structure as\n"
		    << mb << endl << endl;
	    }
	    break;
	}
    }

    D_SELFTEST(630) {
	cout << "\nThe variable is " 
	    << ((hasConflicts) ? "not protected" : "protected")
	    << endl;
    }

    D_SELFTEST_FOOTER(630);

    return hasConflicts;
}





/**
Conditionally create an entry copy for the variable. An entry copy is
needed if and only if the variable has upward exposed uses in the mutex
body (ie, uses whose reaching definitions are outside the mutex body).
*/
void
mbl::createEntryCopy(p_var_sym var, block& local, p_mutex_body mb)
{
    D_SELFTEST_HEADER(640, "mbl::createEntryCopy");

    D_SELFTEST(640) {
	cout << "Checking whether " << var << " needs an entry copy for "
	    << endl << "mutex body " << mb << endl;
	cout << endl;
	cout << "Upward exposed references for the mutex body are:\n\t";
	copy(mb->upwardExposedUses().begin(),
	     mb->upwardExposedUses().end(),
	     ostream_iterator<p_varuse>(cout, "\n\t"));
    }

    bool needEntryCopy = false;
    set_varuse::iterator expuses_iter = mb->upwardExposedUses().begin();
    for (; expuses_iter != mb->upwardExposedUses().end(); expuses_iter++) {
	p_varuse expuse = *expuses_iter;
	/* Ignore clones. Only relevant for original references. */
	if (!expuse->is_clone() && expuse->var() == var) {
	    needEntryCopy = true;
	    break;
	}
    }

    D_SELFTEST(640) {
	if (needEntryCopy) {
	    cout << "Yes, the variable is upward exposed\n";
	} else {
	    cout << "No, the variable is not upward exposed\n";
	}
    }


    /* If we need an entry copy for the local variable, create the
     * statement 'local = var' using the SUIF builder. */
    if (needEntryCopy) {
	D_SELFTEST(640) {
	    cout << "\tAdding '" << local.get_sym()->name() << " = " 
		<< var->name() << ";' to the entry node.\n\n";
	}

	/* Create the statement as a tree_node */
	block B_var(var.ptr());
	block B_entry_copy(local = B_var);
	p_tree_node new_tn = B_entry_copy.make_tree_node(mb->entry_ti().ptr());

	/* Insert it right after the entry to the mutex body */
	tree_node_list *tnl = mb->entry_ti()->parent();
	tree_node_list_e *where = mb->entry_ti()->list_e();
	tnl->insert_after(new_tn.ptr(), where);

	/* Update statistics */
	_stat.addEntryCopyFor(var, mb);
    }

    D_SELFTEST_FOOTER(640);
}



/**
Traverse all the interior references to the given variable 'var' replacing
them by reference to the variable 'local'.
*/
void
mbl::modifyInteriorRefs(p_var_sym var, block& local, p_mutex_body mb)
{
    D_SELFTEST_HEADER(650, "mbl::modifyInteriorRefs");

    set_ccfg_node::iterator node_iter = mb->body().begin();
    for (; node_iter != mb->body().end(); node_iter++) {
	p_ccfg_node node = *node_iter;

	set_varref::iterator refs_iter = node->refs().begin();
	for (; refs_iter != node->refs().end(); refs_iter++) {
	    p_varref ref = *refs_iter;

	    if (ref->is_clone()) {
		continue;	/* Ignore clone references */
	    }

	    /* Only replace real references (D or U) to the variable */
	    if (ref->var() == var && (ref->isD() || ref->isU())) {
		D_SELFTEST(650) {
		    cout << "\tReplacing interior reference ";
		    ref->print_brief(); cout << endl;
		}

		instruction *instr = ref->instr();
		D_SELFTEST(651) {
		    cout << "\t-> Statement to modify:\n";
		    instr->parent()->print(stdout, 8);
		    cout << endl;
		}

		this->replaceRefWith(instr, var, local);

		/* Update statistics */
		_stat.addLocalizedReference(var, mb);

		D_SELFTEST(651) {
		    cout << "\t-> Modified statement:\n";
		    instr->parent()->print(stdout, 8);
		    cout << endl << endl;
		}
	    }
	}
    }

    D_SELFTEST_FOOTER(650);
}


/**
Replace all the operands that reference 'var' in the given instruction to
reference 'local'.

NOTE: No attempt is made to handle pointers, structures and arrays.
*/
void
mbl::replaceRefWith(instruction *instr, p_var_sym var, block& local)
{
    /* Analyze source operands. */
    for (int i = 0; i < instr->num_srcs(); i++) {
	operand op(instr->src_op(i));

	if (op.is_instr()) {
	    replaceRefWith(op.instr(), var, local);
	} else if (op.is_symbol()) {
	    if (op.symbol()->is_var()) {
		if (op.symbol() == var.ptr()) {
		    instr->set_src_op(i, operand((var_sym *)local.get_sym()));
		}
	    } else {
		warning_line(instr, "Source operand '%s' is not a variable\n", 
			    op.symbol()->name());
	    }
	}
    }

    /* Analyze the destination operand. */
    operand dst_op(instr->dst_op());
    if (dst_op.is_symbol()) {
	if (dst_op.symbol()->is_var()) {
	    if (dst_op.symbol() == var.ptr()) {
		instr->set_dst(operand((var_sym *)local.get_sym()));
	    }
	} else {
	    warning_line(instr, "Destination operand '%s' is not a variable\n",
			dst_op.symbol()->name());
	}
    }
}




/**
Conditionaly update the shared variable 'var' on exit of the mutex body. An
exit copy is needed if and only if at least one definition for the variable
inside the mutex body reaches the exit node.
*/
void
mbl::createExitCopy(p_var_sym var, block& local, p_mutex_body mb)
{
    D_SELFTEST_HEADER(660, "mbl::createExitCopy");

    D_SELFTEST(660) {
	cout << "Checking whether " << var << " needs an exit copy for "
	    << endl << "mutex body " << mb << endl;
	cout << endl;
	cout << "Exit reaching definitions for the mutex body are:\n\t";
	copy(mb->exitRDefs().begin(),
	     mb->exitRDefs().end(),
	     ostream_iterator<p_vardef>(cout, "\n\t"));
    }

    boolean needExitCopy = false;
    set_vardef::iterator exitRDefs_iter = mb->exitRDefs().begin();
    for (; exitRDefs_iter != mb->exitRDefs().end(); exitRDefs_iter++) {
	p_vardef exitRDef = *exitRDefs_iter;

	if (!exitRDef->is_clone() && exitRDef->var() == var) {
	    needExitCopy = true;
	    break;
	}
    }

    D_SELFTEST(660) {
	if (needExitCopy) {
	    cout << "Yes, the variable has exit reaching definitions\n";
	} else {
	    cout << "No, the variable doesn't have exit reaching definitions\n";
	}
    }

    /* If we need an exit copy to update the shared variable, create the
     * statement 'var = local' using the SUIF builder. */
    if (needExitCopy) {
	D_SELFTEST(660) {
	    cout << "\tAdding '" << var->name() << " = " 
		<< local.get_sym()->name() << ";' at the exit node.\n\n";
	}

	/* Create the statement as a tree_node. */
	block B_var(var.ptr());
	block B_exit_copy(B_var = local);
	p_tree_node new_tn = B_exit_copy.make_tree_node(mb->exit_ti().ptr());

	/* Insert it right before the exit of the mutex body */
	tree_node_list *tnl = mb->exit_ti()->parent();
	tree_node_list_e *where = mb->exit_ti()->list_e();
	tnl->insert_before(new_tn.ptr(), where);

	/* Update statistics */
	_stat.addExitCopyFor(var, mb);
    }

    D_SELFTEST_FOOTER(660);
}



/*---------------------------------------------------------------------------
		       Methods for the statistic classes
---------------------------------------------------------------------------*/
void
mbl::mblStat::addLocalizableVar(p_var_sym var, p_mutex_body mb)
{
    if (!_active) { return; }
    _localizedVars.insert(locTuple(mb->var(), var));
}

void 
mbl::mblStat::addNonLocalizableVar(p_var_sym var, p_mutex_body mb, 
	mbl::reasons reason)
{
    if (!_active) { return; }
    _nonLocalizedVars.insert(locTuple(mb->var(), var, reason));
}

void
mbl::mblStat::addEntryCopyFor(p_var_sym var, p_mutex_body mb)
{
    if (!_active) { return; }

    locTuple o(mb->var(), var);

    if (_entryCopies.find(o) == _entryCopies.end()) {
	_entryCopies[o] = 1;
    } else {
	_entryCopies[o]++;
    }
}


void 
mbl::mblStat::addLocalizedReference(p_var_sym var, p_mutex_body mb)
{
    if (!_active) { return; }

    locTuple o(mb->var(), var);

    if (_localizedRefs.find(o) == _localizedRefs.end()) {
	_localizedRefs[o] = 1;
    } else {
	_localizedRefs[o]++;
    }
}



void 
mbl::mblStat::addExitCopyFor(p_var_sym var, p_mutex_body mb)
{
    if (_active) {
	locTuple o(mb->var(), var);

	if (_exitCopies.find(o) == _exitCopies.end()) {
	    _exitCopies[o] = 1;
	} else {
	    _exitCopies[o]++;
	}
    }
}


/**
Print statistics for MBL.
*/
ostream&
operator<<(ostream& s, const mbl::locTuple& o) 
{
    s << "<";
    o._lock->print();
    s << ", ";
    o._var->print();
    fflush(stdout);	// Unfortunate but necessary. SUIF doesn't do streams.
    s << ">";
    
    if (o._reason != mbl::OK_TO_LOCALIZE) {
	s << " -> ";
	if (o._reason == mbl::NO_REFERENCES) {
	    s << "No references";
	} else if (o._reason == mbl::NO_CONFLICTS) {
	    s << "No conflicting references";
	} else if (o._reason == mbl::SPILLED) {
	    s << "Variable is spilled";
	} else if (o._reason == mbl::ITS_SYNC_VAR) {
	    s << "Variable is a synchronization object";
	} else if (o._reason == mbl::CONFLICTS_OUTSIDE_MB) {
	    s << "Conflicting references outside mutex structure";
	} else {
	    s << "Yo que se, tarado. This is not good";
	}
	s << "." << flush;
    }

    return s;
}


void
mbl::mblStat::print(ostream& s) const
{
    if (!_active) { return; }

    map<p_var_sym, unsigned> sharedRefs = this->countSharedReferences();

    this->MutexStat::print(s);

    s << "\nLocalized variables:\n\t";
    copy (_localizedVars.begin(), _localizedVars.end(),
	    ostream_iterator<locTuple>(s, "\n\t"));
    s << endl << endl;

    s << "Non localized variables:\n\t";
    copy (_nonLocalizedVars.begin(), _nonLocalizedVars.end(),
	    ostream_iterator<locTuple>(s, "\n\t"));
    s << endl << endl;

    set<locTuple>::iterator iter;
    s << "Localization results:\n";
    for (iter = _localizedVars.begin(); iter != _localizedVars.end(); iter++) {
	locTuple t = *iter;

	unsigned shared = sharedRefs[t._var];
	unsigned local  = _localizedRefs.find(t)->second;
	double percent  = (double(local) / double(shared)) * 100.0;

	s << "\t" << t << ":\n";
	s << "\t\t" << _entryCopies.find(t)->second << " entry copies\n";
	s << "\t\t" << _exitCopies.find(t)->second << " exit copies\n";
	s << "\t\t" << shared << " shared references\n";
	s << "\t\t" << local << " localized references (" 
	    << setiosflags(ios::fixed) << setprecision(0) 
	    << percent << "%)\n";
	s << endl;
    }
}


/**
Compute the number of shared references for each of the localized variables
so that we can determine what percentage of shared references we saved.
*/
map<p_var_sym, unsigned>
mbl::mblStat::countSharedReferences() const
{
    map<p_var_sym, unsigned> sharedRefs;

    set<locTuple>::iterator iter = _localizedVars.begin();
    for ( ; iter != _localizedVars.end(); iter++) {
	p_var_sym var = iter->_var;
	sharedRefs[var] = 0;

	set_varref *refs = ::get_varrefs(var.ptr());
	set_varref::iterator ref_iter = refs->begin();
	for (; ref_iter != refs->end(); ref_iter++) {
	    p_varref ref = *ref_iter;

	    if (ref->is_clone()) {
		continue;	/* Ignore clones. */
	    }

	    vector_thread_body *threads = ::get_thread_bodies(ref->instr());
	    if ((ref->isD() || ref->isU()) && threads && !threads->empty()) {
		sharedRefs[var]++;
	    }
	}
    }

    return sharedRefs;
}
