/**
$Id: rlicm.cc,v 2.4 1999/02/16 16:17:57 diego Exp $
*/
#include "licm.h"


/**
Checks whether the given reference complies with the three conditions for
Relaxed Lock Independence (RLI).

\begin{enumerate}
\item	The target architecture uses a sequential consistent memory model
	or one of its weaker cousins. This means that when a variable is
	written to shared memory, the operation should be propagated
	without any special mechanism like the ones used in release
	consistent architectures.

\item	There is a single thread that writes to the variable.

\item	All the references to the variable are atomic w.r.t. the operations
	on it (ie, the variable is not an aggregate data type that requires
	multiple memory operations to update or retrieve).

\item	The writing thread writes to the variable only once.
\end{enumerate}

Notice that the first requirement is implicitly assumed to be true by this
method. This should really be parameterized.
*/
bool
rlicm::isRelaxedLI(p_varref ref)
{
    D_SELFTEST(2) {
	cout << "\nChecking the reference " << ref;
	cout << " for Relaxed Lock Independence\n";
    }

    /* ATOMICITY -> To guarantee that the reference is atomic
     * w.r.t. the operation performed, we check whether the
     * variable is scalar or not. If it is a scalar variable,
     * we assume that the operation performed is atomic.
     */
    if (ref->var()->is_scalar()) {
	/* Write once -> There can only be one definition for
	 * this variable.
	 */
	unsigned numconfs = this->numConflictingDefs(ref->var());
	if (numconfs == 1) {
	    D_SELFTEST(2) { cout << "Reference is RLI\n"; }
	    return true;	/* Check the next reference */
	} else {
	    D_SELFTEST(2) {
		cout << "Reference is NOT RLI. It has " << numconfs
		    << " conflicting definitions\n";
	    }
	    return false;
	}
    } else {
	D_SELFTEST(2) {
	    cout << "The variable is not a scalar => it's not RLI.\n";
	}
	return false;
    }
}


/**
Count the number of conflicting definitions for the given variable.
*/
unsigned
rlicm::numConflictingDefs(p_var_sym var)
{
    assert(var);
    set_vardef *defs = get_vardefs(var.ptr());

    if (defs == NULL) { return 0; }

    unsigned count = 0;
    set_vardef::iterator iter;
    for (iter = defs->begin(); iter != defs->end(); iter++) {
	p_vardef def = *iter;
	
	if (def->conflicts().size() >= 1) {
	    count++;
	}
    }

    return count;
}
