/**
$Id: cobegin.cc,v 2.8 1999/03/05 14:50:48 diego Exp $
*/
#include <vector>

#include <iostream.h>

#include <suif_copyright.h>
#include <suif.h>

#include <odc-util.h>
#include <d_lib.h>

#include "par.h"


/**
Build a new tree_cobegin object.
*/
tree_cobegin::tree_cobegin(p_tree_node mbr) : tree_par(mbr)
{
    D_SELFTEST_HEADER(670, "tree_cobegin::tree_cobegin");

    assert(is_cobegin(mbr));

    tree_instr *_mbr = (tree_instr *)mbr.ptr();
    _instr = (in_mbr *)_mbr->instr();

    /* Insert all the 'mbr' labels in the set of labels. This set is used
     * when determining whether an instruction is the last instruction of
     * a thread.
     */
    p_label_sym default_lab = _instr->default_lab();

    for (unsigned i = 0; i < _instr->num_labs(); i++) {
	_labels.insert(_instr->label(i));
    }
    _labels.insert(default_lab);

    D_SELFTEST(670) {
	cout << "Building a cobegin tree with " << _instr->num_labs() 
	    << " threads\n";
    }

    /* Now we build the list of threads that are included in the cobegin.
     * The pass 'findpar' has transformed the original switch statement
     * into an mbr that looks like this:
     *		mbr main.suif_tmp1, low 1, L:main.L3, (L:main.L4, L:main.L5)
     *		lab L:main.L4
     *		...
     *		jmp L:main.L3
     *		lab L:main.L5
     *		...
     *		lab L:main.L3
     *
     * Each thread begins with a unique label and ends with a jump to the
     * default label. We iterate until we find the default label and
     * collect the threads as we go down.
     */
    tree_node_list_iter iter(_mbr->parent());
    p_tree_node tn;

    /* Skip over the mbr itself. */
    iter.set(_mbr->list_e());
    iter.step();

    for (unsigned i = 0; i < _instr->num_labs(); i++) {
        bool endOfThread;
        cobegin_thread *thread; 

        /* We should not run out of nodes in the program */
        assert(iter.is_empty() == false);

	thread = new cobegin_thread(this, i, iter.peek(), iter.peek());

	D_SELFTEST(670) {
	    cout << "Found start of thread #" << i << " at tree node:\n",
	    iter.peek()->print();
	    fflush(stdout);
	    cout << endl;
	}

        endOfThread = false;
        while (!endOfThread && !iter.is_empty()) {
            tn = iter.step();
	    thread->extend(tn);

	    D_SELFTEST(670) {
		cout << "Added tree node " << tn->number() << " to thread #"
		    << i << endl;
		tn->print();
		fflush(stdout);
		cout << endl;
	    }

            if (tn->is_instr()) {
                instruction *instr = p_tree_instr(tn)->instr();

		/* We found the end of this thread if the next instruction
		 * is the label for the next thread or the default label
		 * (the default label marks the end of the mbr instruction).
		 */
                if (this->isEndOfThread(instr)) {
		    endOfThread = true;
		    D_SELFTEST(670) {
			cout << "Found end of thread #" << i << " at tree node:"
			    << endl;
			tn->print();
			fflush(stdout);
			cout << endl << endl;
		    }
		}
            }
        }   

        _threads.push_back(thread);
    }

    /* The last instruction of the cobegin is the label instruction
     * containing the default label for the 'mbr'.
     */
    assert(tn->list_e()->next()->contents->is_instr());
    p_tree_instr last = (tree_instr *)tn->list_e()->next()->contents;

    assert(last->instr()->opcode() == io_lab);
    _last = last;

    this->annotate_threads();

    D_SELFTEST_FOOTER(670);
}


/**
Returns true if the given instruction is the last instruction of a thread.
*/
bool
tree_cobegin::isEndOfThread(instruction *instr)
{
    /* If the instruction following 'instr' is one of the 'mbr' labels,
     * then this is the last instruction of the current thread.
     */
    p_tree_node next = instr->parent()->list_e()->next()->contents;
    assert(next);

    if (next->is_instr()) {
	p_tree_instr next_ti = p_tree_instr(next);
	if (next_ti->instr()->opcode() == io_lab) {
	    in_lab *label = (in_lab *)next_ti->instr();
	    return (_labels.find(label->label()) != _labels.end());
	}
    }

    return false;
}
