/**
$Id: odc-trees.cc,v 2.3 1999/03/05 14:50:49 diego Exp $
*/
#include "par.h"

/*---------------------------------------------------------------------------
			   Methods for class tree_par
---------------------------------------------------------------------------*/
static void _set_parallel_ancestors(p_tree_node tn, vector_tree_par *panc);

/**
Base constructor for tree_par objects. Set the parallel ancestor chaining.
Parallel ancestors are all the tree_par objects that hold this tree_par
object (including itself).

This information is required when computing the concurrency relation
between two statements/nodes.
*/
tree_par::tree_par(p_tree_node tn) : _tn(tn)
{
    _panc.push_back(this);
    _set_parallel_ancestors(_tn, &_panc);
}


static void
_set_parallel_ancestors(p_tree_node tn, vector_tree_par *panc)
{
    vector_tree_par *parents = ::get_tree_par_parents(tn.ptr());
    if (!parents) return;

    for (unsigned i = 0; i < parents->size(); i++) {
	p_tree_par parent = (*parents)[i];
	panc->push_back(parent);
	_set_parallel_ancestors(parent->entry(), panc);
    }
}

/**
Destructor. Remove all the annotations from the thread bodies.
*/
tree_par::~tree_par()
{
    for (unsigned i = 0; i < this->num_threads(); i++) {
	thread_node_iter iter(this->thread(i));
	while (!iter.is_empty()) {
	    this->remove_annotation(iter.step());
	}
    }
}


/**
Show all the thread bodies in the parallel structure.
*/
void
tree_par::print(ostream& s, int depth) const
{
    for (unsigned i = 0; i < this->num_threads(); i++) {
	s << "Thread #" << i << endl;
	_threads[i]->print(s, depth);
	s << endl;
    }
}


/**
Add an annotation to all the instructions in the threads pointing back to
this parallel structure.
*/
void
tree_par::annotate_threads() const
{
    tree_par::ann_pair entry_ann = make_pair((tree_par *)this, this->thread(0));
    this->add_annotation(this->entry(), &entry_ann);

    for (unsigned i = 0; i < this->num_threads(); i++) {
	tree_par::ann_pair ann = make_pair((tree_par *)this, this->thread(i));

	thread_node_iter iter(this->thread(i));
	while (!iter.is_empty()) {
	    this->add_annotation(iter.step(), &ann);
	}
    }

    this->add_annotation(this->exit(), &entry_ann);
}



/**
Add the parallel annotations to the given node.
*/
static void _add_annotation(tree_node *tn, void *x);

void
tree_par::add_annotation(p_tree_node tn, tree_par::ann_pair *ann) const
{
    D_SELFTEST_HEADER(680, "tree_par::add_annotation");

    _add_annotation(tn.ptr(), (void *)ann);
    tn->map(_add_annotation, (void *)ann);

    D_SELFTEST(680) {
	cout << "Added parallel annotation to tree node:\n";
	tn->print(); fflush(stdout);
	cout << endl;
    }

    D_SELFTEST_FOOTER(680);
}


static void _insert_in_thread(tree_par::ann_pair *thread_info, suif_object *obj);
static void _add_annotation_instr(instruction *in, void *x);

static void
_add_annotation(tree_node *tn, void *x)
{
    D_STACK(_add_annotation);
    D_SELFTEST_HEADER(700, "_add_annotation");

    _insert_in_thread((tree_par::ann_pair *)x, tn);
    if (tn->is_instr()) {
	((tree_instr *)tn)->instr_map(_add_annotation_instr, x);
    }

    D_SELFTEST(700) {
	cout << "Added parallel annotation to tree node:\n";
	tn->print(); fflush(stdout);
	cout << endl;
    }

    D_SELFTEST_FOOTER(700);
}


static void
_add_annotation_instr(instruction *in, void *x)
{
    _insert_in_thread((tree_par::ann_pair *)x, in);

    if (in->opcode() == io_cal) {
	/* If the procedure is in memory, annotate all the statements in
	 * its body.
	 */
	p_proc_sym fn = ::func_symbol(in);
	if (fn->is_in_memory()) {
	    D_SELFTEST(10) {
		cout << "Found call to a local function '" << fn->name()
		    << "'. Annotating function body.\n";
	    }

	    /* If we have already annotated this function, then the last
	     * element of the vector of tree_par parents for the function's
	     * body should be this tree_par (this is because tree_par
	     * parents are pushed on to the vector as they are found).
	     *
	     * So, if the vector is empty or we are not the last element on
	     * the vector, we recurse into the function. Otherwise, we have
	     * already visited it.
	     */
	    vector_tree_par *parents = ::get_tree_par_parents(fn->block());
	    p_tree_par this_tree_par = ((tree_par::ann_pair *)x)->first;
	    if (!parents || parents->back().ptr() != this_tree_par.ptr()) {
		_add_annotation(fn->block(), (void *)x);
		fn->block()->map(_add_annotation, (void *)x);
	    } else {
		D_SELFTEST(10) {
		    cout << "Function already annotated. Skipping.\n";
		}
	    }
	} else {
	    D_SELFTEST(10) {
		warning_line(in, "procedure %s not defined in current "
			    "file. Thread analysis not performed.\n", 
			    fn->name());
	    }
	}
    }
}


static void
_insert_in_thread(tree_par::ann_pair *thread_info, suif_object *obj)
{
    p_tree_par parent = thread_info->first;
    p_thread_body thread = thread_info->second;

    vector_tree_par *parents = ::get_tree_par_parents(obj);
    if (!parents) {
	parents = new vector_tree_par;	/* [MEMORY LEAK] */
	obj->append_annote(k_par_parents, (void *)parents);
    }
    parents->push_back(parent);

    vector_thread_body *threads = ::get_thread_bodies(obj);
    if (!threads) {
	threads = new vector_thread_body;	/* [MEMORY LEAK] */
	obj->append_annote(k_thread_bodies, (void *)threads);
    }
    threads->push_back(thread);
}




/**
Remove parallel annotations from the tree node.
*/
static void _remove_annotation(tree_node *tn, void *);

void
tree_par::remove_annotation(p_tree_node tn) const
{
    _remove_annotation(tn.ptr(), NULL);
    if (!tn->is_instr()) {
	tn->map(_remove_annotation, NULL);
    }
}

static void _remove_annotation_instr(instruction *in, void *);
static void _remove_from_thread(suif_object *obj);

static void 
_remove_annotation(tree_node *tn, void *)
{
    _remove_from_thread(tn);

    if (tn->is_instr()) {
	((tree_instr *)tn)->instr_map(_remove_annotation_instr, NULL);
    }
}

static void
_remove_annotation_instr(instruction *in, void *)
{
    _remove_from_thread(in);

    if (in->opcode() == io_cal) {
	/* If the procedure is in memory, remove annotations from its body. */
	p_proc_sym fn = ::func_symbol(in);
	if (fn->is_in_memory()) {
	    vector_tree_par *parents = ::get_tree_par_parents(fn->block());
	    if (parents) {
		_remove_annotation(fn->block(), NULL);
		fn->block()->map(_remove_annotation, NULL);
	    }
	}
    }
}


static void
_remove_from_thread(suif_object *obj)
{
    obj->get_annote(k_par_parents);
    obj->get_annote(k_thread_bodies);
}
