/**
$Id: par.cc,v 2.9 1999/03/05 14:50:49 diego Exp $
*/
#include <suif_copyright.h>
#include <suif.h>

#include "par.h"

/* Annotations used to mark cobegin/coend and parloop instructions */
char *k_cobegin;	/* Instruction is the start of a cobegin 
 			   structure. */

char *k_parloop;	/* Instruction is the start of a parloop
 			   structure. */

char *k_par_parents;	/* Pointer to set of parallel parents. A parallel
			   parent is an instruction that starts the
			   parallel structure holding this instruction. */

char *k_thread_bodies;	/* Pointer to set of thread bodies holding this
			   instruction. */


/*---------------------------------------------------------------------------
      Initialization and finalization functions for the Parallel library.
---------------------------------------------------------------------------*/
void
init_par(int & /* argc */, char * /* argv */ [])
{
    /* These two annotations will be written to the output file */
    ANNOTE(k_cobegin, "cobegin", true);
    ANNOTE(k_parloop, "parloop", true);

    /* These are temporary annotations */
    k_par_parents = lexicon->enter("cobegin_parents")->sp;
    k_thread_bodies = lexicon->enter("thread_bodies")->sp;
}


void
exit_par()
{
}

/**
Returns true if the given instruction is one of the parallel structures.
*/
bool
is_par(p_tree_node tn)
{
    return is_cobegin(tn) || is_parloop(tn);
}


/**
Returns true if the given instruction is a cobegin.
*/
bool
is_cobegin(instruction *instr)
{
    if (instr) { return (instr->peek_annote(k_cobegin) != NULL); }
    return false;
}


/**
Returns true if the given tree node is a cobegin. Note: this assumes
that the program has been processed by 'findpar' already!
*/
bool
is_cobegin(p_tree_node tn)
{
    if (!tn || !tn->is_instr()) return false;
    return is_cobegin(p_tree_instr(tn)->instr());
}


/**
Returns true if the given tree_node is a parloop statement.
*/
bool
is_parloop(p_tree_node tn)
{
    if (!tn || !tn->is_for()) return false;
    return (tn->peek_annote(k_parloop) != NULL);
}


/**
Returns the thread body associated with the given statement.
*/
vector_thread_body *
get_thread_bodies(suif_object *obj)
{
    if (!obj) return NULL;
    return (vector_thread_body *)obj->peek_annote(k_thread_bodies);
}


/**
Returns the tree_par object holding the given node. Useful when you're not
sure which type of parallel structure you're dealing with.
*/
vector_tree_par *
get_tree_par_parents(suif_object *obj)
{
    if (!obj) return NULL;
    return (vector_tree_par *)obj->peek_annote(k_par_parents);
}


/**
Returns the function symbol for the given function call.
*/
p_proc_sym
func_symbol(instruction *instr)
{
    if_ops op = instr->opcode();
    assert_msg(op == io_cal, ("Instruction #%d is not a function call!\n",
		instr->number()));

    /* Retrieve the name of the function being called */
    in_cal *call = (in_cal *)instr;
    operand addr = call->addr_op();
    assert(addr.is_instr() && addr.instr()->opcode() == io_ldc);
    p_sym_node fn = ((in_ldc *)addr.instr())->value().addr().symbol();
    assert(fn->is_proc());

    return p_proc_sym(fn);
}


/**
Returns true if any two threads from the given thread vectors can execute
concurrently
*/
bool
have_concurrent_threads(vector_thread_body *tv1, vector_thread_body *tv2)
{
    vector_thread_body::iterator i1, i2;
    for (i1 = tv1->begin(); i1 != tv1->end(); i1++) {
	p_thread_body thread1 = *i1;
	for (i2 = tv2->begin(); i2 != tv2->end(); i2++) {
	    p_thread_body thread2 = *i2;
	    if (thread1->is_concurrent_with(thread2)) {
		return true;
	    }
	}
    }

    return false;
}
