/**
$Id: ccfg2graph.cc,v 1.6 1999/02/19 20:06:53 diego Exp $

This module outputs the CCFG in any of several graph formats. The only
assumption we make out of the graph format is that it is possible to
specify the nodes and the edges of the graph separately. 

All the functions that do the actual writing to the graph file are
clustered at the end of this file. Only those functions should be
modified when adding new graph file formats.
*/
#include <vector>

#include <stdio.h>

#include <suif.h>
#include <par.h>
#include <ccfg.h>
#include <str.h>
#include <odc-util.h>

#include "ccfg2graph.h"

/* Local data */
static FILE *_f;
static p_ccfg _graph;
static GraphFormat _format;
static unsigned _numnodes;
static vector<bool> _alreadyProcessed;
static vector<int> _nodemap;
static String _indent("    ");		// Indent 4 spaces.

static void layoutNodes(bool raw);
static void layoutCtrlEdges();
static void layoutConfEdges();
static void layoutSyncEdges();

/* Functions to process nodes */
static void processNode(p_ccfg_node node);
static void processBeginNode(p_ccfg_begin node);
static void processForNode(p_ccfg_begin node, p_tree_for tf);
static void processIfNode(p_ccfg_begin node, p_tree_if ti);
static void processCobeginNode(p_ccfg_begin node, p_tree_cobegin tc);
static void processBody(p_ccfg_node first, p_ccfg_node begin, p_ccfg_node end);

/* Functions that write to the graph file */
static void writeNode(p_ccfg_node node);
static void writeBeginSubgraph(p_ccfg_node node);
static void writeEndSubgraph();
static void writeHeader();
static void writeTrailer();
static void writeCtrlEdge(p_ccfg_node source, p_ccfg_node target);
static void writeConfEdge(conflict conf);
static void writeSyncEdge(conflict conf);
static void writeEmptyLine();

/* Functions that return graphic attributes for nodes */
static char *nodeLabel(p_ccfg_node node);
static char *nodeShape(p_ccfg_node node);
static char *nodeStyle(p_ccfg_node node);


//---------------------------------------------------------------------------
// ccfg2graph()
// Generates a graph file out of the given CCFG. Several format are
// recognized and specified in the argument 'format'.
//---------------------------------------------------------------------------
void 
ccfg2graph(FILE *file, p_ccfg graph, GraphFormat format, bool raw)
{
    unsigned i;

    _f = file;

    _graph = graph;
    _numnodes = _graph->num_nodes();
    _format = format;

    _graph->find_dominators();
    _graph->find_postdominators();

    _alreadyProcessed.reserve(_numnodes);
    _nodemap.reserve(_numnodes);

    for (i = 0; i < _numnodes; i++) {
	_alreadyProcessed[i] = FALSE;
	_nodemap[i] = i;
    }

    /* Write down the header */
    writeHeader();

    /* Put down nodes */
    layoutNodes(raw);
    writeEmptyLine();

    /* Put down edges and backedges */
    layoutCtrlEdges();
    writeEmptyLine();

    /* Put down conflict edges */
    layoutConfEdges();
    writeEmptyLine();

    /* Put down synchronization edges */
    layoutSyncEdges();
    writeEmptyLine();

    /* Write the trailer */
    writeTrailer();
}

//---------------------------------------------------------------------------
// layoutNodes()
// If the argument 'raw' is TRUE, no attempt is made to group
// nodes in high-level control structures like ifs and loops.
//---------------------------------------------------------------------------
void
layoutNodes(bool raw)
{
    if (raw == FALSE) {
	processNode(_graph->node(0));
    } else {
	unsigned i;
	for (i = 0; i < _numnodes; i++) {
	    writeNode(_graph->node(i));
	}
    }
}


//---------------------------------------------------------------------------
// layoutCtrlEdges()
// Connect all the nodes in the CCFG.
//---------------------------------------------------------------------------
void
layoutCtrlEdges()
{
    p_ccfg_node node;
    unsigned i;
    
    for (i = 0; i < _numnodes; i++) {
	node = _graph->node(i);

	ccfg_node_list_iter succ_iter(node->succs());
	while (!succ_iter.is_empty()) {
	    p_ccfg_node succ = succ_iter.step();

	    if (_nodemap[node->number()] != node->number() &&
		_nodemap[succ->number()] != succ->number()) {
		continue;	// Ignore aliased source & target nodes.
	    }

	    p_ccfg_node alias_node = _graph->node(_nodemap[node->number()]);
	    p_ccfg_node alias_succ = _graph->node(_nodemap[succ->number()]);

	    if (alias_node->number() == 0 && alias_succ->number() == 1) {
		continue;	// Don't add BEGIN->END edge.
	    }

	    if (alias_node->number() == alias_succ->number()) {
		continue;	// Ignore self-edges introduced by aliasing.
	    }

	    writeCtrlEdge(alias_node, alias_succ);
	}
    }
}


//---------------------------------------------------------------------------
// layoutConfEdges()
// Add all the conflict edges
//---------------------------------------------------------------------------
void
layoutConfEdges()
{
    p_ccfg_node node;
    int i;
    
    for (i = 0; i < _numnodes; i++) {
	node = _graph->node(i);

	if (_nodemap[i] != i) {
	    continue;		// Ignore aliased source nodes.
	}

	list_conflict::iterator iter = node->conflicts().begin();
	for (; iter != node->conflicts().end(); iter++) {
	    conflict conf = *iter;
	    if (conf.tail()->isSync()) {
		continue;
	    }

	    /* Only show one edge per conflict. If the source node
	     * number is greater than the target of the conflict, then this
	     * edge has already been placed.
	     */
	    int source = conf.tail()->node()->number();
	    int target = conf.head()->node()->number();
	    if (source > target) {
		continue;
	    }

	    writeConfEdge(conf);
	}
    }
}


//---------------------------------------------------------------------------
// layoutSyncEdges()
// Add all the synchronization edges
//---------------------------------------------------------------------------
void
layoutSyncEdges()
{
    p_ccfg_node node;
    int i;
    
    for (i = 0; i < _numnodes; i++) {
	node = _graph->node(i);

	if (_nodemap[i] != i) {
	    continue;		// Ignore aliased source nodes.
	}

	list_conflict::iterator iter = node->conflicts().begin();
	for (; iter != node->conflicts().end(); iter++) {
	    conflict conf = *iter;
	    if (!conf.tail()->isTSync() && !conf.tail()->isUnlock()) {
		continue;
	    }

	    writeSyncEdge(conf);
	}
    }
}


//***************************************************************************
//			 Internal helper functions
//***************************************************************************
//---------------------------------------------------------------------------
// processNode()
// Top level node placing function. It marks the node processed and calls
// the appropriate handler. All the process...() functions call this
// function when they need to write out a node.
//---------------------------------------------------------------------------
void
processNode(p_ccfg_node node)
{
    if (_alreadyProcessed[node->number()]) {
	return;
    }

    _alreadyProcessed[node->number()] = TRUE;

    if (node->kind() == CCFG_BEGIN) {
	processBeginNode(p_ccfg_begin(node));
    } else {
	writeNode(node);
    }
}

//---------------------------------------------------------------------------
// processBeginNode()
// Puts all the nodes inside the block of code delimited by the given BEGIN
// node and its corresponding END node.
//---------------------------------------------------------------------------
void
processBeginNode(p_ccfg_begin begin)
{
    /*
     * First check for control structures. Each of these call specialized
     * functions that place all the nodes within the control structure.
     */
    tree_kinds kind = begin->node()->kind();

    if (kind == TREE_FOR) {
	p_tree_for tf = p_tree_for(begin->node());
	processForNode(begin, tf);
    } else if (kind == TREE_IF) {
	p_tree_if ti = p_tree_if(begin->node());
	processIfNode(begin, ti);
    } else if (begin->is_cobegin()) {
	p_tree_cobegin tc(new tree_cobegin(begin->node()));
	processCobeginNode(begin, tc);
    } else {
	p_ccfg_node first = _graph->node(begin->number() + 1);
	p_ccfg_end end = begin->companion();

	/* Put the BEGIN node */
	writeNode(begin);

	/* Put the body */
	processBody(first, begin, end);

	/* Put the END node */
	processNode(end);
    }
}


//---------------------------------------------------------------------------
// processForNode()
// Writes down the nodes for the for loop starting at node. We skip most
// nodes in the for loop header (LB, UB and STEP) and only write down the
// TEST node of the loop. We map all the skipped nodes to the TEST node so
// the edges are placed properly later on. Finally, we write down all the
// nodes in the body of the loop.
//---------------------------------------------------------------------------
void
processForNode(p_ccfg_begin begin, p_tree_for tf)
{
    p_ccfg_test test = ccfg_test_node(tf);
    p_ccfg_node first = get_node(tf->body()->head()->contents);
    p_ccfg_end end = begin->companion();
    int nodenum = begin->number();

    /* Map all the nodes in the header of the loop to the BEGIN node and
     * marked them visited. The nodes in a for loop are created in this
     * order: BEGIN, LB, UB, STEP, LPAD, TEST and body. We only map the
     * nodes TEST, LB, UB and STEP.
     */
    _nodemap[test->number()] = begin->number();
    _nodemap[nodenum + 1] = begin->number();
    _nodemap[nodenum + 2] = begin->number();
    _nodemap[nodenum + 3] = begin->number();

    _alreadyProcessed[test->number()] = TRUE;
    _alreadyProcessed[nodenum + 1] = TRUE;
    _alreadyProcessed[nodenum + 2] = TRUE;
    _alreadyProcessed[nodenum + 3] = TRUE;
   
    /* Put the BEGIN node. Don't call processNode() here to avoid infinite
     * recursion.
     */
    writeNode(begin);

    /* Put the body of the loop */
    processBody(first, begin, end);

    /* Put the END node */
    processNode(end);
}


//---------------------------------------------------------------------------
// processIfNode()
// Writes down all the nodes in an if statement. 
//---------------------------------------------------------------------------
void
processIfNode(p_ccfg_begin begin, p_tree_if ti)
{
    int beginnum = begin->number();

    /* Map the header node for the if statement to the BEGIN node */
    _nodemap[beginnum + 2] = begin->number();
    _alreadyProcessed[beginnum + 2] = TRUE;

    /* Put the BEGIN node. Don't call processNode() here to avoid infinite
     * recursion.
     */
    writeNode(begin);

    /* Put the THEN body. The nodes in the THEN body are all the nodes
     * dominated by the first node of the body and postdominated by the END
     * node of the if statement.
     */
    p_ccfg_node first = get_node(ti->then_part()->head()->contents);
    p_ccfg_end end = begin->companion();

    processBody(first, begin, end);

    /* Put the ELSE body if it is not empty */
    if (!ti->else_part()->is_empty()) {
	first = get_node(ti->else_part()->head()->contents);

	processBody(first, begin, end);
    }

    /* Put the END node */
    processNode(end);
}


//---------------------------------------------------------------------------
// processCobeginNode()
//---------------------------------------------------------------------------
void
processCobeginNode(p_ccfg_begin begin, p_tree_cobegin tc)
{
    int beginnum = begin->number();
    p_ccfg_end end = begin->companion();

    /* Put the COBEGIN node. Don't call processNode() here to avoid infinite
     * recursion.
     */
    writeNode(begin);

    /* Put all the thread bodies */
    unsigned i;
    for (i = 0; i < tc->num_threads(); i++) {
	p_ccfg_node first = get_node(tc->thread(i)->first());
	p_ccfg_node thread_end = get_node(tc->thread(i)->last());

	processBody(first, begin, thread_end);
    }

    /* Put the COEND node */
    processNode(end);
}


//---------------------------------------------------------------------------
// processBody()
// Writes all the nodes inside the body starting at the given first node
// and delimited by the nodes begin and end.
//---------------------------------------------------------------------------
void
processBody(p_ccfg_node first, p_ccfg_node begin, p_ccfg_node end)
{
    if (!first || !begin || !end) return;

    writeBeginSubgraph(begin);

    /* Increase indentation level */
    String previndent(_indent);
    _indent += "    ";

    unsigned i;
    for (i = first->number(); i < _numnodes; i++) {
	p_ccfg_node node = _graph->node(i);
	if (_graph->dominates(begin, node)	&&
	    _graph->postdominates(end, node)	&&
	    node != begin			&&
	    node != end				) {
	    processNode(_graph->node(i));
	}
    }

    /* Return to previous indentation level */
    _indent = previndent;

    writeEndSubgraph();
}




//***************************************************************************
//			 Graph dependent functions. 
//
// These functions do the actual writing to the graph file. Modify the
// switch() statements when creating new formats.
//***************************************************************************
//---------------------------------------------------------------------------
// writeHeader()
//---------------------------------------------------------------------------
void
writeHeader()
{
    switch (_format) {
	case DOT: {
	    fprintf(_f, "digraph ccfg {\n");
	    fprintf(_f, "%scenter=1;\n", _indent());
	    fprintf(_f, "%sordering=out;\n", _indent());
	    fprintf(_f, "%ssize=\"6,9\";\n", _indent());
	    fprintf(_f, "\n");
	    break;
	}

	case VCG: {
	    fprintf(_f, "graph: { title: \"CCFG_GRAPH\"\n");
	    fprintf(_f, "%sarrow_mode: fixed\n", _indent());
	    fprintf(_f, "%s//dirty_edge_labels:	yes\n", _indent());
	    fprintf(_f, "%s//display_edge_labels: yes\n", _indent());
	    fprintf(_f, "%sedge.color: red\n", _indent());
	    fprintf(_f, "%sedge.thickness: 2\n", _indent());
	    fprintf(_f, "%sfinetuning:  no\n", _indent());
	    fprintf(_f, "%s//hidden:  2\n", _indent());
	    fprintf(_f, "%s//hidden:  3\n", _indent());
	    fprintf(_f, "%s//late_edge_labels: no\n", _indent());
	    fprintf(_f, "%slayoutalgorithm: maxdepth\n", _indent());
	    fprintf(_f, "%s//layout_downfactor: 15\n", _indent());
	    fprintf(_f, "%snearedges:  yes\n", _indent());
	    fprintf(_f, "%snode.bordercolor: blue\n", _indent());
	    fprintf(_f, "%snode.borderwidth: 3\n", _indent());
	    fprintf(_f, "%snode.color:  blue\n", _indent());
	    fprintf(_f, "%snode.textcolor: white\n", _indent());
	    fprintf(_f, "%sport_sharing:  yes\n", _indent());
	    fprintf(_f, "%spriority_phase: yes\n", _indent());
	    fprintf(_f, "%s//xspace:  40\n", _indent());
	    fprintf(_f, "%s//yspace:  20\n", _indent());
	    fprintf(_f, "\n");
	    break;
	}

	case GML: {
	    fprintf(_f, "Creator \"ccfg2graph\"\n");
	    fprintf(_f, "graph [\n");
	    fprintf(_f, "%sdirected 1\n", _indent());
	    fprintf(_f, "\n");
	    break;
	}

	default: {
	    printf("Unrecognized graph format %d\n", _format);
	    exit(1);
	}
    }
}


//---------------------------------------------------------------------------
// writeTrailer()
//---------------------------------------------------------------------------
void
writeTrailer()
{
    switch (_format) {
	case DOT: {
	    fprintf(_f, "}\n");
	    break;
	}

	case VCG: {
	    fprintf(_f, "}\n");
	    break;
	}

	case GML: {
	    fprintf(_f, "]\n");
	    break;
	}

	default: {
	    printf("Unrecognized graph format %d\n", _format);
	    exit(1);
	}
    }
}


//---------------------------------------------------------------------------
// writeNode()
// Write out an entry for the given node.
//---------------------------------------------------------------------------
void
writeNode(p_ccfg_node node)
{
    switch (_format) {
	case DOT: {
	    fprintf(_f, "%sn%d [", _indent(), node->number());
	    fprintf(_f, "shape=%s, ", nodeShape(node));
	    char *style = nodeStyle(node);
	    if (style != NULL) {
		fprintf(_f, "style=%s, ", style);
	    }
	    fprintf(_f, "label=\"%s\"", nodeLabel(node));
	    fprintf(_f, "];\n");
	    break;
	}

	case VCG: {
	    fprintf(_f, "%snode: {", _indent());
	    fprintf(_f, "title: \"%d\" ", node->number());
	    fprintf(_f, "shape: %s ", nodeShape(node));
	    fprintf(_f, "label: \"%s\"", nodeLabel(node));
	    fprintf(_f, "}\n");
	    break;
	}

	case GML: {
	    fprintf(_f, "%snode [ ", _indent());
	    fprintf(_f, "id %d ", node->number());
	    fprintf(_f, "label \"%s\" ", nodeLabel(node));
 	    fprintf(_f, "graphics [ ");
 	    fprintf(_f, "type \"%s\" ", nodeShape(node));
 	    fprintf(_f, "]");
	    fprintf(_f, "]\n");
	    break;
	}

	default: {
	    printf("Unrecognized graph format %d\n", _format);
	    exit(1);
	}
    }
}


//---------------------------------------------------------------------------
// writeBeginSubgraph()
// Marks the start of a subgraph in the output file.
//---------------------------------------------------------------------------
void
writeBeginSubgraph(p_ccfg_node begin)
{
    switch (_format) {
	case DOT: {
	    fprintf(_f, "%s{\n", _indent());
	    break;
	}

	case VCG: {
	    static int _graphid = 0;
	    fprintf(_f, "%sgraph: {\n", _indent());
	    fprintf(_f, "%s%stitle: \"(%d-%03d) %s\"\n", _indent(), _indent(), 
		    begin->number(), _graphid, nodeLabel(begin));
	    fprintf(_f, "%s%sfolding: 0\n", _indent(), _indent());
	    fprintf(_f, "%s%sshape: %s\n", _indent(), _indent(),
		    nodeShape(begin));
	    fprintf(_f, "\n");
	    _graphid++;
	    break;
	}

	case GML: {
	    break;
	}

	default: {
	    printf("Unrecognized graph format %d\n", _format);
	    exit(1);
	}
    }
}


//---------------------------------------------------------------------------
// writeEndSubgraph()
// Marks the end of a subgraph in the output file.
//---------------------------------------------------------------------------
void
writeEndSubgraph()
{
    switch (_format) {
	case DOT: {
	    fprintf(_f, "%s}\n", _indent());
	    break;
	}

	case VCG: {
	    fprintf(_f, "%s}\n", _indent());
	    break;
	}

	case GML: {
	    break;
	}

	default: {
	    printf("Unrecognized graph format %d\n", _format);
	    exit(1);
	}
    }
}


//---------------------------------------------------------------------------
// writeCtrlEdge()
//---------------------------------------------------------------------------
void
writeCtrlEdge(p_ccfg_node source, p_ccfg_node target)
{
    char *style;
    if (source->is_cobegin() || target->is_coend()) {
	style = "dashed";
    } else {
	style = "solid";
    }

    switch (_format) {
	case DOT: {
	    fprintf(_f, "%sn%d -> n%d [", _indent(), source->number(),
		    target->number());
	    fprintf(_f, "style=%s, ", style);
	    fprintf(_f, "dir=forward,");
	    fprintf(_f, "minlen=1");
	    fprintf(_f, "];\n");
	    break;
	}

	case VCG: {
	    fprintf(_f, "%sedge: {", _indent());
	    fprintf(_f, "class: 1 ");
	    fprintf(_f, "priority: 80 ");
	    fprintf(_f, "sourcename: \"%d\" ", source->number());
	    fprintf(_f, "targetname: \"%d\" ", target->number());
	    fprintf(_f, "linestyle: %s", style);
	    fprintf(_f, "}\n");
	    break;
	}

	case GML: {
	    fprintf(_f, "%sedge [ ", _indent());
	    fprintf(_f, "source %d ", source->number());
	    fprintf(_f, "target %d ", target->number());
 	    fprintf(_f, "graphics [ ");
 	    fprintf(_f, "type \"line\" ");
 	    fprintf(_f, "arrow \"last\" ");
 	    fprintf(_f, "]");
	    fprintf(_f, "]\n");
	    break;
	}
    }
}


//---------------------------------------------------------------------------
// writeConfEdge()
//---------------------------------------------------------------------------
void
writeConfEdge(conflict conf)
{
    p_ccfg_node source = _graph->node(_nodemap[conf.tail()->node()->number()]);
    p_ccfg_node target = _graph->node(_nodemap[conf.head()->node()->number()]);

    switch (_format) {
	case DOT: {
	    fprintf(_f, "%sn%d -> n%d [", _indent(), source->number(),
		    target->number());
	    fprintf(_f, "style=dotted, ");
	    fprintf(_f, "dir=both, ");
	    fprintf(_f, "constraint=false, ");
	    fprintf(_f, "weight=0, ");
	    fprintf(_f, "label=\"%s%s(%s)\"", conf.tail()->ckind(), 
		    conf.head()->ckind(), conf.tail()->var()->name());
	    fprintf(_f, "];\n");
	    break;
	}

	case VCG: {
	    fprintf(_f, "%sedge: {", _indent());
	    fprintf(_f, "sourcename: \"%d\" ", source->number());
	    fprintf(_f, "targetname: \"%d\" ", target->number());
	    fprintf(_f, "linestyle: dashed ");
	    fprintf(_f, "color: green ");
	    fprintf(_f, "class: 2 ");
	    fprintf(_f, "priority: 0 ");
	    fprintf(_f, "arrowstyle: solid ");
	    fprintf(_f, "arrowsize: 10 ");
	    fprintf(_f, "backarrowstyle: solid ");
	    fprintf(_f, "backarrowsize: 10 ");
	    fprintf(_f, "label: \"%s%s(%s)\"", conf.tail()->ckind(),
		    conf.head()->ckind(), conf.tail()->var()->name()
		    );
	    fprintf(_f, "}\n");
	    break;
	}

	case GML: {
	    fprintf(_f, "%sedge [ ", _indent());
	    fprintf(_f, "source %d ", source->number());
	    fprintf(_f, "target %d ", target->number());
	    fprintf(_f, "label  \"%s%s(%s)\" ", conf.tail()->ckind(),
		    conf.head()->ckind(), conf.tail()->var()->name()
		    );
 	    fprintf(_f, "graphics [ ");
 	    fprintf(_f, "type \"line\" ");
 	    fprintf(_f, "arrow \"last\" ");
 	    fprintf(_f, "]");
	    fprintf(_f, "]\n");
	    break;
	}
    }
}

//---------------------------------------------------------------------------
// writeSyncEdge()
//---------------------------------------------------------------------------
void
writeSyncEdge(conflict conf)
{
    p_ccfg_node source = _graph->node(_nodemap[conf.tail()->node()->number()]);
    p_ccfg_node target = _graph->node(_nodemap[conf.head()->node()->number()]);

    char *dir;
    if (conf.tail()->isUnlock()) {
	dir = "none";
    } else {
	dir = "forward";
    }

    switch (_format) {
	case DOT: {
	    fprintf(_f, "%sn%d -> n%d [", _indent(), source->number(),
		    target->number());
	    fprintf(_f, "style=dashed, ");
	    fprintf(_f, "dir=%s, ", dir);
	    fprintf(_f, "constraint=false, ");
	    fprintf(_f, "weight=0");
// 	    fprintf(_f, "label=\"%s-%s(%s)\"", conf.tail()->ckind(), 
// 		    conf.head()->ckind(), conf.tail()->var()->name());
	    fprintf(_f, "];\n");
	    break;
	}

	case VCG: {
	    fprintf(_f, "%sedge: {", _indent());
	    fprintf(_f, "sourcename: \"%d\" ", source->number());
	    fprintf(_f, "targetname: \"%d\" ", target->number());
	    fprintf(_f, "linestyle: dashed ");
	    fprintf(_f, "color: blue ");
	    fprintf(_f, "class: 3 ");
	    fprintf(_f, "priority: 0 ");
	    if (conf.tail()->isUnlock()) {
		fprintf(_f, "arrowstyle: none ");
	    }
// 	    fprintf(_f, "label: \"%s%s(%s)\"", conf.tail()->ckind(),
// 		    conf.head()->ckind(), conf.tail()->var()->name()
// 		    );
	    fprintf(_f, "}\n");
	    break;
	}

	case GML: {
	    fprintf(_f, "%sedge [ ", _indent());
	    fprintf(_f, "source %d ", source->number());
	    fprintf(_f, "target %d ", target->number());
	    fprintf(_f, "label \"%s%s(%s)\" ", conf.tail()->ckind(),
		    conf.head()->ckind(), conf.tail()->var()->name()
		    );
 	    fprintf(_f, "graphics [ ");
 	    fprintf(_f, "type \"line\" ");
 	    if (conf.tail()->isUnlock()) {
 		fprintf(_f, "arrow \"none\" ");
 	    } else {
 		fprintf(_f, "arrow \"last\" ");
 	    }
 	    fprintf(_f, "]");
	    fprintf(_f, "]\n");
	    break;
	}
    }
}


//---------------------------------------------------------------------------
// writeEmptyLine()
//---------------------------------------------------------------------------
void
writeEmptyLine()
{
    fprintf(_f, "\n");
}


//---------------------------------------------------------------------------
// nodeShape()
// returns a string with the shape of the given node.
//---------------------------------------------------------------------------
char *
nodeShape(p_ccfg_node node)
{
    static char *dot_shapes[] = { "ellipse", "diamond", "box", "circle" };
    static char *vcg_shapes[] = { "ellipse", "rhomb", "box", "ellipse" };
    static char *gml_shapes[] = { "oval", "polygon", "rectangle", "oval" };
    char **shapes = NULL;

    switch (_format) {
	case DOT: shapes = dot_shapes; break;
	case VCG: shapes = vcg_shapes; break;
	case GML: shapes = gml_shapes; break;
    }

    ccfg_node_kind kind = node->kind();

    if (node->is_cobegin() || node->is_coend()) {
	return shapes[0];
    } else if (kind == CCFG_BEGIN || kind == CCFG_END) {
	tree_kinds tk = p_ccfg_begin(node)->node()->kind();
	if (tk == TREE_IF && kind == CCFG_BEGIN) {
	    return shapes[1];
	} else {
	    return shapes[0];
	}
    } else if (kind == CCFG_INSTR || kind == CCFG_BLOCK) {
	return shapes[2];
    } else if (kind == CCFG_TEST) {
	return shapes[1];
    } else { /* kind == CCFG_LABEL */
	return shapes[3];
    }
}

//---------------------------------------------------------------------------
// nodeLabel()
// returns a string with the label for the given node. Typically this
// string contains the corresponding source code lines for the node.
//---------------------------------------------------------------------------
char *
nodeLabel(p_ccfg_node node)
{
    String label("");
    String fname(_dirname + "/" + source_file_name(node->last_instr().ptr()));
    ccfg_node_kind kind = node->kind();

    if (node->is_cobegin()) {
	label += "cobegin";
    } else if (node->is_coend()) {
	label += "coend";
    } else if (kind == CCFG_BEGIN && node->number() == 0) {
	label += "begin"; 
    } else if (kind == CCFG_END) {
	label += "end"; 
	p_ccfg_end end = node;
	switch (end->node()->kind()) {
	    case TREE_IF: label += "if"; break;
	    case TREE_FOR: label += "for"; break;
	    case TREE_LOOP: label += "loop"; break;
	}
    } else if (kind == CCFG_LABEL) {
	label += "label";
    } else {
	unsigned firstline, lastline;

	if (kind == CCFG_INSTR) {
	    p_tree_instr ti = p_ccfg_instr(node)->instr();
	    firstline = lastline = source_line_num(ti.ptr());
	} else if (kind == CCFG_BEGIN) {
	    firstline = source_line_num(p_ccfg_begin(node)->node().ptr());
	    lastline = firstline;
	} else if (kind == CCFG_BLOCK) {
	    p_ccfg_block block = node;
	    firstline = lastline = source_line_num(block->first_exec().ptr());
	    ccfg_node_instr_iter iter(block.ptr(), FALSE, TRUE);
	    while (!iter.is_empty()) {
		p_tree_instr ti = iter.step();
		lastline = source_line_num(ti.ptr());
	    }
	} else if (kind == CCFG_TEST) {
	    p_ccfg_test test = node;
	    p_tree_for tf = test->for_loop();
	    firstline = lastline = source_line_num(tf.ptr());
	}

	label += file2String((char *)fname(), firstline, lastline, DF_QUOTES | 
		    DF_LTRIM | DF_XLATE | DF_TRIMLAST);
	
	/* For the DOT format, specify left justification by replacing \n
	 * with \l's. Add two spaces to move the label away from the node's
	 * border.
	 */
	if (_format == DOT) {
	    label.Replace("\\n", "\\l  ");

	    /* On multi-line labels, make the last line left-justified and
	     * add two spaces at the start of the label.
	     */
	    if (lastline > firstline) {
		label.Insert(0, "  ");
		label += "\\l";
	    }
	}
    }

    if (_show_nodenum) {
	label.Insert(0, String("(") + String(node->number()) + ") ");
    }

    return (char *)label();
}


//---------------------------------------------------------------------------
// nodeStyle()
// returns a string with the style of the given node.
//---------------------------------------------------------------------------
char *
nodeStyle(p_ccfg_node node)
{
    if (_format == DOT) {
	if (node->is_cobegin() || node->is_coend()) {
	    return "bold";
	} else if (node->kind() == CCFG_BEGIN || node->kind() == CCFG_END) {
	    return "bold";
	}
    }

    return NULL;
}
