/**
$Id: main.cc,v 2.4 1999/02/09 18:03:51 diego Exp $
*/
#include <suif.h>
#include <par.h>
#include <ccfg.h>
#include <odc-util.h>

#include "ccfg2graph.h"

bool _show_nodenum;
String _dirname;
FILE *STDOUT;
FILE *_f;

static boolean _help;
static char *_format;
static char *_output_name;
static boolean _split;
static boolean _raw;

static cmd_line_option options[] = {
    { CLO_NOARG, "-h", "", &_help },
    { CLO_NOARG, "-help", "", &_help },
    { CLO_STRING, "-g", "dot", &_format },
    { CLO_STRING, "-o", "", &_output_name },
    { CLO_NOARG, "-split", "", &_split },
    { CLO_NOARG, "-num", "", &_show_nodenum },
    { CLO_NOARG, "-raw", "", &_raw }
};


//---------------------------------------------------------------------------
// usage()
// Shows usage information and exits.
//---------------------------------------------------------------------------
void
usage()
{
    printf("\n");
    printf("Usage: ccfg2graph [flags] file.odc\n");
    printf("\n");
    printf("Writes the CCFG for file.odc to a graph file or stdout.\n");
    printf("\n");
    printf("-g {dot|vcg|gml}: Format of the output file (default = dot).\n");
    printf("-o name:      Name of the output file. If missing, the graph goes to stdout.\n");
    printf("-split:       Write each function to a separate file. If a file name is given\n");
    printf("              with -o, the name of the function will be used as a prefix.\n");
    printf("              This option has no effect if -o is not used.\n");
    printf("-num:         Includes node numbers in the graph.\n");
    printf("-raw:         Write *all* the nodes, ignoring high-level constructs.\n");
    printf("-h:           Displays this message and quits.\n");
    printf("\n");
    exit(1);
}


//---------------------------------------------------------------------------
// do_proc()
// Builds a CCFG for the given procedure and writes it out in the graph
// format specified by the string 'format'.
//---------------------------------------------------------------------------
void 
do_proc(tree_proc *tp)
{
    p_ccfg graph(new ccfg(tp));;
    proc_sym *psym = tp->proc();

    /* Check whether we have to open a new graph file or reuse the existing
     * one.
     */
    if (_split && strlen(_output_name) > 0) {
	String fname(_output_name);
	fname.Insert(0, String(psym->name()) + "-");
	_f = fopen(fname(), "w");
	if (!_f) {
	    perror(fname());
	    exit(1);
	}
    }

    /* Determine what format should be used to show the graph */
    GraphFormat fmt;
    if (strcmp(_format, "dot") == 0) {
	fmt = DOT;
    } else if (strcmp(_format, "vcg") == 0) {
	fmt = VCG;
    } else if (strcmp(_format, "gml") == 0) {
	fmt = GML;
    } else {
	printf("ERROR: Unrecognized graph format '%s'\n", _format);
	usage();
    }

    ccfg2graph(_f, graph, fmt, _raw);

    /* Close the file if we have to write each function to a different file */
    if (_split && strlen(_output_name) > 0) {
	fclose(_f);
    } else {
	/* Otherwise leave some blank space before the next graph is
	 * written.
	 */
	fprintf(_f, "\n");
	fprintf(_f, "\n");
    }
}


//---------------------------------------------------------------------------
// main()
//---------------------------------------------------------------------------
main(int argc, char *argv[])
{
    parse_cmd_line(argc, argv, options,
	    sizeof(options)/sizeof(cmd_line_option));

    if (_help) {
	usage();
    } 
    
    LIBRARY(par, init_par, exit_par);
    LIBRARY(cssame, init_cssame, exit_cssame);
    init_suif(argc, argv);

    STDOUT = stdout;

    /* If the output goes to a file, open the file before iterating
     * over the procedures in the program. We only open the file here if
     * the output goes to a single file, otherwise the procedure iterator
     * will open and close each file in turn.
     */
    if (strlen(_output_name) > 0) {
	if (!_split) {
	    _f = fopen(_output_name, "w");
	    if (!_f) {
		perror(_output_name);
		exit(1);
	    }
	}
    } else {
	_f = stdout;
    }

    String filename(argv[1]);
    _dirname = filename.dirname();

    suif_proc_iter(argc, argv, do_proc);

    if (!_split && strlen(_output_name) > 0) {
	fclose(_f);
    }

    return 0;
}
