/**
$Id: wrappers.h,v 1.3 1999/03/02 18:38:46 diego Exp $

Wrapper classes for some of the SUIF base classes. These wrappers define
some methods and operators not defined by SUIF. They're useful to build STL
containers.
*/
#ifndef __WRAPPERS_H
#define __WRAPPERS_H

#include <iostream.h>

#include <string.h>

#include <suif.h>


/**
Pointer wrapper class. This is the most commonly used one. Instead of
building a container of SUIF object pointer, build a container of
pWrappers. Eg,

set<pWrapper<var_sym *>> _setOfVars;
var_sym *var;
_setOfVars.insert(var);

Messages to the pointer are forwarded via operator->():

pWrapper wVar;
wVar->print();	// Invokes var_sym::print()


Notice that the comparison operator '<' compares the
pointers, not the pointed objects. There are two ways around it:

1- Have a wrapper class for each specific SUIF class so that we could have
   a meaningful comparison.

2- When you build a container with pWrappers, specify a less-than structure
   with the boolean operator(). Eg,

   struct ltvars {
	bool operator()(var_sym *v1, var_sym *v2) {
	    return ::strcmp(v1->name(), v2->name()) < 0;
	}
    };


We are using strategy (1). There is a base pWrapper class that includes all
the required machinery, except that its default operator<() and
operator==() work by comparing the pointers, not the objects.

To build smarter wrappers, sub-class the main pWrapper class and overload
operator<() and operator==(). Smart pointer wrappers for some common SUIF
classes are included.
*/
template <class T>
class pWrapper {
public:
    /* Building a wrapper out of a pointer to the original type */
    pWrapper(T p = 0) : _p(p) {}
    pWrapper& operator=(T p) { _p = p; return *this; }
    virtual ~pWrapper() {}

    /* Copy constructor and assignment operator for pWrappers */
    pWrapper(const pWrapper& w) : _p(w._p) {}
    pWrapper& operator=(const pWrapper& w) { _p = w._p; return *this; }

    /* Arrow (->) operator forwards messages to the wrapped pointer object. */
    T operator->() const { return _p; }

    /* Return the wrapped pointer. We should get rid of this eventually. */
    T ptr() const { return _p; }

    operator bool() const { return (_p != NULL); }

    /* Comparison operators. NOTE: They work on the pointers! Subclass if
     * you need something more meaningful.
     */
    bool operator<(const pWrapper& w) const { return _p < w._p; }
    bool operator==(const pWrapper& w) const { return _p == w._p; }

protected:
    T _p;
};




/**
Standard stream insert operator. NOTE: This assumes that your class
understands the print() method! We are "forced" to assume this instead of
the more standard operator<< because SUIF classes only provide a print()
method.
*/
template <class T>
inline ostream& operator<<(ostream& s, const pWrapper<T>& w) {
    if (w) {
	w->print();
    } else {
	s << "[NULL]";
    }
    return s;
}



/**
Pointer wrapper for pointers to SUIF symbols. Redefine operator<() and
operator==() to compare the symbol names.
*/
template <class T>
class pWrapperSym : public pWrapper<T> {
public:
    pWrapperSym(T v = 0) : pWrapper<T>(v) {}
    pWrapperSym(const pWrapperSym& w) : pWrapper<T>(w) {}
    virtual ~pWrapperSym() {}

    bool operator<(const pWrapperSym& w) const {
	if (!_p || !w._p) { return false; }
	return (::strcmp(_p->name(), w._p->name()) < 0);
    }

    bool operator==(const pWrapperSym& w) const {
	if (!_p || !w._p) { return false; }
	return (::strcmp(_p->name(), w._p->name()) == 0);
    }
};


/**
Pointer wrapper for pointers to Odyssey objects. These objects provide
operator<(), operator==() and output operators.
*/
template <class T>
class pWrapperStd : public pWrapper<T> {
public:
    pWrapperStd(T o = 0) : pWrapper<T>(o) {}
    pWrapperStd(const pWrapperStd& o) : pWrapper<T>(o) {}
    virtual ~pWrapperStd() {}

    bool operator<(const pWrapperStd& o) const {
	if (!_p || !o._p) { return false; }
	return *_p < *(o._p);
    }

    bool operator==(const pWrapperStd& o) const {
	if (!_p || !o._p) { return false; }
	return *_p == *(o._p);
    }
};

/**
Standard stream insert operator for Odyssey pointer wrappers.
*/
template <class T>
ostream& operator<<(ostream& s, const pWrapperStd<T>& w) {
    if (w) {
	s << *(w.ptr());
    } else {
	s << "[NULL]";
    }
    return s;
}



/****************************************************************************
	  Wrappers for some of the most commonly used SUIF hierarchies
****************************************************************************/
/*---------------------------------------------------------------------------
		 Wrappers for SUIF symbol and variable objects
---------------------------------------------------------------------------*/
class p_sym_node : public pWrapperSym<sym_node *> {
public:
    p_sym_node(sym_node *p = 0) : pWrapperSym<sym_node *>(p) {}
    p_sym_node(const p_sym_node& o) : pWrapperSym<sym_node *>(o) {}
};

class p_label_sym : public p_sym_node {
public:
    p_label_sym(label_sym *p = 0) : p_sym_node((sym_node *)p) {}
    p_label_sym(const p_label_sym& o) : p_sym_node(o) {}
    p_label_sym(const p_sym_node& o) : p_sym_node(o) {}
    label_sym *operator->() const { return (label_sym *)_p; }
//     label_sym *operator()() const { return (label_sym *)_p; }
};

class p_var_sym : public p_sym_node {
public:
    p_var_sym(var_sym *p = 0) : p_sym_node((sym_node *)p) {}
    p_var_sym(const p_var_sym& o) : p_sym_node(o) {}
    p_var_sym(const p_sym_node& o) : p_sym_node(o) {}
    var_sym *operator->() const { return (var_sym *)_p; }
//     var_sym *operator()() const { return (var_sym *)_p; }
};

class p_proc_sym : public p_sym_node {
public:
    p_proc_sym(proc_sym *p = 0) : p_sym_node((sym_node *)p) {}
    p_proc_sym(const p_proc_sym& o) : p_sym_node(o) {}
    p_proc_sym(const p_sym_node& o) : p_sym_node(o) {}
    proc_sym *operator->() const { return (proc_sym *)_p; }
//     proc_sym *operator()() const { return (proc_sym *)_p; }
};



/*---------------------------------------------------------------------------
			  Wrappers for SUIF tree nodes
---------------------------------------------------------------------------*/
class p_tree_node : public pWrapper<tree_node *> {
public:
    p_tree_node(tree_node *p = 0) : pWrapper<tree_node *>(p) {}
    p_tree_node(const p_tree_node& o) : pWrapper<tree_node *>(o) {}
};

class p_tree_block : public p_tree_node {
public:
    p_tree_block(tree_block *p = 0) : p_tree_node(p) {}
    p_tree_block(const p_tree_block& o) : p_tree_node(o) {}
    p_tree_block(const p_tree_node& o) : p_tree_node(o) {}
    tree_block *operator->() const { return (tree_block *)_p; }
//     tree_block *operator()() const { return (tree_block *)_p; }
};

class p_tree_proc : public p_tree_block {
public:
    p_tree_proc(tree_proc *p = 0) : p_tree_block(p) {}
    p_tree_proc(const p_tree_proc& o) : p_tree_block(o) {}
    p_tree_proc(const p_tree_block& o) : p_tree_block(o) {}
    p_tree_proc(const p_tree_node& o) : p_tree_block(o) {}
    tree_proc *operator->() const { return (tree_proc *)_p; }
    tree_proc *ptr() const { return (tree_proc *)_p; }
//     tree_proc *operator()() const { return (tree_proc *)_p; }
};

class p_tree_for : public p_tree_node {
public:
    p_tree_for(tree_for *p = 0) : p_tree_node(p) {}
    p_tree_for(const p_tree_for& o) : p_tree_node(o) {}
    p_tree_for(const p_tree_node& o) : p_tree_node(o) {}
    tree_for *operator->() const { return (tree_for *)_p; }
//     tree_for *operator()() const { return (tree_for *)_p; }
};

class p_tree_if : public p_tree_node {
public:
    p_tree_if(tree_if *p = 0) : p_tree_node(p) {}
    p_tree_if(const p_tree_if& o) : p_tree_node(o) {}
    p_tree_if(const p_tree_node& o) : p_tree_node(o) {}
    tree_if *operator->() const { return (tree_if *)_p; }
//     tree_if *operator()() const { return (tree_if *)_p; }
};

class p_tree_instr : public p_tree_node {
public:
    p_tree_instr(tree_instr *p = 0) : p_tree_node(p) {}
    p_tree_instr(const p_tree_instr& o) : p_tree_node(o) {}
    p_tree_instr(const p_tree_node& o) : p_tree_node(o) {}
    tree_instr *operator->() const { return (tree_instr *)_p; }
//     tree_instr *operator()() const { return (tree_instr *)_p; }
};

class p_tree_loop : public p_tree_node {
public:
    p_tree_loop(tree_loop *p = 0) : p_tree_node(p) {}
    p_tree_loop(const p_tree_loop& o) : p_tree_node(o) {}
    p_tree_loop(const p_tree_node& o) : p_tree_node(o) {}
    tree_loop *operator->() const { return (tree_loop *)_p; }
//     tree_loop *operator()() const { return (tree_loop *)_p; }
};

#endif	// __WRAPPERS_H
